/* 
 * CyberLinkC AV API for Control Points
 *
 * cavtcontrolpoint.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#include <cybergarage/avcontrol/cavcontrolpoint.h>
#include <cybergarage/avtransport/cavtcontrolpoint.h>
#include <cybergarage/upnp/caction.h>
#include <cybergarage/upnp/cupnp_status.h>
#include <cybergarage/upnp/control/ccontrol.h>

#include <cybergarage/avdebug.h>

#include <stdarg.h>

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and CgString* ARG_VALUE.
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer.
 */
int cg_upnp_av_avt_control_invokeaction(CgUpnpDevice* upnpDevice,
					 char* actionName,
					 char** errorString,
					 ...)
{
	va_list va;
	CgUpnpService* avt = NULL;
	CgUpnpAction* action;
	int result = 0;

	DEBUG("BEGIN: cg_upnp_av_avt_control_invokeaction\n");

	if (upnpDevice == NULL || actionName == NULL)
	{
		DEBUG("Invalid parameters!\n");
		DEBUG("END: cg_upnp_av_avt_control_invokeaction\n");
		return CG_UPNP_STATUS_INVALID_ARGS;
	}

	/* Get the rendering control service */
	avt = cg_upnp_device_getservicebytype(upnpDevice, CG_UPNP_AVT_SERVICE_TYPE);
	if (avt == NULL)
	{
		return CG_UPNP_STATUS_INVALID_ACTION;
	}
	
	/* Get the action specified in actionName argument */
	action = cg_upnp_service_getactionbyname(avt, actionName);
	if (action == NULL)
	{		
		return CG_UPNP_STATUS_INVALID_ACTION;
	}
	
	/* Set the IN arguments */
	va_start(va, errorString);
	result = cg_upnp_av_control_parseactionarguments(action, TRUE, &va);
	va_end(va);

	if (!cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		DEBUG("Action's IN argument parsing failed\n");
		DEBUG("END: cg_upnp_av_avt_control_invokeaction\n");
		return result;
	}
	
	/* Invoke the action */
	if (cg_upnp_action_post(action) == TRUE)
	{
		/* Get the OUT arguments */
		va_start(va, errorString);
		result = cg_upnp_av_control_parseactionarguments(action, 
								 FALSE,
								 &va);
		va_end(va);
				
		if (!cg_upnp_av_control_iserrorcodesuccessful(result))
		{
			DEBUG("Action's OUT argument parsing failed\n");
			DEBUG("END: cg_upnp_av_avt_control_invokeaction\n");
			return result;
		}
		else
		{
			DEBUG("Action [%s] post successful\n", actionName);
		}
	}
	else
	{
		DEBUG("Action [%s] post failed:\n",
			actionName);
		DEBUG("-->Error: [%s]\n",
			cg_upnp_action_getstatusdescription(action));
			
		if (errorString)
		{
			*errorString = 
			cg_strdup(cg_upnp_action_getstatusdescription(action));
		}
	}

	/* Return the resulting action status code */
	result = cg_upnp_action_getstatuscode(action);

	DEBUG("END: cg_upnp_av_avt_control_invokeaction\n");

	return result;
}
