/*
 * UPnP Browser for Maemo
 *
 * media_control.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-defines.h>
#include <osso-mime.h>
#include "player.h"
#endif

#include <gtk/gtk.h>
#include <string.h>
#include "media_control.h"
#include "interface.h"
#include "browser.h"
#include "now_playing.h"
#include "upnp.h"
#include "resource_selection.h"

extern UPnPBrowserWidgets* widgets;

/**
 * Start playing selected media in the selected renderer device
 *
 * @param folder_model
 */
gboolean play_media()
{
	gchar *name = NULL;
	gchar *node_class = NULL;
	gchar *uri = NULL;
	gchar *id = NULL;
	gchar *srv_udn = NULL;
	gchar *renderer_udn = NULL;
	gchar *selected_uri = NULL;
	
	gchar *cached_uri = NULL;
	gchar *cached_name = NULL;
	
	gboolean result = FALSE;

	if (get_current_content_item(&name, &uri, &node_class,
				     &id, &srv_udn) == TRUE)
	{
		if (resource_selection_dialog_run(uri, &selected_uri) == TRUE)
		{
			if (get_current_renderer(&renderer_udn,
						 &cached_uri,
						 &cached_name) == TRUE)
			{
				result = play_remote_media(renderer_udn, 
							   selected_uri,
							   node_class,
							   id,
							   srv_udn);
			}
			else
			{
				result = play_local_media(selected_uri,
							  node_class);
			}
			
			if (result == TRUE)
			{
				/* Save URI and name temporarily */
				cache_current_track_for_current_renderer(
						selected_uri, name);
			}

			g_free(renderer_udn);
			g_free(cached_uri);
			g_free(cached_name);
			g_free(selected_uri);
		}
	}
	else
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Nothing selected");
#else
		fprintf(stderr, "Nothing selected\n");
#endif
	}
	
	update_now_playing_text();

	g_free(name);
	g_free(uri);
	g_free(node_class);
	g_free(id);
	g_free(srv_udn);

	return result;
}

/**
 * Start playing media locally, MAEMO version
 *
 * @param uri The URI of the media to play
 * @param node_class The UPnP class of the media
 */
#ifdef MAEMO

gboolean play_local_media(gchar* uri, gchar* node_class)
{
	DBusConnection *conn = NULL;
	gboolean result = FALSE;
	
	printf("Playing URI: %s\n", uri);
	
	if (uri == NULL || node_class == NULL) 
	{
		gtk_infoprintf(NULL, "Can't find resource");
		return FALSE;
	}

	if (cg_strstr(node_class, "object.item.audioItem") != -1)
	{
		player_stop(widgets->osso, NULL);
		player_set_volume(widgets->osso, widgets->volume, NULL);
		player_set_mute(widgets->osso, widgets->mute, NULL);
		result = player_play_uri(widgets->osso, uri, NULL);
		
		if (result == FALSE)
		{
			gtk_infoprint(NULL, "Unable to play audio");
		}
	}
	else
	{
		/*
		 It should be possible to "play" any kind of media 
		 file with libosso-mime by just using
	 	 gint osso_mime_open_file(DBusConnection *con,
					  const gchar *file)

		 - Maybe we should still use media-engine for showing
		   audio and video? (at least for audio, it wouldn't
		   launch an external player)	
		
		 osso-mime is too crappy to do this, needed to fix
		 their functions.
		*/
		conn = (DBusConnection*)osso_get_dbus_connection(
							widgets->osso);
		if (osso_mime_open_file(conn, uri) < 1)
		{
			gtk_infoprint(NULL, "Unable to play media");
		}
		else
		{
			result = TRUE;
		}
	}
	
	return result;	
}

#else

/**
 * Start playing media locally, UNIX version
 *
 * @param uri The URI of the media to play
 * @param node_class The UPnP class of the media
 */
gboolean play_local_media(gchar* uri, gchar* node_class)
{
	gchar* browserInvoke = NULL;
	
	if (uri == NULL || node_class == NULL) 
	{
		fprintf(stderr, "Can't find resource\n");
		return FALSE;
	}

	/* A 100% foobar solution: invoke the URI with a browser */
	browserInvoke = g_new0(gchar, strlen(uri) + 
			       strlen("/usr/bin/mozilla") + 5);
	sprintf(browserInvoke, "/usr/bin/mozilla %s", uri);
	g_spawn_command_line_async(browserInvoke, NULL);
	g_free(browserInvoke);
	
	return TRUE;
}

#endif

/**
 * Stop playing a media (remote or local)
 *
 */
void stop_media()
{
	GtkTreeIter renderer_iter;
	gchar* renderer_device = NULL;

	/* Get active renderer selection */
	gtk_combo_box_get_active_iter(GTK_COMBO_BOX(widgets->renderer_combo),
					&renderer_iter);
	/* Get renderer device */
	gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter,
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_device, -1);

	if (renderer_device != NULL)
	{
		stop_remote_media(renderer_device);
	}
	else
	{
		stop_local_media();
	}
	
	update_now_playing_text();
}

/**
 * Stop playing media locally
 *
 */
void stop_local_media()
{
#ifdef MAEMO
	player_stop(widgets->osso, NULL);
#else
	fprintf(stderr, "Not implemented\n");
#endif
}

/**
 * Pause playing media (remote or local)
 *
 */
void pause_media()
{
	GtkTreeIter renderer_iter;
	gchar* renderer_device = NULL;

	/* Get active renderer selection */
	gtk_combo_box_get_active_iter(GTK_COMBO_BOX(widgets->renderer_combo),
					&renderer_iter);
	/* Get renderer device */
	gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter,
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_device, -1);

	if (renderer_device != NULL)
	{
		pause_remote_media(renderer_device);
	}
	else
	{
		pause_local_media();
	}
	
	update_now_playing_text();
}

/**
 * Pause playing media locally
 *
 */
void pause_local_media()
{
#ifdef MAEMO
	player_pause(widgets->osso, NULL);
#else
	fprintf(stderr, "Not implemented\n");
#endif
}


/**
 * Next media
 *
 */
void next_media()
{	
	GtkTreeIter renderer_iter;
	gchar* renderer_device = NULL;

	/* Get active renderer selection */
	gtk_combo_box_get_active_iter(GTK_COMBO_BOX(widgets->renderer_combo),
					&renderer_iter);
	/* Get renderer device */
	gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter,
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_device, -1);

	if (renderer_device != NULL)
	{
		next_remote_media(renderer_device);
	}
	else
	{
		next_local_media();
	}
	
	g_free(renderer_device);
	
	update_now_playing_text();
}

/**
 * Next local media
 *
 */
void next_local_media()
{
#ifdef MAEMO
	gtk_infoprint(NULL, "Not implemented");
#else
	fprintf(stderr, "Not implemented\n");
#endif
}

/**
 * Previous media
 *
 */
void previous_media()
{	
	GtkTreeIter renderer_iter;
	gchar* renderer_device = NULL;

	/* Get active renderer selection */
	gtk_combo_box_get_active_iter(GTK_COMBO_BOX(widgets->renderer_combo),
					&renderer_iter);
	/* Get renderer device */
	gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter,
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_device, -1);

	if (renderer_device != NULL)
	{
		previous_remote_media(renderer_device);
	}
	else
	{
		previous_local_media();
	}
	
	update_now_playing_text();
}

/**
 * Skip to previous local media
 *
 */
void previous_local_media()
{
#ifdef MAEMO
	gtk_infoprint(NULL, "Not implemented");
#else
	fprintf(stderr, "Not implemented\n");
#endif
}

/**
 * Un/Mute media
 *
 */
void mute_media()
{
	gboolean result = TRUE;
	gchar* udn = NULL;
	gchar* uri = NULL;
	gchar* name = NULL;
	gboolean mute = FALSE;
	
	if (get_current_renderer(&udn, &uri, &name) == TRUE)
	{
		if (get_remote_mute(udn, &mute) == TRUE && mute == TRUE)
		{
			result = mute_remote_media(udn, FALSE);
		}
		else
		{
			result = mute_remote_media(udn, TRUE);
		}
	}
	else
	{
		mute_local_media();
	}
	
	if (result == FALSE)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Failed");
#else
		fprintf(stderr, "Un/Mute failed");
#endif
	}
	
	g_free(udn);
	g_free(uri);
	g_free(name);
}

/**
 * Un/Mute local media
 *
 */
void mute_local_media()
{
	if (widgets->mute)
	{ 
		widgets->mute = FALSE;
#ifdef MAEMO
		gtk_image_set_from_icon_name(
			GTK_IMAGE(gtk_tool_button_get_icon_widget(GTK_TOOL_BUTTON(widgets->mute_button))),
			"qgn_indi_gene_volume", HILDON_ICON_SIZE_26);
#endif
	} else {
		widgets->mute = TRUE;
#ifdef MAEMO
		gtk_image_set_from_icon_name(
			GTK_IMAGE(gtk_tool_button_get_icon_widget(GTK_TOOL_BUTTON(widgets->mute_button))),
			"qgn_indi_gene_mute", HILDON_ICON_SIZE_26);
#endif
	}
	
#ifdef MAEMO
	player_set_mute(widgets->osso, widgets->mute, NULL);
#else
	fprintf(stderr, "Not implemented\n");
#endif	
}

/**
 * Decrease volume
 *
 */
void decrease_volume()
{
	gboolean result = TRUE;
	gchar* udn = NULL;
	gchar* uri = NULL;
	gchar* name = NULL;
	
	if (get_current_renderer(&udn, &uri, &name) == TRUE)
	{
		result = decrease_remote_volume(udn);
	}
	else
	{
		decrease_local_volume();
	}
	
	if (result == FALSE)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Failed");
#else
		fprintf(stderr, "Volume decrease failed");
#endif
	}
	
	g_free(udn);
	g_free(uri);
	g_free(name);
}

/**
 * Decrease local volume
 *
 */
void decrease_local_volume()
{
	widgets->volume -= 5;
	
	if (widgets->volume < 5)
	{
		widgets->volume = 0;
	}

#ifdef MAEMO
	player_set_volume(widgets->osso, widgets->volume, NULL);
#else
	fprintf(stderr, "Not implemented\n");
#endif
}

/**
 * Increase volume
 *
 */
void increase_volume()
{
	gboolean result = TRUE;
	gchar* udn = NULL;
	gchar* uri = NULL;
	gchar* name = NULL;
	
	if (get_current_renderer(&udn, &uri, &name) == TRUE)
	{
		result = increase_remote_volume(udn);
	}
	else
	{
		increase_local_volume();
	}

	if (result == FALSE)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Failed");
#else
		fprintf(stderr, "Volume increase failed");
#endif
	}
	
	g_free(udn);
	g_free(uri);
	g_free(name);
}

/**
 * Increase local volume
 *
 */
void increase_local_volume()
{
	widgets->volume += 5;
	
	if (widgets->volume > 100)
	{
		widgets->volume = 100;
	}
	
#ifdef MAEMO
	player_set_volume(widgets->osso, widgets->volume, NULL);
#else
	fprintf(stderr, "Not implemented\n");
#endif
}

/**
 * Seek media to a certain point in time
 */
void seek_media(long seekto)
{
	gchar* udn = NULL;
	gchar* uri = NULL;
	gchar* name = NULL;
       
	if (get_current_renderer(&udn, &uri, &name) == TRUE)
	{
		seek_remote_media(udn, seekto);
	}
	else
	{
		seek_local_media(seekto);
	}
       
	g_free(udn);
	g_free(uri);
	g_free(name);
}

/**
 * Seek local media to a certain point in time
 */
void seek_local_media(long seekto)
{
#ifdef MAEMO
	gtk_infoprint(NULL, "Not implemented");
#else
	fprintf(stderr, "Not implemented\n");
#endif
}
