/* 
 * CyberLinkC AV API for Control Points
 *
 * cdidllite.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */
 
#ifndef CDIDLLITE_H
#define CDIDLLITE_H

#include <cybergarage/xml/cxml.h>

/****************************************************************************
 * DIDL-lite tag, attributes & values
 ****************************************************************************/
#define DIDL_LITE_NAME "DIDL-Lite"
#define DIDL_LITE_XMLNS "xmlns"
#define DIDL_LITE_XMLNS_URL "urn:schemas-upnp-org:metadata-1-0/DIDL-Lite"
#define DIDL_LITE_XMLNS_DC "xmlns:dc"
#define DIDL_LITE_XMLNS_DC_URL "http://purl.org/dc/elements/1.1"
#define DIDL_LITE_XMLNS_UPNP "xmlns:upnp"
#define DIDL_LITE_XMLNS_UPNP_URL "urn:schemas-upnp-org:metadata-1-0/upnp"

/****************************************************************************
 * Container tag & attributes
 ****************************************************************************/
#define DIDL_LITE_CONTAINER "container"
#define DIDL_LITE_CONTAINER_SEARCHABLE "searchable"
#define DIDL_LITE_CONTAINER_CHILDCOUNT "childCount"

/**
 * DIDL_LITE_SEARCHABLE is deprecated since 1.0.10.
 * Use DIDL_LITE_CONTAINER_SEARCHABLE instead.
 */
static const char* searchable_deprecated = "searchable";
static inline const char* didl_lite_searchable_deprecated()
{
	fprintf(stderr, "\nWarning: Deprecated clinkc-av macro DIDL_LITE_SEARCHABLE used!\n");
	return searchable_deprecated;
}
#define DIDL_LITE_SEARCHABLE didl_lite_searchable_deprecated()

/**
 * DIDL_LITE_CHILDCOUNT is deprecated since 1.0.10.
 * Use DIDL_LITE_CONTAINER_CHILDCOUNT instead.
 */
static const char* childcount_deprecated = "childCount";
static inline const char* didl_lite_childcount_deprecated()
{
	fprintf(stderr, "\nWarning: Deprecated clinkc-av macro DIDL_LITE_CHILDCOUNT used!\n");
	return childcount_deprecated;
}
#define DIDL_LITE_CHILDCOUNT didl_lite_childcount_deprecated()

/****************************************************************************
 * Item tag & attributes
 ****************************************************************************/
#define DIDL_LITE_ITEM "item"
#define DIDL_LITE_ITEM_ID "id"
#define DIDL_LITE_ITEM_REFID "refID"

/**
 * DIDL_LITE_ID is deprecated since 1.0.10.
 * Use DIDL_LITE_OBJECT_ID instead.
 */
static const char* id_deprecated = "id";
static inline const char* didl_lite_id_deprecated()
{
	fprintf(stderr, "\nWarning: Deprecated clinkc-av macro DIDL_LITE_ID used!\n");
	return id_deprecated;
}
#define DIDL_LITE_ID didl_lite_id_deprecated()

/**
 * DIDL_LITE_PARENTID is deprecated since 1.0.10.
 * Use DIDL_LITE_OBJECT_PARENTID instead.
 */
static const char* parentid_deprecated = "parentID";
static inline const char* didl_lite_parentid_deprecated()
{
	fprintf(stderr, "\nWarning: Deprecated clinkc-av macro DIDL_LITE_PARENTID used!\n");
	return parentid_deprecated;
}
#define DIDL_LITE_PARENTID didl_lite_parentid_deprecated()

/**
 * DIDL_LITE_RESTICTED is deprecated (and misspelled) since 1.0.10.
 * Use DIDL_LITE_OBJECT_RESTRICTED instead.
 */
static const char* resticted_deprecated = "restricted";
static inline const char* didl_lite_resticted_deprecated()
{
	fprintf(stderr, "\nWarning: Deprecated clinkc-av macro DIDL_LITE_RESTICTED used!\n");
	return resticted_deprecated;
}
#define DIDL_LITE_RESTICTED didl_lite_resticted_deprecated()

/****************************************************************************
 * Container & Item shared attributes
 ****************************************************************************/
#define DIDL_LITE_OBJECT_ID "id"
#define DIDL_LITE_OBJECT_PARENTID "parentID"
#define DIDL_LITE_OBJECT_RESTRICTED "restricted"

/****************************************************************************
 * Desc-specific tags & attributes
 ****************************************************************************/
#define DIDL_LITE_DESC "desc"

/****************************************************************************
 * Resource tag and its associated properties
 ****************************************************************************/
#define DIDL_LITE_RES "res"
#define DIDL_LITE_RES_SIZE "size"
#define DIDL_LITE_RES_DURATION "duration"
#define DIDL_LITE_RES_BITRATE "bitrate"
#define DIDL_LITE_RES_SAMPLEFREQUENCY "sampleFrequency"
#define DIDL_LITE_RES_BITSPERSAMPLE "bitsPerSample"
#define DIDL_LITE_RES_NRAUDIOCHANNELS "nrAudioChannels"
#define DIDL_LITE_RES_RESOLUTION "resolution"
#define DIDL_LITE_RES_COLORDEPTH "colorDepth"
#define DIDL_LITE_RES_PROTOCOLINFO "protocolInfo"
#define DIDL_LITE_RES_PROTECTION "protection"
#define DIDL_LITE_RES_IMPORTURI "importUri"

/****************************************************************************
 * DIDL-Lite tags belonging to the UPnP namespace
 ****************************************************************************/
#define DIDL_LITE_UPNP_ACTOR "upnp:actor"
#define DIDL_LITE_UPNP_ACTOR_ROLE "role"
#define DIDL_LITE_UPNP_ALBUM "upnp:album"
#define DIDL_LITE_UPNP_ALBUMARTURI "upnp:albumArtURI"
#define DIDL_LITE_UPNP_ARTIST "upnp:artist"
#define DIDL_LITE_UPNP_ARTIST_ROLE "role"
#define DIDL_LITE_UPNP_ARTISTDISCOGRAPHYURI "upnp:artistDiscographyURI"
#define DIDL_LITE_UPNP_AUTHOR "upnp:author"
#define DIDL_LITE_UPNP_AUTHOR_ROLE "role"
#define DIDL_LITE_UPNP_CHANNELNAME "upnp:channelName"
#define DIDL_LITE_UPNP_CHANNELNR "upnp:channelNr"
#define DIDL_LITE_UPNP_CLASS "upnp:class"
#define DIDL_LITE_UPNP_CREATECLASS "upnp:createClass"
#define DIDL_LITE_UPNP_CREATECLASS_INCLUDEDERIVED "includeDerived"
#define DIDL_LITE_UPNP_CREATECLASS_NAME "name"
#define DIDL_LITE_UPNP_DIRECTOR "upnp:director"
#define DIDL_LITE_UPNP_DVDREGIONCODE "upnp:DVDRegionCode"
#define DIDL_LITE_UPNP_GENRE "upnp:genre"
#define DIDL_LITE_UPNP_ICON "upnp:icon"
#define DIDL_LITE_UPNP_LONGDESCRIPTION "upnp:longDescription"
#define DIDL_LITE_UPNP_LYRICSURI "upnp:lyricsURI"
#define DIDL_LITE_UPNP_ORIGINALTRACKNUMBER "upnp:originalTrackNumber"
#define DIDL_LITE_UPNP_PLAYLIST "upnp:playlist"
#define DIDL_LITE_UPNP_PRODUCER "upnp:producer"
#define DIDL_LITE_UPNP_PROTECTION "upnp:protection"
#define DIDL_LITE_UPNP_RADIOBAND "upnp:radioBand"
#define DIDL_LITE_UPNP_RADIOCALLSIGN "upnp:radioCallSign"
#define DIDL_LITE_UPNP_RADIOSTATIONID "upnp:radioStationID"
#define DIDL_LITE_UPNP_RATING "upnp:rating"
#define DIDL_LITE_UPNP_REGION "upnp:region"
#define DIDL_LITE_UPNP_SCHEDULEDENDTIME "upnp:scheduledEndTime"
#define DIDL_LITE_UPNP_SCHEDULEDSTARTTIME "upnp:scheduledStartTime"
#define DIDL_LITE_UPNP_SEARCHCLASS "upnp:searchClass"
#define DIDL_LITE_UPNP_SEARCHCLASS_INCLUDEDERIVED "includeDerived"
#define DIDL_LITE_UPNP_SEARCHCLASS_NAME "name"
#define DIDL_LITE_UPNP_STORAGEFREE "upnp:storageFree"
#define DIDL_LITE_UPNP_STORAGEMAXPARTITION "upnp:storageMaxPartition"
#define DIDL_LITE_UPNP_STORAGEMEDIUM "upnp:storageMedium"
#define DIDL_LITE_UPNP_STORAGETOTAL "upnp:storageTotal"
#define DIDL_LITE_UPNP_STORAGEUSED "upnp:storageUsed"
#define DIDL_LITE_UPNP_TOC "upnp:toc"
#define DIDL_LITE_UPNP_USERANNOTATION "upnp:userAnnotation"
#define DIDL_LITE_UPNP_WRITESTATUS "upnp:writeStatus"

/****************************************************************************
 * DIDL-Lite tags belonging to the Dublin Core namespace
 ****************************************************************************/
#define DIDL_LITE_DC_CONTRIBUTOR "dc:contributor"
#define DIDL_LITE_DC_CREATOR "dc:creator"
#define DIDL_LITE_DC_DATE "dc:date"
#define DIDL_LITE_DC_DESCRIPTION "dc:description"
#define DIDL_LITE_DC_LANGUAGE "dc:language"
#define DIDL_LITE_DC_PUBLISHER "dc:publisher"
#define DIDL_LITE_DC_RELATION "dc:relation"
#define DIDL_LITE_DC_RIGHTS "dc:rights"
#define DIDL_LITE_DC_TITLE "dc:title"

/****************************************************************************
 * Object class hierarchy tags
 ****************************************************************************/
#define DIDL_LITE_OBJECT "object"

#define DIDL_LITE_OBJECT_ITEM "object.item"
#define DIDL_LITE_OBJECT_ITEM_IMAGEITEM "object.item.imageItem"
#define DIDL_LITE_OBJECT_ITEM_IMAGEITEM_PHOTO "object.item.imageItem.photo"
#define DIDL_LITE_OBJECT_ITEM_AUDIOITEM "object.item.audioItem"
#define DIDL_LITE_OBJECT_ITEM_AUDIOITEM_MUSICTRACK "object.item.audioItem.musicTrack"
#define DIDL_LITE_OBJECT_ITEM_AUDIOITEM_AUDIOBROADCAST "object.item.audioItem.audioBroadcast"
#define DIDL_LITE_OBJECT_ITEM_AUDIOITEM_AUDIOBOOK "object.item.audioItemaudioBook"
#define DIDL_LITE_OBJECT_ITEM_VIDEOITEM "object.item.videoItem"
#define DIDL_LITE_OBJECT_ITEM_VIDEOITEM_MOVIE "object.item.videoItem.movie"
#define DIDL_LITE_OBJECT_ITEM_VIDEOITEM_VIDEOBROADCAST "object.item.videoItem.videoBroadcast"
#define DIDL_LITE_OBJECT_ITEM_VIDEOITEM_MUSICVIDEOCLIP "object.item.videoItem.musicVideoClip"
#define DIDL_LITE_OBJECT_ITEM_PLAYLISTITEM "object.item.playlistItem"
#define DIDL_LITE_OBJECT_ITEM_TEXTITEM "object.item.textItem"

#define DIDL_LITE_OBJECT_CONTAINER "object.container"
#define DIDL_LITE_OBJECT_CONTAINER_PERSON "object.container.person"
#define DIDL_LITE_OBJECT_CONTAINER_PERSON_MUSICARTIST "object.container.person.musicArtist"
#define DIDL_LITE_OBJECT_CONTAINER_PLAYLISTCONTAINER "object.container.playlistContainer"
#define DIDL_LITE_OBJECT_CONTAINER_ALBUM "object.container.album"
#define DIDL_LITE_OBJECT_CONTAINER_ALBUM_MUSICALBUM "object.container.album.musicAlbum"
#define DIDL_LITE_OBJECT_CONTAINER_ALBUM_PHOTOALBUM "object.container.album.photoAlbum"
#define DIDL_LITE_OBJECT_CONTAINER_GENRE "object.container.genre"
#define DIDL_LITE_OBJECT_CONTAINER_GENRE_MUSICGENRE "object.container.genre.musicGenre"
#define DIDL_LITE_OBJECT_CONTAINER_GENRE_MOVIEGENRE "object.container.genre.movieGenre"
#define DIDL_LITE_OBJECT_CONTAINER_STORAGESYSTEM "object.container.storageSystem"
#define DIDL_LITE_OBJECT_CONTAINER_STORAGEVOLUME "object.container.storageVolume"
#define DIDL_LITE_OBJECT_CONTAINER_STORAGEFOLDER "object.container.storageFolder"

/****************************************************************************
 * Functions
 ****************************************************************************/

/**
 * Check the validity of a DIDL-Lite nodelist and return the direct list
 * of children for the DIDL-Lite root node
 *
 * @param node A DIDL-Lite root node
 * @param strict If TRUE, check that xmlns, xmlns:dc and xmlns:upnp match
 *               exactly the ones in the CDS specification
 *
 * @return The direct list of children if valid; otherwise NULL
 */
CgXmlNodeList* cg_upnp_av_cds_didllite_getchildren(CgXmlNodeList* nodelist,
						   BOOL strict);

/**
 * Extract the protocol from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the protocol or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getprotocol(CgXmlNode* resNode);

/**
 * Extract the network parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the network or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getnetwork(CgXmlNode* resNode);

/**
 * Extract the mime type parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the mime type or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getmimetype(CgXmlNode* resNode);

/**
 * Extract the additional info parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the additional info or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getadditionalinfo(CgXmlNode* resNode);

/****************************************************************************
 * Node name <name>...</name>
 ****************************************************************************/

/**
 * Get the name of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 */
#define cg_upnp_av_cds_didllite_node_getname(node) \
	cg_xml_node_getname(node)

/**
 * Set the name of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 * @param name The name to set
 */
#define cg_upnp_av_cds_didllite_node_setname(node, name) \
	cg_xml_node_setname(node, name)

/**
 * Check, whether a node is a container
 *
 * @return TRUE if the node is a container; otherwise FALSE
 */
#define cg_upnp_av_cds_didllite_node_iscontainer(node) \
	( (cg_strcmp(cg_xml_node_getname(node), DIDL_LITE_CONTAINER) == 0) ? \
	  TRUE : FALSE )

#define cg_upnp_av_cds_didllite_node_isname(node, nodename) \
	cg_xml_node_isname(node, nodename)

/****************************************************************************
 * Node value <node> value </node>
 ****************************************************************************/

/**
 * Get the value of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 */
#define cg_upnp_av_cds_didllite_node_getvalue(node) \
	cg_xml_node_getvalue(node)

/**
 * Set the value of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 * @param value The value to set
 */
#define cg_upnp_av_cds_didllite_node_setvalue(node, value) \
	cg_xml_node_setvalue(node, value)

/****************************************************************************
 * Node attributes <node attribute1="foo" attribute2="bar"> ... </node>
 ****************************************************************************/

/**
 * Get an attribute value of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 * @param attribute Name of the attribute to get
 */
#define cg_upnp_av_cds_didllite_node_getattribute(node, attribute) \
	cg_xml_node_getattributevalue(node, attribute)

/**
 * Set an attribute value of a DIDL-lite node
 *
 * @param node The CgXmlNode*
 * @param attribute Name of the attribute to get
 */
#define cg_upnp_av_cds_didllite_node_setattribute(node, attribute) \
	cg_xml_node_setattributevalue(node, attribute)

/****************************************************************************
 * Iterators
 ****************************************************************************/

/**
 * Get the next DIDL node
 *
 */
#define cg_upnp_av_cds_didllite_node_next(node) \
	cg_xml_node_next(node)

/****************************************************************************
 * Node's children
 ****************************************************************************/

/**
 * Get the children of a node
 *
 * @param node The CgXmlNode*
 *
 * @return CgXmlNodeList*
 */
#define cg_upnp_av_cds_didllite_node_getchildnodelist(node) \
	cg_xml_node_getchildnodelist(node)

/**
 * Get the first child node from a DIDL node list
 *
 * @param nodelist The CgXmlNodeList* containing children
 *
 * @return The child CgXmlNode*
 */
#define cg_upnp_av_cds_didllite_nodelist_gets(nodelist) \
	cg_xml_nodelist_gets(nodelist)

#endif /* CDIDLLITE_H */
