/* 
 * CyberLinkC AV API for Control Points
 *
 * cavtcontrolpoint.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifndef CG_UPNP_AV_CAVTCONTROLPOINT_H
#define CG_UPNP_AV_CAVTCONTROLPOINT_H

#include <cybergarage/avtransport/cavtransport.h>
#include <cybergarage/avtransport/cavtactionarguments.h>
#include <cybergarage/upnp/cdevice.h>

/*****************************************************************************
 * AV Transport Service action invocations
 *****************************************************************************/

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param numArgs The number of action-specific arguments
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and char** ARG_VALUE.
 *
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer. Finally, put a single NULL to end the argument list.
 */
int cg_upnp_av_avt_control_invokeaction(CgUpnpDevice* upnpDevice,
					char* actionName,
					char** errorString,
					...);

/*****************************************************************************
 * URI setting
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to set the AV transport URI for the
 * given AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param uri The AV Transport URI (IN)
 * @param uriMetaData Optional meta data (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_setavtransporturi(upnpDevice, \
						 errorString, \
						 instanceID, \
						 uri, \
						 uriMetaData) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
			CG_UPNP_AVT_ACTION_SET_AVTRANSPORTURI, \
			errorString, \
			CG_UPNP_AVT_ARG_INSTANCEID, instanceID, \
			CG_UPNP_AVT_ARG_CURRENTURI, uri, \
			CG_UPNP_AVT_ARG_CURRENTURIMETADATA, uriMetaData, \
			NULL)

/**
 * Invoke an action on the upnp device to set the next AV transport URI for
 * the given AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param nextUri The next transport URI (IN)
 * @param nextUriMetaData Optional meta data for next URI (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_setnextavtransporturi(upnpDevice, \
						     errorString, \
						     instanceID, \
						     nextUri, \
						     nextUriMetaData) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
			CG_UPNP_AVT_ACTION_SET_NEXTAVTRANSPORTURI, \
			errorString, \
			CG_UPNP_AVT_ARG_INSTANCEID, instanceID, \
			CG_UPNP_AVT_ARG_NEXTURI, nextUri, \
			CG_UPNP_AVT_ARG_NEXTURIMETADATA, nextUriMetaData, \
			NULL)

/*****************************************************************************
 * Status information getters
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to get media info from the given AVT
 * instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param nrTracks Number of tracks (OUT)
 * @param mediaDuration Current media duration (OUT)
 * @param instanceID The AV transport instance id
 * @param uri The AV Transport URI (OUT)
 * @param uriMetaData Meta data for current URI (OUT)
 * @param nextUri The next transport URI (OUT)
 * @param nextUriMetaData Meta data for next URI (OUT)
 * @param playMedium Current playing medium (OUT)
 * @param recordMedium Current recording medium (OUT)
 * @param writeStatus Current writing status (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_getmediainfo(upnpDevice, \
					    errorString, \
					    instanceID, \
					    nrTracks, \
					    mediaDuration, \
					    currentUri, \
					    currentUriMetaData, \
					    nextUri, \
					    nextUriMetaData, \
					    playMedium, \
					    recordMedium, \
					    writeStatus) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
			CG_UPNP_AVT_ACTION_GET_MEDIAINFO, \
			errorString, \
			CG_UPNP_AVT_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_AVT_ARG_NRTRACKS, \
			nrTracks, \
			CG_UPNP_AVT_ARG_MEDIADURATION, \
			mediaDuration, \
			CG_UPNP_AVT_ARG_CURRENTURI, \
			currentUri, \
			CG_UPNP_AVT_ARG_CURRENTURIMETADATA, \
			currentUriMetaData, \
			CG_UPNP_AVT_ARG_NEXTURI, \
			nextUri, \
			CG_UPNP_AVT_ARG_NEXTURIMETADATA, \
			nextUriMetaData, \
			CG_UPNP_AVT_ARG_PLAYMEDIUM, \
			playMedium, \
			CG_UPNP_AVT_ARG_RECORDMEDIUM, \
			recordMedium, \
			CG_UPNP_AVT_ARG_WRITESTATUS, \
			writeStatus, \
			NULL)

/**
 * Invoke an action on the upnp device to get transport info from the given
 * AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param currentTransportState Current Transport State (OUT)
 * @param currentTransportStatus Current Transport Status (OUT)
 * @param currentSpeed Current Speed (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_gettransportinfo(upnpDevice, \
						errorString, \
					        instanceID, \
						currentTransportState, \
						currentTransportStatus, \
						currentSpeed) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
			CG_UPNP_AVT_ACTION_GET_TRANSPORTINFO, \
			errorString, \
			CG_UPNP_AVT_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_AVT_ARG_CURRENTTRANSPORTSTATE, \
			currentTransportState, \
			CG_UPNP_AVT_ARG_CURRENTTRANSPORTSTATUS, \
			currentTransportStatus, \
			CG_UPNP_AVT_ARG_CURRENTSPEED, \
			currentSpeed, \
			NULL)

/**
 * Invoke an action on the upnp device to get position info from the given
 * AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param track Current track number (OUT)
 * @param trackDuration Current track duration (OUT)
 * @param trackMetaData Current track meta data (OUT)
 * @param trackURI Current track URI (OUT)
 * @param relTime Current track relative time (OUT)
 * @param absTime Current track absolute time (OUT)
 * @param relCount Current track relative count (OUT)
 * @param absCount Current track absolute count (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_getpositioninfo(upnpDevice, \
					       errorString, \
					       instanceID, \
					       track, \
					       trackDuration, \
					       trackMetaData, \
					       trackURI, \
					       relTime, \
					       absTime, \
					       relCount, \
					       absCount) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
		CG_UPNP_AVT_ACTION_GET_POSITIONINFO, \
		errorString, \
		CG_UPNP_AVT_ARG_INSTANCEID, \
		instanceID, \
		CG_UPNP_AVT_ARG_TRACK, \
		track, \
		CG_UPNP_AVT_ARG_TRACKDURATION, \
		trackDuration, \
		CG_UPNP_AVT_ARG_TRACKMETADATA, \
		trackMetaData, \
		CG_UPNP_AVT_ARG_TRACKURI, \
		trackURI, \
		CG_UPNP_AVT_ARG_RELTIME, \
		relTime, \
		CG_UPNP_AVT_ARG_ABSTIME, \
		absTime, \
		CG_UPNP_AVT_ARG_RELCOUNT, \
		relCount, \
		CG_UPNP_AVT_ARG_ABSCOUNT, \
		absCount, \
		NULL)

					    
/*****************************************************************************
 * Capabilities
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to get device capabilities from the
 * given AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param playMedia List of media types that can be played by the device (OUT)
 * @param recMedia List of media types that can be recorded by the device (OUT)
 * @param recQualityModes List of quality modes supported by the device (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_getdevicecapabilities(upnpDevice, \
						     errorString, \
						     instanceID, \
						     playMedia, \
						     recMedia, \
						     recQualityModes) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
		CG_UPNP_AVT_ACTION_GET_DEVICECAPABILITIES, \
		errorString, \
		CG_UPNP_AVT_ARG_INSTANCEID, \
		instanceID, \
		CG_UPNP_AVT_ARG_PLAYMEDIA, \
		playMedia, \
		CG_UPNP_AVT_ARG_RECMEDIA, \
		recMedia, \
		CG_UPNP_AVT_ARG_RECQUALITYMODES, \
		recQualityModes, \
		NULL)

/*****************************************************************************
 * Transport settings
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to get transport settings from the
 * given AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param playMode Current playing mode (OUT)
 * @param recQualityMode Current recording quality mode (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_gettransportsettings(upnpDevice, \
						    errorString, \
						    instanceID, \
						    playMode, \
						    recQualityMode) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
		CG_UPNP_AVT_ACTION_GET_TRANSPORTSETTINGS, \
		errorString, \
		CG_UPNP_AVT_ARG_INSTANCEID, \
		instanceID, \
		CG_UPNP_AVT_ARG_PLAYMODE, \
		playMode, \
		CG_UPNP_AVT_ARG_RECQUALITYMODE, \
		recQualityMode, \
		NULL)

/*****************************************************************************
 * Media transport control
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to stop the current media on the
 * given AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_stop(upnpDevice, \
				    errorString, \
				    instanceID) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_STOP, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	NULL)

/**
 * Invoke an action on the upnp device to start playing the current media on
 * the given AVT instance
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param speed Playing speed (OUT)
 */
#define cg_upnp_av_avt_control_play(upnpDevice, \
				    errorString, \
				    instanceID, \
				    speed) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_PLAY, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	CG_UPNP_AVT_ARG_SPEED, \
	speed, \
	NULL)

/**
 * Invoke an action on the upnp device to pause playing the current media on
 * the given AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_pause(upnpDevice, \
				     errorString, \
				     instanceID) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_PAUSE, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	NULL)

/**
 * Invoke an action on the upnp device to start recording on the given
 * AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_record(upnpDevice, \
				      errorString, \
				      instanceID) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_RECORD, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	NULL)

/**
 * Invoke an action on the upnp device to seek the current media on the
 * given AVT instance to a given position
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param unit Seek mode (OUT)
 * @param target The seek target (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_seek(upnpDevice, \
				    errorString, \
				    instanceID, \
				    unit, \
				    target) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_SEEK, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	CG_UPNP_AVT_ARG_UNIT, \
	unit, \
	CG_UPNP_AVT_ARG_TARGET, \
	target, \
	NULL)

/**
 * Invoke an action on the upnp device to skip the current media on the
 * given AVT instance to the next media in the playlist.
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_next(upnpDevice, \
				    errorString, \
				    instanceID) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_NEXT, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	NULL)

/**
 * Invoke an action on the upnp device to skip the current media on the
 * given AVT instance to the previous media in the playlist.
 *
 * @note Mandatory action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_previous(upnpDevice, \
					errorString, \
					instanceID) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_PREVIOUS, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	NULL)

/*****************************************************************************
 * Mode setting
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to set the play mode for the current
 * media for the given AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param newPlayMode The new play mode (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_setplaymode(upnpDevice, \
					   errorString, \
					   instanceID, \
					   newPlayMode) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_SET_PLAYMODE, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	CG_UPNP_AVT_ARG_NEWPLAYMODE, \
	newPlayMode, \
	NULL)

/**
 * Invoke an action on the upnp device to set the record mode for the given
 * AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param newRecordQualityMode The new record quality mode (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_setrecordqualitymode(upnpDevice, \
						    errorString, \
						    instanceID, \
						    newRecordQualityMode) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_SET_RECORDQUALITYMODE, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	CG_UPNP_AVT_ARG_NEWRECORDQUALITYMODE, \
	newRecordQualityMode, \
	NULL)
	

/*****************************************************************************
 * Current transport action settings
 *****************************************************************************/

/**
 * Invoke an action on the upnp device to get the current transport actions
 * for the given given AVT instance
 *
 * @note Optional action
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The AV transport instance id (IN)
 * @param actions Currently active transport actions (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_avt_control_getcurrenttransportactions(upnpDevice, \
							  errorString, \
							  instanceID, \
							  actions) \
	\
	cg_upnp_av_avt_control_invokeaction(upnpDevice, \
	CG_UPNP_AVT_ACTION_GET_CURRENTTRANSPORTACTIONS, \
	errorString, \
	CG_UPNP_AVT_ARG_INSTANCEID, \
	instanceID, \
	CG_UPNP_AVT_ARG_ACTIONS, \
	actions, \
	NULL)

#endif /* CG_UPNP_AV_CAVTCONTROLPOINT_H */
