/**
    UPnP/AV GnomeVFS browser (Command Line)

    This file contains a wrapper, which uses DBUS to control media playing
    engine.

    Copyright 2006 Nokia Corporation. All rights reserved.
	
    Contact: Aapo Makela <aapo.makela@nokia.com>
    
    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License version 2.1 as 
    published by the Free Software Foundation.
  
    This library is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
    General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <string.h>
#include "player.h"

gboolean player_play_uri(DBusConnection *conn, const gchar *arg_uri, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	gboolean result = FALSE;
	gchar *uri = NULL;
	gchar *query_begin = NULL;
	
	g_print("player_play_uri %s\n", arg_uri);
	
	g_return_val_if_fail(conn != NULL, FALSE);
	
	message = dbus_message_new_method_call(
            PLAYER_DBUS_ENGINE_SERVICE_NAME,
            PLAYER_DBUS_ENGINE_OBJECT_NAME,
            PLAYER_DBUS_ENGINE_INTERFACE_NAME,
            "play_media" );

        g_return_val_if_fail(message != NULL, FALSE);
	
	/* It seems that mediaengine can't handle URIs which have query part,
	   so just strip the query - makes things to work e.g. with WMC */
	uri = g_strdup(arg_uri);
	query_begin = strstr(uri, "?");
	if (query_begin != NULL)
	{
		query_begin[0] = 0;
	}
	
	result = dbus_message_append_args(
			message,
			DBUS_TYPE_STRING, &uri,
			DBUS_TYPE_INVALID );
	
	if (query_begin != NULL)
	{
		query_begin[0] = '?';
	}
	
	if (result == FALSE) {
		g_free(uri);
		return FALSE;
	}
	
        dbus_error_init( &dbus_error );

	reply = dbus_connection_send_with_reply_and_block(
	      conn, message, PLAYER_PLAY_TIMEOUT,
              &dbus_error);
	
	g_free(uri);
	dbus_message_unref(message);
	if (reply != NULL) dbus_message_unref(reply);
	
	return TRUE;
}

gboolean player_stop(DBusConnection *conn, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	dbus_uint32_t client_id = 0;

	g_return_val_if_fail(conn != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"stop" );

	g_return_val_if_fail(message != NULL, FALSE);
	
	dbus_message_set_no_reply( message, TRUE );
		
	result = dbus_connection_send(conn, 
				      message, &client_id);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_connection_flush(conn);
	
	dbus_message_unref(message);

	return TRUE;
}

gboolean player_pause(DBusConnection *conn, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	
	g_return_val_if_fail(conn != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"pause" );
	
	g_return_val_if_fail(message != NULL, FALSE);
	
        reply = dbus_connection_send_with_reply_and_block(
			conn, 
			message, -1, &dbus_error);
	
        dbus_message_unref( message );
	if (reply != NULL) dbus_message_unref( reply );
	
	return TRUE;
}

gboolean player_set_mute(DBusConnection *conn, gboolean muted, GError **err)
{
	DBusError dbus_error;
	dbus_uint32_t client_id = 0;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	
	g_return_val_if_fail(conn != NULL, FALSE);
	
	dbus_error_init( &dbus_error );
	
	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"mute" );

	g_return_val_if_fail(message != NULL, FALSE);
	
	result = dbus_message_append_args(
			message,
			DBUS_TYPE_BOOLEAN, &muted,
			DBUS_TYPE_INVALID);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_message_set_no_reply(message, TRUE);

	result = dbus_connection_send(
		conn, 
		message, &client_id);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_message_unref(message);
	
	return TRUE;
}

gboolean player_set_volume(DBusConnection *conn, gint value, GError **err)
{
	DBusError dbus_error;
	dbus_uint32_t client_id = 0;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	gdouble volume_level;
	
	g_return_val_if_fail(conn != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"set_volume" );

	dbus_message_set_no_reply( message, TRUE );

	volume_level = (gdouble)(value / PLAYER_VOLUME_MODIFIER);
	
        result = dbus_message_append_args(
			message,
			DBUS_TYPE_DOUBLE, &volume_level,
			DBUS_TYPE_INVALID);
        
	result = dbus_connection_send(
            conn, message, &client_id);
        
	dbus_connection_flush(conn);

        dbus_message_unref( message );
	
	return TRUE;
}

gboolean player_get_position(DBusConnection *conn, gint *pos, gint *length, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	DBusMessageIter iter;
	dbus_int32_t int32;
	
	g_return_val_if_fail(conn != NULL, FALSE);
	
	message = dbus_message_new_method_call(
            PLAYER_DBUS_ENGINE_SERVICE_NAME,
            PLAYER_DBUS_ENGINE_OBJECT_NAME,
            PLAYER_DBUS_ENGINE_INTERFACE_NAME,
            "get_position");

        g_return_val_if_fail(message != NULL, FALSE);

	dbus_error_init(&dbus_error);

	reply = dbus_connection_send_with_reply_and_block(
	      conn, 
	      message, PLAYER_PLAY_TIMEOUT,
              &dbus_error);
	
	g_return_val_if_fail(reply != NULL, FALSE);

	if (reply != NULL)
	{
		dbus_message_iter_init(reply, &iter);
		dbus_message_iter_get_basic( &iter, &int32 );
		*pos = (gint)int32;
		
		if (length != NULL && dbus_message_iter_next(&iter) == TRUE)
		{
			dbus_message_iter_get_basic( &iter, &int32 );
			*length = (gint)int32;
		}
	}
	
	dbus_message_unref(message);
	if (reply != NULL) dbus_message_unref(reply);

	return TRUE;
}
