#include <unistd.h>
#include <sys/socket.h>
#include <errno.h>
#include <glib/gstdio.h>
#include "gwebbus.h"
#include "gweb.h"

// PRIVATE DECLARATIONS

typedef struct {
    guint msg_id;
    GWebBusCallback callback;
    gpointer user_data;
} GWebBusCallbackData;

typedef struct {
    guint msg_id;
    gint msg_priority;
    guint data_len;
} GWebBusMessageHeader;

typedef struct {
    GWebBusCallbackData *callback_data;
    GWebBus * self;
    guint event_source_id;
    char buf[1];
} GWebBusMessageHeader2;

typedef struct {
    GWebBusMessageHeader header;
    guchar data[1]; // data will actually be of <data_len> bytes
} GWebBusMessage;

#define G_WEB_BUS_MAX_DATAGRAM_SIZE (sizeof(GWebBusMessageHeader) + G_WEB_BUS_MAX_MSG_SIZE)
#define G_WEB_BUS_MSG_HANDLER_DATA_SIZE (G_WEB_BUS_MAX_DATAGRAM_SIZE + sizeof(GWebBusCallbackData))
#define G_WEB_BUS_SOCKET_FORMAT "/tmp/browser-webbus-%s"

// GWebBus object type definition
G_DEFINE_TYPE(GWebBus, g_web_bus, G_TYPE_OBJECT);

static void create_address(struct sockaddr_un* addr, const gchar* name);
static gboolean watch_handler(GIOChannel* source, GIOCondition condition, GWebBus* self);

static gboolean source_remove_hash(gpointer key, gpointer value, gpointer user_data)
{
  if (key)
    g_source_remove ((guint)(key));
  return TRUE;
}

static void
g_web_bus_finalize(GObject* gobject)
{
    GWebBus* self = G_WEB_BUS(gobject);
    g_return_if_fail (self);
    self->shutdown = TRUE;

    if (self->channel) {
        g_io_channel_unref(self->channel);
        if (self->source) {
            g_source_destroy(self->source);
            self->source = NULL;
        }
        self->channel = NULL;
    }
    g_unlink(self->address.sun_path); // remove the socket from filesystem

    g_debug("wb-fin:%p:%s", self, self->name);
    // Remove any pending message handler event sources.
    g_hash_table_foreach_remove
        (self->callback_event_sources, source_remove_hash, NULL);
    g_hash_table_destroy (self->callback_event_sources);
    self->callback_event_sources = NULL;

    if (self->callbacks) {
        g_array_free(self->callbacks, TRUE);
        self->callbacks = NULL;
    }

    if (self->name) {
        g_free(self->name);
        self->name = NULL;
    }
  
    self->presenddata = NULL;
    self->prerecvdata = NULL;
    self->presendhandler = NULL;
    self->prerecvhandler = NULL;

    G_OBJECT_CLASS(g_web_bus_parent_class)->finalize(gobject);
}

static void
g_web_bus_class_init(GWebBusClass* klass)
{
    GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->finalize = g_web_bus_finalize;
}

static void
g_web_bus_init(GWebBus* self)
{
    self->channel = NULL;
    self->name = NULL;
    // preallocate 16 slots for messages, cleared upon allocation
    self->callbacks = g_array_sized_new(FALSE, TRUE, sizeof(GWebBusCallbackData), 16);
    self->callback_event_sources = g_hash_table_new (g_direct_hash, g_direct_equal);
    self->source = NULL;
    self->presenddata = NULL;
    self->prerecvdata = NULL;    
    self->presendhandler = NULL;
    self->prerecvhandler = NULL;
    self->shutdown = FALSE;
}

static gboolean
g_web_bus_init_socket(GWebBus* self)
{
    // create UNIX datagram socket
    gint sock = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (sock < 0) {
        return FALSE;
    }

    if (self->channel) {
        g_io_channel_unref(self->channel);
        self->channel = NULL;
    }

    if (self->address.sun_path)
      g_unlink(self->address.sun_path); // remove the socket from filesystem

    // bind the socket to known unix file self->name
    create_address(&self->address, self->name);
    g_unlink(self->address.sun_path);
    if (bind(sock, (struct sockaddr *) &self->address, sizeof(self->address)) == -1) {
        close(sock);
        return FALSE;
    }
    // allow any user to write here
    g_chmod(self->address.sun_path, S_IRWXU|S_IRWXG|S_IRWXO);

    // create I/O channel for polling the socket
    self->channel = g_io_channel_unix_new(sock);
    // disallow buffering and encoding
    if (g_io_channel_set_encoding(self->channel, NULL, NULL) != G_IO_STATUS_NORMAL)
        return FALSE;
    g_io_channel_set_buffered(self->channel, FALSE);
    // yield ownership of the socket
    g_io_channel_set_close_on_unref(self->channel, TRUE);
    // make the channel non-blocking so watch_handler() won't block
    if (g_io_channel_set_flags(self->channel, G_IO_FLAG_NONBLOCK, NULL) != G_IO_STATUS_NORMAL)
        return FALSE;
    // add a watch function to io channel
    guint id = g_io_add_watch_full(self->channel,
                                   G_PRIORITY_HIGH,
                                   G_IO_IN|G_IO_HUP,
                                   (GIOFunc) watch_handler,
                                   self,
                                   NULL);
    if (!id)
      return FALSE;

    self->source = g_main_context_find_source_by_id(g_main_context_default(), id);

    return TRUE;
}

GWebBus*
g_web_bus_new(const gchar* name)
{
    g_return_val_if_fail(name, NULL);

    GWebBus* self = g_object_new(G_TYPE_WEB_BUS, NULL);
    g_return_val_if_fail(self, NULL);

    memset(&(self->address), 0, sizeof(self->address));

    self->name = g_strdup(name);
    gboolean sock_init = g_web_bus_init_socket(self);
    if (!sock_init) {
        g_object_unref(self);
        return NULL;
    }
    g_debug("wb-new:%p:%s", self, self->name);

    return self;
}

gboolean
g_web_bus_register_callback(GWebBus* self,
                            guint msg_id,
                            GWebBusCallback callback,
                            gpointer user_data)
{
    g_return_val_if_fail(self, FALSE);
    g_return_val_if_fail(self->callbacks, FALSE);
    g_return_val_if_fail(callback, FALSE);

    // ensure that the array has enough slots
    if (msg_id+1 > self->callbacks->len) {
        self->callbacks = g_array_set_size(self->callbacks, msg_id+1);
    }

    // create callback information structure and append it to callback list
    GWebBusCallbackData* data = &g_array_index(self->callbacks, GWebBusCallbackData, msg_id);
    g_return_val_if_fail(data, FALSE);
    data->msg_id = msg_id;
    data->callback = callback;
    data->user_data = user_data;
    return TRUE;
}

gboolean
g_web_bus_is_registered(GWebBus* self, guint msg_id)
{
    g_return_val_if_fail(self, FALSE);

    if (msg_id < self->callbacks->len) {
      GWebBusCallbackData* data = &g_array_index(self->callbacks, GWebBusCallbackData, msg_id);
      if (data && data->callback) {
        return TRUE;
      }
    }
    return FALSE;
}

gboolean
g_web_bus_send_message_with_priority(GWebBus* self,
                                     const gchar* name,
                                     guint msg_id,
                                     gint priority,
                                     const gpointer data,
                                     guint data_len)
{
    g_return_val_if_fail(self, FALSE);
    g_return_val_if_fail(name, FALSE);
    g_return_val_if_fail(data || (data_len == 0), FALSE);
    g_return_val_if_fail(data_len <= G_WEB_BUS_MAX_MSG_SIZE, FALSE);
    g_return_val_if_fail (self->channel, FALSE);

    // allocate a message datagram
    ssize_t msg_len = sizeof(GWebBusMessageHeader) + data_len;
    GWebBusMessage* msg = (GWebBusMessage *) g_alloca(msg_len); // allocated from stack
    g_return_val_if_fail(msg, FALSE);

    // set the fields of the datagram
    msg->header.msg_id = msg_id;
    msg->header.msg_priority = priority;
    msg->header.data_len = data_len;
    memcpy(msg->data, data, data_len); // this is unavoidable if we want to send
                                       // both id and data as single datagram
    // send the datagram
    struct sockaddr_un addr;
    create_address(&addr, name);
    g_return_val_if_fail(self->channel, FALSE);
    gint sock = g_io_channel_unix_get_fd(self->channel);
    ssize_t len = 0;

    // handle presend stuff, regardless if message was sent successfuly or not
    if (self->presendhandler && self->source) {
         GTimeVal now = { 0 };
         g_source_get_current_time(self->source, &now);
         (*self->presendhandler)(msg_id, data, data_len, &now, self->presenddata);
    }

    if (((len = sendto(sock, msg, msg_len, MSG_DONTWAIT, (const struct sockaddr *) &addr, sizeof(addr))) == -1)
          || len < msg_len) {
       g_warning("Message sending from \"%s\" to socket \"%s\" failed, len:%i error: %s (%d)",
                 self->name, name, len, strerror(errno), errno);
       return FALSE;
    }

    return TRUE;
}

gboolean
g_web_bus_send_message(GWebBus* self,
                       const gchar* name,
                       guint msg_id,
                       const gpointer data,
                       guint data_len)
{
    return g_web_bus_send_message_with_priority (self,
                                                 name,
                                                 msg_id,
                                                 G_PRIORITY_DEFAULT_IDLE,
                                                 data,
                                                 data_len);
}

gboolean
g_web_bus_update_channel(GWebBus* self)
{
    return g_web_bus_init_socket(self);
}

static void
create_address(struct sockaddr_un* addr, const gchar* name)
{
    // sanitize the supplied name to contain only filename-safe characters
    gchar* sname = g_ascii_strdown(name, sizeof(addr->sun_path));
    g_strcanon(sname, "abcdefghijklmnopqrstuvwxyz0123456789-_.", '_');

    // create uniform unix socket name for given name
    addr->sun_family = AF_UNIX;
    g_snprintf(addr->sun_path, sizeof(addr->sun_path), G_WEB_BUS_SOCKET_FORMAT, sname);

    // release sanitized name
    g_free(sname);
}

/** This callback is used for freeing GWebBusMessageHeader2 instances
  * attached to message callback event sources (see watch_handler).
  * @param data A GWebBusMessageHeader2 instance.
  */
static void
watch_handler_destroy_data_cb(GWebBusMessageHeader2 *data)
{
  g_return_if_fail(data);
  g_return_if_fail(data->self);
  if (data->self->callback_event_sources) {
    // If we are shutting down and deleting event sources,
    // we don't need to delete the current event source here
    if (!(data->self->shutdown)) {
      g_hash_table_remove (data->self->callback_event_sources,
                           (gpointer)(data->event_source_id));
    }
  } else {
    g_warning("callback_event_sources == 0, data:%p, data->self:%p, srcid:%i", data, data->self, data->event_source_id);
  }
  // release the allocated block
  g_slice_free1(G_WEB_BUS_MSG_HANDLER_DATA_SIZE, data);
}

static gboolean
watch_handler_timeout(GWebBusMessageHeader2 *data)
{
  g_return_val_if_fail(data, FALSE);

  if (!g_source_is_destroyed (g_main_current_source ())) {
      GWebBusMessage *msg = (GWebBusMessage *)data->buf;
      if (msg) {
          GWebBusCallbackData *call_data = data->callback_data;
          if (call_data) {
              (*call_data->callback)(msg->header.msg_id, msg->data, msg->header.data_len, call_data->user_data);
          }
      }
  }

  // NOTE: memory allocated to data is automatically released in
  //       watch_handler_destroy_data_cb after we return FALSE.
  return FALSE;
}

static gboolean
watch_handler(GIOChannel* source, GIOCondition condition, GWebBus* self)
{
    gint recv_len, sock;
    struct sockaddr_un from;
    socklen_t from_len = sizeof(from);
    (void) source;
    g_return_val_if_fail (self && self->channel, FALSE);

    if (condition != G_IO_IN) {
        // some error condition, stop watching
        return FALSE;
    }

    // allocate buffer for incoming datagram
    GWebBusMessageHeader2* msg2 = 
            (GWebBusMessageHeader2 *) g_slice_alloc(G_WEB_BUS_MSG_HANDLER_DATA_SIZE);
    if (!msg2) {
        // OOM: can not handle the message
        g_error("Out of memory while receiving messages");
        return FALSE;
    }

    GWebBusMessage *msg = (GWebBusMessage *)msg2->buf;
    // read the incoming datagram
    sock = g_io_channel_unix_get_fd(self->channel);
    recv_len = recvfrom(sock, msg, G_WEB_BUS_MAX_DATAGRAM_SIZE, 0,
                        (struct sockaddr *) &from, &from_len);
    if (recv_len < 0) {
        g_warning("Socket got closed: %s (%d)", strerror(errno), errno);
        if (errno == EAGAIN) // resource unavailable, reset channel
            g_web_bus_update_channel(self);
        // socket got closed, stop watching
        g_slice_free1(G_WEB_BUS_MSG_HANDLER_DATA_SIZE, msg2);
        return FALSE;
    }

    // call prerecv if any
    if (self->source && self->prerecvhandler) {
        GTimeVal now;
        g_source_get_current_time(self->source, &now);

        (*self->prerecvhandler)(msg->header.msg_id, msg->data, msg->header.data_len, &now, self->prerecvdata);
    }
    // modest sanity check for message datagram
    if (msg->header.data_len != (recv_len - sizeof(GWebBusMessageHeader))) {
        g_warning("Dropping invalid datagram");
    }
    else {
        // look for handler function for the message
        GWebBusCallbackData* data = NULL;
        if (msg->header.msg_id < self->callbacks->len) {
            data = &g_array_index(self->callbacks, GWebBusCallbackData, msg->header.msg_id);
        }
        if (data && data->callback) {
            msg2->callback_data = data;
            msg2->self = self;
            // found a handler, call the supplied callback function with callback data and message data
            msg2->event_source_id =
                g_idle_add_full(msg->header.msg_priority,
                                (GSourceFunc)watch_handler_timeout,
                                msg2,
                                (GDestroyNotify)watch_handler_destroy_data_cb);
            // Event source IDs are stored in a hash table while the event
            // source is active. This way we can remove all active event
            // sources when this GWebBus instance is being finalized.
            g_hash_table_insert (self->callback_event_sources,
                                 (gpointer)(msg2->event_source_id), 0);

            return TRUE;
        }
#ifdef DEBUG
        // handler was not found, so we'll have to drop the message here
        g_debug("Dropping unhandled message: %d\n", msg->header.msg_id);
#endif
    }
    // release the allocated block after callback has finished
    g_slice_free1(G_WEB_BUS_MSG_HANDLER_DATA_SIZE, msg2);
    return TRUE;
}

void
g_web_bus_register_pre_send(GWebBus *self,
                            GWebBusPreHandler handler,
                            gpointer data)
{
    self->presendhandler = handler;
    self->presenddata = data;
}

void
g_web_bus_register_pre_recv(GWebBus *self, 
                            GWebBusPreHandler handler,
                            gpointer data)
{
    self->prerecvhandler = handler;
    self->prerecvdata = data;
}
