/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Leonid Zolotarev <leonid.zolotarev@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib-object.h>
#include "gwebengine.h"
#include "gwebtransferitem.h"
#include "gwebhistorylist.h"
#include "gwebpluginlist.h"
#include "gwebtransferitemlist.h"
#include "gwebpassman.h"

#ifndef __GWEB_H__
#define __GWEB_H__

G_BEGIN_DECLS

/** GCookieAcceptance.
  * Cookies Support.
  */
typedef enum
{
	G_ACCEPT_COOKIE_ALWAYS = 0, /**< Accept and handle all cookies except illegal domain cookies */
	G_ACCEPT_COOKIE_PROMPT,     /**< Confirm cookie before processing them */
	G_ACCEPT_COOKIE_NEVER       /**< Refuse all cookies */
} GCookieAcceptance;

/** GPopupAcceptance.
  * Popup Support.
  */
typedef enum
{
	G_ACCEPT_POPUP_ALWAYS = 0, /**< Accept popup support Always */
	G_ACCEPT_POPUP_PROMPT,     /**< Ask before popping up */
	G_ACCEPT_POPUP_NEVER       /**< No popups at all */
} GPopupAcceptance;

typedef enum
{
	G_CALLBACK_NO_ACTION = 0,
	G_CALLBACK_OOM_ACTION = 1,
	G_CALLBACK_RESET_ACTION = 2
} GCallbackAction;


/** GMemoryCache.
  * Memory Cache.
  */
typedef enum
{
	G_MEMORY_CACHE_AUTOMATIC = 0, /**< Automatic memory cahe */
	G_MEMORY_CACHE_LARGE,         /**< Memory cache 4096kB */
	G_MEMORY_CACHE_MEDIUM,	      /**< Memory cache 2048kB */
	G_MEMORY_CACHE_DEFAULT,       /**< Memory cache 1024kB */
	G_MEMORY_CACHE_SMALL,         /**< Memory cache 512kB */
	G_MEMORY_CACHE_OFF            /**< Memory cahe off */
} GMemoryCache;

/** GMemoryCacheSize.
  * Memory Cache Size.
  */
typedef enum
{
	G_MEMORY_CACHE_AUTOMATIC_SIZE	= -1,	 /**< Automatic memory cahe */
	G_MEMORY_CACHE_LARGE_SIZE	= 40960, /**< Memory cache 40960kB */
	G_MEMORY_CACHE_MEDIUM_SIZE	= 20480, /**< Memory cache 20480kB */
	G_MEMORY_CACHE_DEFAULT_SIZE	= 10240, /**< Memory cache 10240kB */
	G_MEMORY_CACHE_SMALL_SIZE	= 5120,	 /**< Memory cache 5120kB */
	G_MEMORY_CACHE_OFF_SIZE		= 0	 /**< Memory cahe off */
} GMemoryCacheSize;

/** GWeb
  */

#define G_TYPE_WEB             (g_web_get_type ())
#define G_WEB(obj)             (G_TYPE_CHECK_INSTANCE_CAST ((obj), G_TYPE_WEB, GWeb))
#define G_WEB_CLASS(vtable)    (G_TYPE_CHECK_CLASS_CAST ((vtable), G_TYPE_WEB, GWebIface))
#define G_IS_WEB(obj)          (G_TYPE_CHECK_INSTANCE_TYPE ((obj), G_TYPE_WEB))
#define G_IS_WEB_CLASS(vtable) (G_TYPE_CHECK_CLASS_TYPE ((vtable), G_TYPE_WEB))
#define G_WEB_GET_CLASS(inst)  (G_TYPE_INSTANCE_GET_INTERFACE ((inst), G_TYPE_WEB, GWebIface))

typedef struct _GWeb GWeb; /* dummy object */
typedef struct _GWebIface GWebIface;

/** @struct GWebIface.
  * Defines web object interface.
  */
struct _GWebIface {
	GTypeInterface parent;

	void (*destroy) (GWeb *self);
    GWebEngine* (*new_web_engine_window) (GWeb *self);
    GWebEngine* (*new_web_engine_window_with_context) (GWeb *self, gpointer context, gboolean leave_tracks, gboolean without_history);

    gboolean (*web_set_pref) (GWeb *self, const gchar* name, GValue* value);
    gboolean (*web_get_pref) (GWeb *self, const gchar* name, GValue* value);
    
    GWebTransferItem* (*new_web_transfer_item) (GWeb *self, const gchar* url, const gchar* filename);
    GWebTransferItem* (*initialize_web_transfer_item) (GWeb *self, GObject *web_transfer_item);
    GWebTransferItemList* (*get_web_transfer_item_list) (GWeb *self);
    GWebStatus (*set_home_page) (GWeb *self, const gchar* home_page);
    gchar* (*get_home_page) (GWeb *self);
    GWebStatus (*set_language) (GWeb *self, const gchar* language);
    void (*reset_cache_history_length) (GWeb *self);
    gchar* (*get_language) (GWeb *self);
    gint (*get_maximum_cache_size) (GWeb *self);
    GWebStatus (*set_default_home_page) (GWeb *self, const gchar* default_page);
    gchar* (*get_default_home_page) (GWeb *self);
    gboolean (*home_page_set) (GWeb *self);
    void (*set_frame_borders_enabled) (GWeb *self, gboolean enabled);
    gboolean (*get_frame_borders_enabled) (GWeb *self);
    GWebStatus (*set_location_completion_suffixes) (GWeb *self, const gchar* suffixes);
    gchar* (*get_location_completion_suffixes) (GWeb *self);
    GWebStatus (*set_plugins_enabled) (GWeb *self, gboolean enabled);
    GWebStatus (*set_oom_cb) (GWeb *self, GFunc func);
    GWebStatus (*set_ssl2_support) (GWeb *self, gboolean enabled);
    gboolean (*get_ssl2_support) (GWeb *self);
    gboolean (*get_plugins_enabled) (GWeb *self);
    GWebStatus (*set_js_enabled) (GWeb *self, gboolean enabled);
    gboolean (*get_js_enabled) (GWeb *self);
    void (*set_memory_cache) (GWeb *self, GMemoryCache acceptance);
    GMemoryCache (*get_memory_cache) (GWeb *self);
    GWebStatus (*set_cache_size) (GWeb *self, gint size);
    gint (*get_cache_size) (GWeb *self);
    GWebStatus (*set_browse_offline) (GWeb *self, gboolean offline);
    gboolean (*get_browse_offline) (GWeb *self);
    GWebStatus (*set_automatic_redirection_enabled) (GWeb *self, gboolean enabled);
    gboolean (*get_automatic_redirection_enabled) (GWeb *self);
	GWebStatus (*set_load_images) (GWeb *self, GWebEngineImagePolicy level);
	GWebEngineImagePolicy (*get_load_images) (GWeb *self); 
    void (*disconnect) (GWeb *self);
    GWebStatus (*clear_cache) (GWeb *self);
    GWebPluginList* (*get_plugins_list) (GWeb *self);
    GWebHistoryList* (*get_history_list) (GWeb *self);
    GWebStatus (*set_new_window_loads) (GWeb *self, gboolean loads);
    gboolean (*get_new_window_loads) (GWeb *self);
    void (*set_cookie_acceptance) (GWeb *self, GCookieAcceptance acceptance);
    GCookieAcceptance (*get_cookie_acceptance) (GWeb *self);
    void (*set_popup_acceptance) (GWeb *self, GPopupAcceptance acceptance);
    GPopupAcceptance (*get_popup_acceptance) (GWeb *self);
    GWebStatus (*suspend) (GWeb *self);
    gboolean (*suspended) (GWeb *self);
    void (*resume) (GWeb *self);
    void (*set_ua_string) (GWeb *self, gchar *ua_string);
    GWebStatus (*set_user_agent_id) (GWeb *self, gint ua_id);
    gint (*get_user_agent_id) (GWeb *self);
    void (*enable_url_loading) (GWeb *self, gboolean enable);
    void (*free_memory) (GWeb *self, gboolean progressive);
    GWebStatus (*store_transfers) (GWeb *self);
    GWebStatus (*restore_transfers) (GWeb *self);
    GWebStatus (*delete_all_cookies) (GWeb *self);
    GWebStatus (*confirm_cookie) (GWeb *self, gpointer context, gboolean accept, GWebCookieAction action);
    GObject* (*get_global) (GWeb *self);
    GWebCertificateMessage (*get_certificate_enum_message) (GWeb *self, GObject *context);
    gint (*username_matchinfo_get_matches) (GWeb *self, GObject *info);
    void (*username_matchinfo_delete) (GWeb *self, GObject *info, gint index);
    GWebStatus (*username_matchinfo_get_username) (GWeb *self, GObject *info, gint index, gchar** username);
    GWebPasswordManager* (*get_password_manager) (GWeb *self, const gchar* server, gint index);
    GWebStatus (*store_personal_data)(GWeb *self, const gchar* server, const gchar* username, const gchar* password);
    GWebStatus (*delete_personal_data)(GWeb *self, const gchar* server, const gchar* username);
    GWebCertificateReaction (*perform_crtificate_context_action) (GWeb *self,
                                                                    GObject *context,
                                                                    GWebCertificateAction action,
                                                                    GObject *cert);
    GWebStatus (*get_certificate_context_iter) (GWeb *self, GObject* context, GObject **iterator);
    GObject* (*get_certificate_iter) (GWeb *self, GObject** iterator);
    void (*certificate_free) (GWeb *self, GObject* cert);
    void (*certificate_iter_free) (GWeb *self, GObject* iterator);
    void (*notify_plugins) (GWeb *self, gint event);	
    GWebStatus (*clear_passwords) (GWeb *self);
    GWebStatus (*clear_authenticated_sessions) (GWeb *self);
    void (*clear_personal_data) (GWeb *self);
    GWebStatus (*clear_all_history) (GWeb *self);
    GWebStatus (*set_remember_password) (GWeb *self, gboolean rpassword);
    gboolean (*get_remember_password) (GWeb *self);
    GWebStatus (*set_remember_password_for_forms) (GWeb *self, gboolean rpassword);
    gboolean (*get_remember_password_for_forms) (GWeb *self);
    gboolean (*get_plugin_enabled) (GWeb *self, gchar* name);
    GWebStatus (*set_plugin_enabled) (GWeb *self, gchar* name, gboolean enable);
    GWebStatus (*set_plugin_ignore_filepath) (GWeb *self);
    void (*notify_service) (GWeb *self, const gchar *service_name, gpointer *context, const gchar *topic, gchar *data);
    void (*create_web_buses) (GWeb* self);
};

GType g_web_get_type (void);
/** Set the preference value.
  * @param self Object pointer
  * @param name Name of the preference
  * @param value #GValue with preference value.
  * @return TRUE on success, FALSE otherwise
  */
gboolean g_web_set_pref (GWeb *self, const gchar* name, GValue* value);
/** Get the preference value.
  * @param self Object pointer
  * @param name Name of the preference
  * @param value A #GValue to place the result 
  * @return TRUE on success, FALSE otherwise
  */
gboolean g_web_get_pref (GWeb *self, const gchar* name, GValue* value);
/** Destroy object.
  * @param self Object pointer
  */
void g_web_destroy (GWeb *self);

/** Create Web Engine object.
  * @param self Object pointer
  * @return Web Engine object
  */
GWebEngine* g_web_new_web_engine_window (GWeb *self);

/** Create Web Engine object with context.
  * @param self Object pointer
  * @param context Context from where the Web Engine was requested
  * @param leave_tracks TRUE create a new widget, where visited pages won't be listed in the history
  *        and their contents won't be stored in the cache. else FALSE.
  * @param without_history TRUE Create a new widget without history. Visited pages will not be listed in the global history
  * @return Web Engine object
  */
GWebEngine* g_web_new_web_engine_window_with_context (GWeb *self, gpointer context, gboolean leave_tracks, gboolean without_history);

/** Create Web Transfer Item.
  * @param self Object pointer
  * @param url The URL to download from 
  * @param filename The file to save to 
  * @return Web Transfer Item object
  */
GWebTransferItem* g_web_new_web_transfer_item (GWeb *self, const gchar* url, const gchar* filename);

/** Initialized Web Transfer Item.
  * @param self Object pointer
  * @param web_transfer_item Transfer Item pointer
  * @return Web Transfer Item object
  */
GWebTransferItem* g_web_initialize_web_transfer_item (GWeb *self, GObject *web_transfer_item);

/** Get Web Transfer Item List.
  * @param self Object pointer
  * @return Web Transfer Item List object
  */
GWebTransferItemList* g_web_get_web_transfer_item_list (GWeb *self);

/** Set the Home page URL.
  * @param self Object pointer
  * @param home_page Home page to be set
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_home_page (GWeb *self, const gchar* home_page);

/** Get the Home page URL.
  * @param self Object pointer
  * @return Returns home page URL string
  */
gchar* g_web_get_home_page (GWeb *self);

/** Set language.
  * @param self Object pointer
  * @param language Language to be set
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_language (GWeb *self, const gchar* language);

/** Get language.
  * @param self Object pointer
  * @return Returns language string
  */
gchar* g_web_get_language (GWeb *self);

/** Reset the cache size and history length.
  * @param self Object pointer
  */
void g_web_reset_cache_history_length (GWeb *self);

/** Get the Maximum Cache size.
  * @param self Object pointer
  * @return Returns Maximum Cache size.
  */
gint g_web_get_maximum_cache_size (GWeb *self);

/** Set the Default home page URL.
  * @param self Object pointer
  * @param default_page Default home page to be set
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_default_home_page (GWeb *self, const gchar* default_page);

/** Get the Default Home page URL.
  * @param self Object pointer
  * @return Returns default home page url produced by Nokia or a service provider.
  */
gchar* g_web_get_default_home_page (GWeb *self);

/** Returns whether Home Page is set or not.
  * @param self Object pointer
  * @return TRUE, if home page is set. FALSE, if not
  */
gboolean g_web_home_page_set (GWeb *self);

/** Set the Frame Borders always be displayed even if the page says otherwise.
  * @param self Object pointer
  * @param enabled TRUE to enable the frame borders always and FALSE otherwise
  */
void g_web_set_frame_borders_enabled (GWeb *self, gboolean enabled);

/** Get whether Frame Borders are enabled or disabled.
  * @param self Object pointer
  * @return TRUE, if frame borders enabled. FALSE, if not
  */
gboolean g_web_get_frame_borders_enabled (GWeb *self);

/** Set the Location Completion Suffixes(eg "com, org" will be added to the URL).
  * @param self Object pointer
  * @param suffixes Suffixes to be set
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_location_completion_suffixes (GWeb *self, const gchar* suffixes);

/** Get the Location Completion Suffixes.
  * @param self Object pointer
  * @return Returns location completion suffixes
  */
gchar* g_web_get_location_completion_suffixes (GWeb *self);

/** Enable/Disable Plugins.
  * @param self Object pointer
  * @param enabled TRUE to enable the plugins, FALSE to disable the plugins
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_plugins_enabled (GWeb *self, gboolean enabled);

/** Enable/Disable SSL2 support.
  * @param self Object pointer
  * @param enabled TRUE to enable SSL2 support, FALSE to disable SSL2 support
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_ssl2_support (GWeb *self, gboolean enabled);

/** Get SSL2 support status
  * @param self Object pointer
  * @return TRUE, if SLL2 support enabled. FALSE, if not
  */
gboolean g_web_get_ssl2_support (GWeb *self);

/** Get whether plugins are enabled or disabled.
  * @param self Object pointer
  * @return TRUE, if plugins enabled. FALSE, if not
  */
gboolean g_web_get_plugins_enabled (GWeb *self);

/** Enable/Disable the ecmascript.
  * @param self Object pointer
  * @param enabled TRUE to enable the ecmascript, FALSE otherwise
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_js_enabled (GWeb *self, gboolean enabled);

/** Get whether ecmascript is enabled or not.
  * @param self Object pointer
  * @return TRUE, if ecmascript is enabled. FALSE, if not
  */
gboolean g_web_get_js_enabled (GWeb *self);

/** Set the Memory Cache value.
  * @param self Object pointer
  * @param acceptance MemoryCache value
  */
void g_web_set_memory_cache (GWeb *self, GMemoryCache acceptance);

/** Get the value of Memory Cache.
  * @param self Object pointer
  * @return Memory Cache.
  */
GMemoryCache g_web_get_memory_cache (GWeb *self);

/** Set the cache size.
  * @param self Object pointer
  * @param size size of the Cache
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_cache_size (GWeb *self, gint size);

/** Get the Cache size.
  * @param self Object pointer
  * @return The size of Cache
  */
gint g_web_get_cache_size (GWeb *self);

/** Enable/Disable offline browsing.
  * @param self Object pointer
  * @param offline TRUE to enable offline browsing, FALSE otherwise
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_browse_offline (GWeb *self, gboolean offline);

/** Get whether offline browsing is enabled or not.
  * @param self Object pointer
  * @return TRUE, if browse offline is enabled, FALSE, if not
  */
gboolean g_web_get_browse_offline (GWeb *self);

/** Enable/Disable the automatic redirection through client pull.
  * @param self Object pointer
  * @param enabled TRUE to enable the automatic redirection, FALSE otherwise
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_automatic_redirection_enabled (GWeb *self, gboolean enabled);

/** Get whether Automatic redirection is enabled or not.
  * @param self Object pointer
  * @return TRUE, if Automatic Redirection is Enabled, FALSE otherwise
  */
gboolean g_web_get_automatic_redirection_enabled (GWeb *self);

/** Set the Image policy value(Always or Only loaded or Never).
  * @param self Object pointer
  * @param level Image policy value
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_load_images (GWeb *self, GWebEngineImagePolicy level);

/** Get the Images policy value(Always or Only loaded or Never).
  * @param self Object pointer
  * @return Image policy
  */
GWebEngineImagePolicy g_web_get_load_images (GWeb *self); 

/** Close all connections. 
  * @param self Object pointer
  */
void g_web_disconnect (GWeb *self);

/** Empty the disk cache.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_clear_cache (GWeb *self);

/** Get the Plugin List object.
  * @param self Object pointer
  * @return Web Plugins List object
  */
GWebPluginList* g_web_get_plugins_list (GWeb *self);

/** Get the history list object.
  * @param self Object pointer
  * @return Web History List object
  */
GWebHistoryList* g_web_get_history_list (GWeb *self);

/** Enable/Disable to load default local home page.
  * @param self Object pointer
  * @param loads TRUE to load default local home page, FALSE to load the defined home page 
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_new_window_loads (GWeb *self, gboolean loads);

/** Get whether default local home page will be loaded or not.
  * @param self Object pointer
  * @return TRUE, if default local home page will be loaded, FALSE otherwise
  */
gboolean g_web_get_new_window_loads (GWeb *self);

/** Set the cookies Acceptance value (Always or Ask first or Never).
  * @param self Object pointer
  * @param acceptance Cookie Acceptance value
  */
void g_web_set_cookie_acceptance (GWeb *self, GCookieAcceptance acceptance);

/** Get the value(Always or Ask first or Never) of cookies Acceptance.
  * @param self Object pointer
  * @return Cookie Acceptance value (Always or Ask first or Never)
  */
GCookieAcceptance g_web_get_cookie_acceptance (GWeb *self);

/** Set the Popup Acceptance value (Always or Ask first or Never).
  * @param self Object pointer
  * @param acceptance Popup Acceptance value
  */
void g_web_set_popup_acceptance (GWeb *self, GPopupAcceptance acceptance);

/** Get the value (Always or Ask first or Never) of Popup Acceptance.
  * @param self Object pointer
  * @return Popup Acceptance value (Always or Ask first or Never)
  */
GPopupAcceptance g_web_get_popup_acceptance (GWeb *self);

/** Suspend the browser and reclaims the memory.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_suspend (GWeb *self);

/** Check if browser was suspended.
  * @param self Object pointer
  * @return TRUE, if browser was suspended, FALSE otherwise
  */
gboolean g_web_suspended (GWeb *self);

/** Resume settings, state, etc, switch on times.
  * @param self Object pointer
  */
void g_web_resume (GWeb *self);

/** Sets a custom extension for the user agent string.
  * The user agent string is composed of the internal ua string with the extension appended at the end.
  * @param self Object pointer
  * @param ua_string The new ua string extension or NULL to remove the current extension.
  */
void g_web_set_ua_string (GWeb *self, gchar *ua_string);

/** Configures engine to identify itself as a specific browser type
  * @param self Object pointer
  * @param ua_id browser UserAgent ID.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_user_agent_id (GWeb *self, gint ua_id);

/** Get the UserAgent ID. 
  * @param self Object pointer
  * @return UserAgent ID.
  */
gint g_web_get_user_agent_id (GWeb *self);

/** Enable or disable URL loading.
  * @param self Object pointer
  * @param enable Enable flag.
  */
void g_web_enable_url_loading (GWeb *self, gboolean enable);

/** Free Memory
  * @param self Object pointer
  * @param progressive free.
  */
void g_web_free_memory (GWeb *self, gboolean progressive);


/** Store the state of all transfers.
  * The transfers are typically stored just before closing down the browser so 
  * that the transfers can be restored and continued the next time the browser is started. 
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_store_transfers (GWeb *self);

/** Restore transfers that where previously stored by storeTransfers().
  * The transfers should be restored at startup before any transfers are initiated. 
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_restore_transfers (GWeb *self);

/** Delete All cookies.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_delete_all_cookies (GWeb *self);

/** Confirm a cookie request.
  * @param self Object pointer
  * @param context The context for which the cookie was requested
  * @param accept The cookie (accept was pressed)
  * @param action The action that was chosen
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_confirm_cookie (GWeb *self, gpointer context, gboolean accept, GWebCookieAction action);

/** Get the global instance.
  * @param self Object pointer
  * @return Pointer to Web Global 
  */
GObject* g_web_get_global (GWeb *self);

/** Gets an enumeration value for the dialog message for certificate dialog.
  * @param self Object pointer
  * @param context The certificate context.
  *
  * @return The certificate message, which is one of the following:
  * G_WEB_CERT_MESS_UNTRUSTED         - Untrusted certificate
  * G_WEB_CERT_MESS_WRONGNAME         - Invalid or mismatched name
  * G_WEB_CERT_MESS_EXPIRED           - Certificate expired
  * G_WEB_CERT_MESS_PREVALID          - Certificate not valid yet
  * G_WEB_CERT_MESS_ALGORITHM         - Wrong certificate algorithm
  * G_WEB_CERT_MESS_VERSION           - Wrong certificate version
  * G_WEB_CERT_MESS_UNKNOWN           - Unknown error
  * G_WEB_CERT_MESS_VALID             - Valid certificate
  */
GWebCertificateMessage g_web_get_certificate_enum_message (GWeb *self, GObject *context);

/** Performs a certificate action as a result of a button press in a certificates dialog.
  *
  * For every action the caller must perform the reaction specified by the return value.
  *
  * @param self Object pointer
  * @param context  The certificate context.
  * @param action   The action to perform.
  * @param cert     The currently selected certificate, or NULL to let Engine decide.
  *
  * @return A reaction to be performed by on the dialog by the caller.
  *         A negative value indicates one of the following errors:
  *         - @c G_WEB_ERR               - Internal error occured.
  *         - @c G_WEB_ERR_NO_MEMORY     - Out of memory while performing the action.
  *         - @c G_WEB_ERR_NULL_POINTER  - Got a @c NULL context
  *         - @c G_WEB_ERR_OUT_OF_RANGE  - non exisiting action
  *         - @c G_WEB_ERR_NOT_SUPPORTED - The action is not supported for this certificate context.
  */
GWebCertificateReaction g_web_perform_crtificate_context_action (GWeb *self, GObject* context, GWebCertificateAction action, GObject *cert);

/** Gets an iterator with the certificates handled by this certificate context.
  *
  * @param self Object pointer
  * @param context   The certificate context.
  * @param iterator  The iterator.
  *
  * @return @c G_WEB_STATUS_OK on success.
  *         A negative value indicates one of the following errors:
  *         - @c G_WEB_ERR_NULL_POINTER  - Got a @c NULL argument.
  */
GWebStatus g_web_get_certificate_context_iter (GWeb *self, GObject* context, GObject **iterator);

/** Gets the certificate at the current position.
  *
  * @param self Object pointer
  * @param iterator The iterator.
  *
  * @return The certificate, or @c NULL if the certificate has been deleted or if the iterator is empty.
  */
GObject* g_web_get_certificate_iter (GWeb *self, GObject** iterator);

/** Free iterator object.
  *
  * @param self Object pointer
  * @param iterator The iterator.
  *
  */
void g_web_certificate_iter_free (GWeb *self, GObject* iterator);

/** Free certificate object.
  *
  * @param self Object pointer
  * @param cert Certificate object.
  *
  */
void g_web_certificate_free (GWeb *self, GObject* cert);

/** Sends a event message to all plugins currently open
  * Using the NPP_HandleEvent signal
  * @param self Object pointer
  * @param event The event to be sent to the plugin
  */
void g_web_notify_plugins(GWeb *self, gint event);


/** Get the number of stored username/password sets
  * stored for a forms element or all forms elements at a server.
  *
  * @param self Object pointer
  * @param info The struct that has been emitted from the engine.
  *
  * @return The number of matches, or -1 if an error has occurred.
  */
gint g_web_username_matchinfo_get_matches(GWeb *self, GObject *info);

/** Delete the n:th item from the password storage.
 * 
 * @param self Object pointer
 * @param info The pasword storage struct info.
 * @param index The index
 */
void g_web_username_matchinfo_delete(GWeb *self, GObject *info, gint index);


/** Get the username of the n:th match.
  *
  * @param self Object pointer
  * @param info The pasword storage struct info.
  * @param index The index.
  * @param username Return argument for the username. The string must be freed using g_free().
  *
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_username_matchinfo_get_username(GWeb *self, GObject* info, gint index, gchar** username);

/** Create the password manager to get the personal data 
  * that is store for the server.
  * 
  * @param self Object pointer
  * @param server The server.
  * @param index The index.
  * 
  * @return Web password manager object
  */
GWebPasswordManager* g_web_get_password_manager (GWeb *self, const gchar* server, gint index);

/** Store personal data. Should be used for authentication dialogs,
  * the forms data is saved via the on-submit signal return value.
  * 
  * @param self Object pointer
  * @param server The server.
  * @param username The username.
  * @param password The password.
  *
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_store_personal_data(GWeb *self, const gchar* server, const gchar* username, const gchar* password);

/** Delete personal data saved for an authentication dialog.
  * 
  * @param self Object pointer
  * @param server The server.
  * @param username The username.
  *
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_delete_personal_data(GWeb *self, const gchar* server, const gchar* username);

/** Clear all passwords.
  * 
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_clear_passwords (GWeb *self);

/** Clear all authenticated sessions.
  *
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_clear_authenticated_sessions (GWeb *self);


/** Empty the personal data
  *  
  * @param self Object pointer
  */
void g_web_clear_personal_data (GWeb *self);

/** Clear all global history.
  *  
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_clear_all_history (GWeb *self);

/** Enable/Disable to remember password for authentication dialog
  * 
  * @param self Object pointer
  * @param rpassword TRUE to remember password, FALSE, if not
  *  
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_remember_password (GWeb *self, gboolean rpassword);

/** Returns whether remember password is on or off for authentication dialog.
  * 
  * @param self Object pointer
  * 
  * @return TRUE, if remember password is on. FALSE, if not
  */
gboolean g_web_get_remember_password (GWeb *self);

/** Enable/Disable to remember password for forms.
  * 
  * @param self Object pointer
  * @param rpassword TRUE to remember password, FALSE, if not
  * 
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_remember_password_for_forms (GWeb *self, gboolean rpassword);

/** Returns whether remember password is on or off for forms.
  *  
  * @param self Object pointer
  * 
  * @return TRUE, if remember password is on. FALSE, if not
  */
gboolean g_web_get_remember_password_for_forms (GWeb *self);

/** Enable/Disable the plugin at runtime.
  * @param self Object pointer
  * @param name The name of the plugin, Ex:libdefaultplugin.so
  * @param enabled TRUE to enable the plugin, FALSE otherwise
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_plugin_enabled (GWeb *self, gchar* name, gboolean enabled);

GWebStatus g_web_set_oom_cb (GWeb *self, GFunc func);

/** Get whether plugin is enabled or not.
  * @param self Object pointer
  * @param name The name of the plugin, Ex:libdefaultplugin.so
  * @return TRUE, if the plugin is enabled. FALSE, FALSE otherwise.
  */
gboolean g_web_get_plugin_enabled (GWeb *self, gchar* name);

/** Set the plugin ignore file path.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_set_plugin_ignore_filepath (GWeb *self);

/** Sends a event message to web
  * @param self Object pointer
  * @param service_name name of Web service
  * @param context Some context pointer
  * @param topic Topic name
  * @param data Some gunichar data
  */
void g_web_notify_service(GWeb *self, const gchar *service_name, gpointer *context, const gchar *topic, gchar *data);

/**
  * Global Signals.
  */

/**
  * "modal-dialog" signal.
  */
#define G_WEB_SIGNAL_MODAL_DIALOG "modal-dialog"

/** Callback that Engine uses to request a modal dialog with a message
  * Signal name: "modal-dialog"
  * @param global The global browser instance from where the signal is emitted
  * @param caption The caption for the dialog
  * @param message The message in the dialog
  * @param type The type of dialog
  * @param button1 First button
  * @param button2 Second button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @param button3 Third button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @return The implementation if this function should return the button that was clicked as a reply
  */
GWebDialogButton g_web_signal_modal_dialog(GObject *global,
                                                                               const gchar* caption,
																			   const gchar* message,
																			   GWebDialogMessageType type,
																			   GWebDialogButton button1,
																			   GWebDialogButton button2,
																			   GWebDialogButton button3);

/**
  * "ask-cookie" signal.
  */
#define G_WEB_SIGNAL_ASK_COOKIE "ask-cookie"

/** Request a cookie dialog
  * Signal name: "ask-cookie"
  * @param global The global browser instance
  * @param context The cookie context. Return this context with a call to g_web_confirm_cookie
  * @param actions Integer with GCookieAction flags. Tells which actions are allowed on this cookie
  * @param url The cookie url
  * @param name The cookie name
  * @param value The cookie value
  * @param domain The cookie domain
  * @param path The cookie path
  * @param illegal_path TRUE if the cookie has an illegal path
  * @param expire The expiry date of the cookie in seconds since the epoch
  */
void g_web_signal_ask_cookie(GObject *global,
                                                  gpointer context,
												  gint actions,
												  const gchar* url,
												  const gchar* name,
												  const gchar* value,
												  const gchar* domain,
												  const gchar* path,
												  gboolean illegal_path);

/**
  * "certificate-dialog" signal.
  */
#define G_WEB_SIGNAL_CERTIFICATE_DIALOG "certificate-dialog"

/** Callback that Engine uses to request a certificate dialog.
  * Signal name: "certificate-dialog"
  * @param global   The global browser instance from where the signal is emitted.
  * @param ctx      The certificate context describing the dialog.
  * @return TRUE on success, FALSE on error.
  */
gboolean g_web_signal_certificate_dialog (GObject *global, gpointer ctx);

/**
  * "certificate-password-dialog" signal.
  */
#define G_WEB_SIGNAL_CERTIFICATE_PASSWORD_DIALOG "certificate-password-dialog"

/** Request a certificate password dialog.
  * Signal name: "certificate-password-dialog"
  * @param global   The global browser instance
  * @param title    The password dialog title
  * @param text     The password dialog text
  * @param password Pointer to the password string. The implementor must allocate the space. It will be freed by Engine. 
  */
void g_web_signal_certificate_password_dialog (GObject *global,
                                                                              const gchar* title,
																			  const gchar* text,
																			  gchar** password);
																			  
/**
  * "certificate-details" signal.
  */
#define G_WEB_SIGNAL_CERTIFICATE_DETAILS "certificate-details"

/** Certificate details were offered
  * Signal name: "certificate-details"
  *
  * @param engine The global browser instance
  * @param details. A X509 certificate
  */ 
void g_web_signal_certificate_details (GObject *global, gpointer details);

/**
  * "on-submit" signal.
  */
#define G_WEB_SIGNAL_ON_SELECT "on-submit"

/** Request for a usernames choice dialog, as a page was submitted.
  * Signal name: "on-submit"
  *
  * @param global  The global browser instance
  * @return  The GWebPasswordAction chosen by the user.
  */
GWebPasswordAction
g_web_signal_on_submit_cb (GObject *global);

/**
  * "select-match" signal.
  */
#define G_WEB_SIGNAL_SELECT_MATCH "select-match"

/** Callback that Engine uses when multiple username/password entries
  * exists when the password storage functionality is activated.
  * Signal name: "select-match"
  *
  * @param global The global browser instance
  * @param info Containing the different usernames.
  *
  * @return  The selected username index
  */
gint
g_web_signal_select_match (GObject *global,  gpointer info);

// WEB BUS MESSAGE DECLARATIONS

enum {
    UPDATE_AREA_MSG,
    SEND_EVENT_MSG,
    SHM_INFO_MSG,
    ELEMENT_INFO_MSG,
    PAINT_NOTIFY_MSG,
    VISIBLE_AREA_MSG,
    CURSOR_UPDATE_MSG,
    SEND_KEY_EVENT_MSG,
    INTERNAL_MESSAGES = 1024,
    PING_MSG,
    PONG_MSG,
    NULL_MSG
};

/** Web bus message to request for view update
  */
typedef struct {
    guint32 engine_id;
    int id;
    int left;
    int top;
    int width;
    int height;
    int zoom;
} UpdateAreaMessage;

/** Web bus message to request for sending events
  */
typedef struct {
    guint32 engine_id;
    int type;
    int x;
    int y;
    int mod;
    int button;
    int clickcount;
    int time;
} SendEventMessage;

/** Web bus message to request for sending key events
  */
typedef struct {
    guint32 engine_id;
    int type;
    int keyval;
    int state;
    int hardware_keycode;
    int group;
    int time;
} SendKeyEventMessage;

/** Web bus message to request for shm-info event
  */
typedef struct {
    guint32 engine_id;
    int id;
    int width;
    int height;
    int depth;
} ShmInfoMessage;

/** ElementInfoMessage message to request for element-info event
  */
typedef enum {
    G_WEB_SCROLLABLE_NONE  = 0x0000,
    G_WEB_SCROLLABLE_UP    = 0x0001,
    G_WEB_SCROLLABLE_DOWN  = 0x0002,
    G_WEB_SCROLLABLE_LEFT  = 0x0004,
    G_WEB_SCROLLABLE_RIGHT = 0x0008,
    G_WEB_SCROLLABLE_ANY   = 0x0010,
    G_WEB_SCROLLABLE_HOR_HIDDEN  = 0x0020,
    G_WEB_SCROLLABLE_VERT_HIDDEN = 0x0040,
} GWebScrollableType;

typedef enum {
    G_WEB_ELEMENT_INFO_NOID,
    G_WEB_ELEMENT_INFO_ID_MOUSE_OVER,
    G_WEB_ELEMENT_INFO_ID_DOM_FOCUS,
    G_WEB_ELEMENT_INFO_ID_UI_REQUEST,
    G_WEB_ELEMENT_INFO_ID_SCROLLVIEW,
    G_WEB_ELEMENT_INFO_ID_FIXED_FRAMES,
} GWebElementInfoType;

typedef struct {
    guint32 engine_id;
    int requestID;
    int el_type; // See GWebEnginePopupMenuType.
    int el_scroll_frame_type; // See GWebScrollableType
    int x;
    int y;
    int width;
    int height;
    int error;
} ElementInfoMessage;

/** Web bus message for paint_notify signal 
  */
typedef struct {
    guint32 engine_id;
    int requestId;
    int l;
    int t;
    int w;
    int h;
    int docwidth;
    int docheight;
    int scrollX;
    int scrollY;
    int zoom;
} PaintNotifyMessage;

/** Web bus message for visible/offscreen area signal
  */
typedef struct {
    guint32 engine_id;
    int vx; /**< Visible area rectange */
    int vy;
    int vw;
    int vh;
    int tx; /**< Rendered area rectange */
    int ty;
    int tw;
    int th;
    int wx; /**< Widget/window area rectange */
    int wy;
    int ww;
    int wh;
} VisibleAreaMessage;

/** Supported interaction mode cursor types
  */
typedef enum {
    G_WEB_CURSOR_NONE,      // must be first
    G_WEB_CURSOR_DEFAULT,
    G_WEB_CURSOR_LINK,
    G_WEB_CURSOR_TEXT,
    G_WEB_CURSOR_GRAB,
    G_WEB_CURSOR_GRABBING,
    G_WEB_NUM_OF_CURSORS    // must be last
} GWebCursorType;

/** Web bus message for interaction mode cursor updates
  */
typedef struct {
    guint32 engine_id;
    GWebCursorType cursor_type;
} CursorUpdateMessage;

G_END_DECLS

#endif /* __GWEB_H__ */
