/**
 * @file ui_folder.c GUI folder handling
 * 
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <gtk/gtk.h>
#include "support.h"
#include "interface.h"
#include "callbacks.h"
#include "conf.h"
#include "folder.h"
#include "ui_feedlist.h"
#include "ui_folder.h"
#include "ui_mainwindow.h"

#include "debug_new.h"
#include <hildon/hildon-banner.h>
#include <osso-log.h>

static GtkWidget *foldernamedialog = NULL;
extern AppData *app_data;

gint ui_get_marked_from_iter(GtkTreeIter * iter);
void ui_folder_empty_check(folderPtr folder);
void ui_folder_remove(gpointer callback_data, guint callback_action,
                      GtkWidget * widget);


/** Looks up a folder id from the folder tree
  *
  * @param from the iter to start the search from
  * @param id of the folder to look for
  * @return pointer to the folder
  */
folderPtr ui_folder_lookup_by_id(GtkTreeIter *from, gchar *id)
{
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL; 
    GtkTreeModel *model = NULL;
    folderPtr found_folder = NULL;

    g_assert(NULL != feedstore);

    model = GTK_TREE_MODEL(feedstore);

    if (from != NULL) 
    {
        gtk_tree_model_get(model, from, FS_PTR, &node, -1);
        valid = gtk_tree_model_iter_children(model, &iter, from);

    } 
    else 
    {
        valid = gtk_tree_model_get_iter_first(model, &iter);
    }

    while (valid) 
    {
        gtk_tree_model_get(model, &iter, FS_PTR, &node, -1);

        if (NULL != node)
        {
            if (FST_FOLDER == feed_get_type((feedPtr) node))
            {
                gchar *current_id = NULL;

                current_id = folder_get_id((folderPtr) node);

                if(current_id != NULL && id != NULL)
                {
                    if(!strcmp(current_id, id))
                        return (folderPtr) node;
                }
                found_folder = ui_folder_lookup_by_id(&iter, id);

                if(found_folder)
                    break;
            }
        }
        valid = gtk_tree_model_iter_next(model, &iter);
    }
    return found_folder;
}


/** Checks if a folder exists in the feed tree
  *
  * @param from the iter to start the search from
  * @param candidate the folder name to look for
  * @return TRUE if found
  * tvh: made this non-recursive
  */
gboolean ui_folder_check_if_exists(GtkTreeIter * from, gchar * candidate)
{
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter root;
    g_assert(NULL != feedstore);

    model = GTK_TREE_MODEL(feedstore);

    if (from != NULL) {
        gtk_tree_model_get(model, from, FS_PTR, &node, -1);
        valid = gtk_tree_model_iter_children(model, &iter, from);

    } else {
        valid = gtk_tree_model_get_iter_first(model, &root);
        /*get first child of root */
        valid = gtk_tree_model_iter_children(model, &iter, &root);
    }

    while (valid) {
        gtk_tree_model_get(model, &iter, FS_PTR, &node, -1);

        if (NULL != node) {
            if (FST_FOLDER == feed_get_type((feedPtr) node)) {
                gchar *existing_title = NULL;

                existing_title = folder_get_title((folderPtr) node);
                if (existing_title != NULL && candidate != NULL) {
                    if (!strcmp(existing_title, candidate))
                        return TRUE;
                }
            }
        }
        valid = gtk_tree_model_iter_next(model, &iter);
    }
    return FALSE;
}

/** Selects the appropriate icon for a folder depending on
  * the expanded state of the folder.
  *
  * @param np the folder the icon is needed for
  * @return the icon
  */
GdkPixbuf *ui_folder_select_icon(gboolean expanded)
{
    return icons[expanded ? ICON_FOLDER_OPEN_WITH_EMBLEM : ICON_FOLDER_CLOSED_WITH_EMBLEM];
}

/** Updates a folder
  *
  * @param iter iterator to the folder to update
  */
static void ui_folder_update_from_iter(GtkTreeIter * iter)
{
    gboolean rc = FALSE;
    GtkTreeIter child;
    gint unread = 0, count = 0, marked = 0;
    gchar *title = NULL, *label = NULL;
    GtkTreeModel *model = NULL;
    folderPtr ptr = NULL;
    nodePtr childptr = NULL;

    model = GTK_TREE_MODEL(feedstore);

    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);

    g_assert(FST_FOLDER == ptr->type);
    g_assert(ptr != NULL);
    g_assert(ptr->ui_data);

    title = g_markup_escape_text(folder_get_title(ptr), -1);

    unread = 0;
    marked = ui_get_marked_from_iter(iter);
    rc = gtk_tree_model_iter_children(model, &child, iter);

    while (rc) {
        gtk_tree_model_get(model, &child, FS_UNREAD, &count,
                           FS_PTR, &childptr, -1);
        unread += count;
        rc = gtk_tree_model_iter_next(model, &child);
    }
    ptr->markCount = marked;

    if (unread > 0 && marked > 0)
        label =
            g_strdup_printf
            ("<span weight=\"bold\" foreground=\"%s\">%s (%d)</span>",
             "red", title, unread);
    else if (unread > 0)
        label =
            g_strdup_printf
            ("<span weight=\"bold\" foreground=\"%s\">%s (%d)</span>",
             "black", title, unread);

    else if (marked > 0)
        label =
            g_strdup_printf
            ("<span foreground=\"%s\">%s</span>",
             "red", title);
    else
        label = g_strdup_printf("<span foreground=\"%s\">%s</span>",
                                "black",
                                title);

    gtk_tree_store_set(feedstore, iter,
                       FS_LABEL, label,
                       FS_ICON, ui_folder_select_icon(ptr->expanded),
                       FS_UNREAD, unread, -1);

    g_free(title);
    g_free(label);
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/



/*------------------------------------------------------------------------------*/
/* new/change/remove folder dialog callbacks                                    */
/*------------------------------------------------------------------------------*/

void on_newfolderbtn_clicked(GtkButton * button, gpointer user_data)
{
    GtkWidget *foldertitleentry = NULL;
    gchar *foldertitle = NULL;
    folderPtr folder = NULL;
    folderPtr folder_tmp = NULL;
    folderPtr parent = NULL;
    gboolean search_result = FALSE;
    GtkTreePath *path = NULL;
    GtkWidget *newfolderdialog = (GtkWidget *) user_data;
    gpointer treeview = NULL;
    gpointer treemodel = NULL;

    g_assert(newfolderdialog != NULL);

    foldertitleentry = lookup_widget(newfolderdialog, "foldertitleentry");
    foldertitle = (gchar *) gtk_entry_get_text(GTK_ENTRY(foldertitleentry));

    if (foldertitle == NULL)
        return;

    /* FIXME: gtk_entry_get_text returns CONST string and asks not to modify it... */
    foldertitle = g_strstrip(foldertitle);

    folder_tmp = g_object_get_data(G_OBJECT(newfolderdialog),
                                   "parent_folder");
    parent = (folder_tmp != NULL)
        ? folder_tmp : ui_feedlist_get_root_folder();

    GtkTreeIter *parent_iter = &((ui_data *) (parent->ui_data))->row;
    if (ui_folder_check_if_exists(parent_iter, foldertitle))
    {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext
                                       (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                        "ckdg_ib_folder_already_exists"));
        return;
    }

    if (strcmp(foldertitle, "")) {
        folder = restore_folder(NULL, foldertitle, NULL, FST_FOLDER);

        if (folder) {
            /* add the new folder to the model */
            ui_feedlist_add(parent, (nodePtr) folder, -1);
            ui_feedlist_select((nodePtr) folder);
            ui_feedlist_update();
            treeview = g_object_get_data(G_OBJECT(newfolderdialog),
                                         "folder_treeview");
            treemodel = g_object_get_data(G_OBJECT(newfolderdialog),
                                          "folder_treemodel");

            if (treeview != NULL && treemodel != NULL) {
                search_result =
                    ui_feedlist_search_model_by_ptr
                    (GTK_TREE_MODEL(treemodel), (nodePtr) folder, &path,
                     NULL);

                if (search_result && path != NULL) {
                    gtk_tree_view_expand_to_path(treeview, path);
                    gtk_tree_view_set_cursor
                        (GTK_TREE_VIEW(treeview), path, NULL, FALSE);
                } else {
                    g_warning("could not find the new folder "
                              "in the model");
                }
                gtk_tree_path_free(path);
                path = NULL;
            }
            gtk_widget_destroy(newfolderdialog);
        } else {
            g_warning("internal error! could not get a new folder key!");
        }
    } else {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext
                                       (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                        "ckdg_ib_enter_name"));
        gtk_entry_set_text(GTK_ENTRY(foldertitleentry),
                           dgettext(HILDON_FM_L10N_PACKAGE,
                                    "ckdg_va_new_folder_name_stub"));
        gtk_editable_select_region(GTK_EDITABLE(foldertitleentry), 0, 1000);
    }
}

void ui_folder_close_rename_dialog(void)
{
    if (NULL == foldernamedialog || !G_IS_OBJECT(foldernamedialog))
        return;
    if (GTK_WIDGET_VISIBLE(foldernamedialog)) {
        gtk_widget_hide(foldernamedialog);
    }
}

void create_folder_rename_dialog(folderPtr folder)
{
    GtkWidget *foldernameentry;
    gchar *title = NULL;

    if ((NULL == folder) || (FST_FOLDER != folder->type)) {
        return;
    }

    if (NULL == foldernamedialog || !G_IS_OBJECT(foldernamedialog))
        foldernamedialog = create_foldernamedialog();

    foldernameentry = lookup_widget(foldernamedialog, "foldernameentry");
    title = folder_get_title(folder);
    gtk_entry_set_text(GTK_ENTRY(foldernameentry), title);
    gtk_object_set_data(GTK_OBJECT(foldernamedialog), "folder", folder);

    gtk_widget_show_all(foldernamedialog);
}

void on_foldernamechangebtn_clicked(GtkButton * button, gpointer user_data)
{
    gchar *tmp= NULL;
    gchar *title= NULL;
    folderPtr folder = NULL;
    GtkWidget *foldernameentry;
    GtkTreeIter parent_iter;

    folder = gtk_object_get_data(GTK_OBJECT(foldernamedialog), "folder");
    foldernameentry = lookup_widget(foldernamedialog, "foldernameentry");
    tmp = (gchar *) gtk_entry_get_text(GTK_ENTRY(foldernameentry));

    /* strip leading and trailing whitespaces tmp dup'ed since its pointer to
     * widgets internal memory */  

    title = g_strdup(tmp);    
    title = g_strstrip(title);

    if (!strcmp(title, "")) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext
                                       (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                        "ckdg_ib_enter_name"));
        g_free(title);
        return;
    }
    if (strcmp(title, folder->title) == 0) {
        /*TODO: just hide ? */
        gtk_widget_hide(foldernamedialog);
        g_free(title);
        return;
    }
    /* get the parent iter of this folder */
    GtkTreeIter *iter = &((ui_data *) (folder->ui_data))->row;
    GtkTreePath *path =
        gtk_tree_model_get_path(GTK_TREE_MODEL(feedstore), iter);
    if (TRUE == gtk_tree_path_up(path)) {
        gtk_tree_model_get_iter(GTK_TREE_MODEL(feedstore),
                                &parent_iter, path);
    }
    gtk_tree_path_free(path);
    if (!ui_folder_check_if_exists(&parent_iter, title))
    {
        folder_set_title(folder, title);
        ui_feedlist_update();
        gtk_widget_hide(foldernamedialog);
    } else
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext
                                       (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                        "ckdg_ib_folder_already_exists"));
    g_free(title);
}

void ui_folder_remove(gpointer callback_data, guint callback_action,
                      GtkWidget * widget)
{
    folderPtr folder = (folderPtr) callback_data;

    g_assert((NULL != folder) && (FST_FOLDER == folder->type));
    if (folder == ui_feedlist_get_root_folder()) {
        ui_feedlist_reset_root_folder();
    }
    folder_free(folder);
}

/*
 * Expansion & Collapsing
 */

gboolean ui_is_folder_expanded(folderPtr folder)
{
    GtkTreeIter *iter = NULL;
    GtkTreePath *path = NULL;
    GtkWidget *treeview = NULL;
    GtkTreeIter visible_iter;
    gboolean expanded = FALSE;

    g_assert(NULL != feedstore);

    g_assert(folder != NULL);

    if (folder->ui_data == NULL)
        return FALSE;

    /* Root folder caused warnings with this function
     * so we return. */
    if (folder == ui_feedlist_get_root_folder()) {
        return TRUE;
    }

    if (NULL != (treeview = feedlist)) {
        iter = &((ui_data *) (folder->ui_data))->row;
        ui_feedlist_convert_store_iter_to_visible_iter(&visible_iter, iter);
        path = gtk_tree_model_get_path(feedmodel, &visible_iter);
        expanded = gtk_tree_view_row_expanded(GTK_TREE_VIEW(treeview), path);
        gtk_tree_path_free(path);
    }
    return expanded;
}

void ui_folder_set_expansion(folderPtr folder, gboolean expanded)
{
    GtkTreeIter *iter = NULL;
    GtkTreePath *path = NULL;
    GtkWidget *treeview = NULL;
    GtkTreeIter visible_iter;
    g_assert(NULL != feedstore);

    g_assert(folder != NULL);

    folder->expanded = expanded;
    if (folder->ui_data == NULL) {
        return;
    }

    /* Root folder caused warnings with this function
     * so we return. */
    if (folder == ui_feedlist_get_root_folder()) {
        return;
    }
    if (NULL != (treeview = feedlist)) {
        iter = &((ui_data *) ((nodePtr) folder)->ui_data)->row;
        ui_feedlist_convert_store_iter_to_visible_iter(&visible_iter, iter);
        path = gtk_tree_model_get_path(feedmodel, &visible_iter);

        if (expanded) {
            gtk_tree_view_expand_row(GTK_TREE_VIEW(treeview), path, FALSE);
        } else {
            gtk_tree_view_collapse_row(GTK_TREE_VIEW(treeview), path);
        }
        gtk_tree_path_free(path);
    }
}

/* Subfolders */

/* this function is a workaround to the cant-drop-rows-into-emtpy-
 * folders-problem, so we simply pack an (empty) entry into each
 * empty folder like Nautilus does... */

void ui_folder_empty_check(folderPtr folder)
{
    GtkTreeIter *parent = &((ui_data *) (folder->ui_data))->row;
    GtkTreeIter iter;
    int count = 0;
    gboolean valid = FALSE;
    nodePtr ptr = NULL;
    /* this function does two things:
     * 
     * 1. add "(empty)" entry to an empty folder
     * 2. remove an "(empty)" entry from a non empty folder
     * (this state is possible after a drag&drop action) */

    /* key is folder keyprefix, value is folder tree iterator */
    count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(feedstore), parent);

    /* case 1 */
    if (0 == count) {
        gchar *empty_folder_label = NULL;

        if (folder == ui_feedlist_get_root_folder())
            empty_folder_label = g_strdup(_("rss_ia_no_subscribed_feeds"));
        else
            empty_folder_label = g_strdup(_("rss_ia_no_feeds"));

        gtk_tree_store_append(feedstore, &iter, parent);
        gtk_tree_store_set(feedstore, &iter,
                           FS_LABEL, empty_folder_label,
                           FS_ICON, icons[ICON_RSS_NEWS_FEED],
                           FS_PTR, NULL, FS_UNREAD, 0, -1);

        g_free(empty_folder_label);

        return;
    }

    if (1 == count)
        return;

    /* else we could have case 2 */
    valid = gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore), &iter, parent);
    while (valid) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &iter, FS_PTR, &ptr,
                           -1);

        if (ptr == NULL) {
            gtk_tree_store_remove(feedstore, &iter);
            return;
        }

        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &iter);
    };
}

void ui_folder_handle_empty_root()
{
    int count = 0;
    GtkTreeModel *model = NULL;
    folderPtr folder = NULL;
    GtkTreeIter *parent = NULL;
    GtkTreeIter iter;

    g_assert(feedstore != NULL);

    model = GTK_TREE_MODEL(feedstore);

    folder = ui_feedlist_get_root_folder();

    if (folder == NULL)
        return;

    parent = &((ui_data *) (folder->ui_data))->row;

    count = gtk_tree_model_iter_n_children(model, parent);

    if (count == 0) {
        gchar *empty_folder_label = NULL;

        empty_folder_label = g_strdup(_("rss_ia_no_subscribed_feeds"));

        gtk_tree_store_append(feedstore, &iter, parent);
        gtk_tree_store_set(feedstore, &iter,
                           FS_LABEL, empty_folder_label,
                           FS_ICON, icons[ICON_RSS_NEWS_FEED],
                           FS_PTR, NULL, FS_UNREAD, 0, -1);

        g_free(empty_folder_label);
    }
}

void ui_folder_check_if_empty(void)
{
    ui_feedlist_do_for_all(NULL,
                           ACTION_FILTER_FOLDER | ACTION_FILTER_ROOT,
                           ui_folder_empty_check);
}

void ui_folder_remove_node(nodePtr ptr)
{
    GtkTreeIter iter;
    gboolean parentExpanded = FALSE;
    folderPtr parent = NULL;

    g_assert(NULL != ptr);

    if (ptr->ui_data == NULL)
        return;

    iter = ((ui_data *) (ptr->ui_data))->row;
    parent = ui_feedlist_get_parent(ptr);

    /* If the folder becomes empty, the folder would collapse */
    if (parent != NULL)
        parentExpanded = parent->expanded;

    gtk_tree_store_remove(feedstore, &iter);

    g_free((ui_data *) (ptr->ui_data));
    ptr->ui_data = NULL;

    if (parent != NULL) {
        ui_folder_check_if_empty();
        if ((parent != NULL) && parentExpanded) {
            ui_folder_set_expansion(parent, TRUE);
        }
    }
}

gint ui_get_marked_from_iter(GtkTreeIter * iter)
{
    gboolean rc = FALSE;
    GtkTreeIter child;
    gint marked = 0;
    GtkTreeModel *model = NULL;
    folderPtr ptr = NULL;
    nodePtr childptr = NULL;
    model = GTK_TREE_MODEL(feedstore);
    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);
    g_assert(FST_FOLDER == ptr->type);
    g_assert(ptr != NULL);
    g_assert(ptr->ui_data);
    rc = gtk_tree_model_iter_children(model, &child, iter);
    while (rc) {
        gtk_tree_model_get(model, &child, FS_PTR, &childptr, -1);
        if (NULL != childptr && FST_FOLDER == childptr->type) {
            marked += ui_get_marked_from_iter(&child);
        } else if (NULL != childptr && FST_FEED == childptr->type) {
            marked += feed_get_mark_counter((feedPtr) childptr);
        }
        rc = gtk_tree_model_iter_next(model, &child);
    }
    return marked;
}

void ui_folder_collapse_expand_cb(GtkTreeView * widget,
                                  GtkTreeIter * iter,
                                  GtkTreePath * path, gpointer user_data)
{
    GtkTreeModel *model;
    GtkTreeIter store_iter;
    folderPtr ptr = NULL;
    nodePtr childptr = NULL;
    GtkTreeIter child;
    gboolean rc;

    if (!lifereaStarted)
        return;

    if(app_data->app_ui_data->search_mode == SFM_SEARCH)
    return;
    
    g_assert(NULL != iter);

    model = GTK_TREE_MODEL(feedstore);
    ui_feedlist_convert_visible_iter_to_store_iter(&store_iter, iter);
    gtk_tree_model_get(GTK_TREE_MODEL(model), &store_iter, FS_PTR, &ptr, -1);

    g_assert(NULL != ptr);

    if (displayed_node != NULL)
        if (ui_feedlist_is_ancestor((nodePtr) displayed_node, &store_iter)) {
            ui_htmlview_clear(ui_mainwindow_get_active_htmlview());
            ui_itemlist_clear();
            ui_mainwindow_finish();

            g_assert(app_data != NULL);
            g_assert(app_data->app_ui_data != NULL);

            gtk_window_set_title(GTK_WINDOW(app_data->app_ui_data->main_view),
                                 "");
        }

    ptr->expanded = ui_is_folder_expanded(ptr);

    gtk_tree_store_set(GTK_TREE_STORE(model), &store_iter,
                       FS_ICON, ui_folder_select_icon(ptr->expanded), -1);

    //if expanded. We should set all the sub-folders to collapsed
    if ((ptr->expanded == TRUE)) {
        rc = gtk_tree_model_iter_children(GTK_TREE_MODEL(model), &child,
                                          &((ui_data *) (ptr->ui_data))->row);
        while (rc) {
            gtk_tree_model_get(GTK_TREE_MODEL(model), &child, FS_PTR,
                               &childptr, -1);
            if (childptr != NULL)
                if (FST_FOLDER == childptr->type) {
                    ui_folder_set_expansion((folderPtr) childptr, FALSE);
                    gtk_tree_store_set(GTK_TREE_STORE(model), &child,
                                       FS_ICON, ui_folder_select_icon(FALSE),
                                       -1);
                }

            rc = gtk_tree_model_iter_next(model, &child);
        }
    }
}

void ui_folder_update(folderPtr folder)
{
    nodePtr node = (nodePtr) folder;

    if (NULL != node) {
        g_assert(FST_FOLDER == node->type);
        g_assert(NULL != node->ui_data);
        ui_folder_update_from_iter(&((ui_data *) (node->ui_data))->row);
    }
}

void ui_folder_mfdialog_collapse_expand_cb(GtkTreeView * widget,
                                           GtkTreeIter * iter,
                                           GtkTreePath * path,
                                           gpointer user_data)
{

    GtkTreeStore *model;
    folderPtr ptr = NULL;
    nodePtr childptr = NULL;
    GtkTreeIter child;
    gboolean rc;

    /* only works with the filtered model of the dialog */
    model = (GtkTreeStore *) gtk_tree_view_get_model(widget);

    gtk_tree_model_get(GTK_TREE_MODEL(model), iter, FS_PTR, &ptr, -1);

    g_assert(NULL != ptr);
    if (FST_FOLDER != ptr->type)
        return;

    ptr->dialog_mode_expanded = !ptr->dialog_mode_expanded;

    if ((ptr->dialog_mode_expanded == TRUE)) {
        rc = gtk_tree_model_iter_children(GTK_TREE_MODEL(model), &child,
                                          iter
                                          );
        while (rc) {
            gtk_tree_model_get(GTK_TREE_MODEL(model), &child, FS_PTR,
                               &childptr, -1);
            if (childptr != NULL)
                //show always be true in dialog model
                if (FST_FOLDER == childptr->type) {
                    ((folderPtr) childptr)->dialog_mode_expanded = FALSE;
                }
            rc = gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &child);
        }
    }
}

