/**
    @file state_save.h

    Copyright (c) 2004-2006 Nokia Corporation. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#ifndef STATE_SAVE_H
#define STATE_SAVE_H


#include "appdata.h"

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#define STATE_FILE_LENGTH 10000


#define STATE_KEY_FILE_GROUP "rss_feed_reader"
#define STATE_KEY_CURRENT_FEED "current_feed"
#define STATE_KEY_SCROLL_POSITION "scroll_position"
#define STATE_KEY_FULLSCREEN "fullscreen"
#define STATE_KEY_SHOWN_DIALOG "shown_dialog"
#define STATE_KEY_ADD_FEED_URL "add_feed_url"


#define STATE_MAX_PATH_DEPTH 15


typedef enum {
    ST_SUCCESS,
    ST_ERR_NO_SAVED_STATE,
    ST_ERR_OPEN,
    ST_ERR_TOO_LONG,
    ST_ERR_KEY_FILE,
    ST_ERR_KEY_FILE_PARSING,
    ST_ERR_WRITE,
    ST_ERR_READ
} StateSaveResultCode;


/* Currently only ADD_FEED is implemented, maybe this more will be added
 * later? */
enum {
    DIALOG_NONE = 0,
    DIALOG_ADD_FEED,
    DIALOG_MANAGE_FOLDERS,
    DIALOG_FEED_DETAILS,
    DIALOG_FEED_UNSUBSCRIBE,
    DIALOG_FEED_DIRECTORY
};

typedef struct {
    gchar **current_feed;
    gint current_feed_depth;
    gboolean fullscreen;
    gint vertical_scroll_window_position;
    gint shown_dialog;
    gchar *add_feed_url;
} AppState;


/**
   Initialize an application state object.

   @param app_state AppState object to be initialized
 */
void init_app_state(AppState * app_state);


/**
   Create an application state object.
   Object is allocated and initialized.

   @return allocated and initialized AppState object
 */
AppState *create_app_state(void);


/**
   Destroy an application state object.
   Field recent_list is not destroyed.

   @param app_state AppState object to be destroyed (not NULL)
 */
void destroy_app_state(AppState * app_state);


/**
   Construct application state object.

   @param app_state AppState object to be constructed (not NULL)
   @param app_ui_data AppUIData object (not NULL)
   @param fmt formatter state (not NULL)
 */
void construct_app_state(AppState * app_state);


/**
   Update application data structures with
   information from an application state object.

   @param app_state AppState object (not NULL)
 */
void update_app_state(const AppState * app_state);


/**
   Save application state.

   @param app_data AppData object
   @param app_state application state object to be saved
   @return result code, ST_SUCCESS on success
 */
StateSaveResultCode reader_save_state(AppData * app_data,
                                      AppState * app_state);


/**
   Save application state to a key file.

   @param app_state application state object (not NULL)
   @return key file object with saved data
 */
GKeyFile *save_app_state_to_key_file(AppState * app_state);


/**
   Do state saving with contents as a string.

   @param osso OSSO context object
   @param contents contents to be saved
   @return result code, ST_SUCCESS on success
 */
StateSaveResultCode do_save_state(osso_context_t * osso,
                                  const gchar * contents);


/**
   Load application state.

   @param app_data AppData object
   @param app_state application state object to be loaded
   @result result code, ST_SUCCESS on success
 */
StateSaveResultCode reader_load_state(AppData * app_data,
                                      AppState * app_state);


/**
   Load state file contents into a string.

   @param osso OSSO context object
   @param buffer character buffer with length
   NOTES_STATE_FILE_MAX_LENGTH (not NULL)
   @return result code, ST_SUCCESS on success
 */
StateSaveResultCode do_load_state(osso_context_t * osso, gchar * buffer);


/**
   Load application state from a key file.
   Application state has to be initialized with empty values.

   @param app_state application state object (not NULL)
   @param key_file key file (not NULL)
   @return TRUE if no errors, FALSE otherwise
 */
gboolean load_app_state_from_key_file(AppState * app_state,
                                      GKeyFile * key_file);



/**
   Reset the state file (no saved state).

   @param osso OSSO context
   @return result code, ST_SUCCESS on success
 */
StateSaveResultCode reset_state_file(osso_context_t * osso);


#endif
