/**
 * @file opml.c generic OPML 1.0 support
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <osso-log.h>
#include <hildon/hildon-note.h>
#include "feed_types.h"
#include "opml.h"
#include "htmlview.h"
#include "ui_feed_directory.h"

extern AppData *app_data;

static GSList *feed_list;              /* List of feeds in the current feed directory */

/* you can find the OPML specification at Userland:
 * 
 * http://www.opml.org/
 * 
 */

/* ---------------------------------------------------------------------------- */
/* OPML parsing and HTML output                         */
/* ---------------------------------------------------------------------------- */

/** Frees allocated memory
  */
static void feed_list_free()
{
    if (feed_list != NULL) {
	GSList *t = feed_list;
	for (; t; t = t->next) {
	    gchar *url = (gchar *) t->data;
	    if (url != NULL) g_free(url);
	}
	g_slist_free(feed_list);
    }
}

/** Process outline tags of type rss and collapse the hieararchical structure
  * into one level
  *
  * @param cur the current node
  */
static void process_outline_tags(xmlNodePtr cur)
{
    gchar *url = NULL;
    xmlNodePtr child = NULL;
    static int depth = 0;

    /* Do not iterate the opml tree infinitely */
    if (depth >= MAX_DEPTH)
        return;

    depth++;

    g_assert(cur != NULL);

    /* process all <outline> tags */
    child = cur->xmlChildrenNode;
    while (child != NULL) {
        /* If the outline is of type rss, save the necessary data */
        if (!xmlStrcmp(child->name, BAD_CAST "outline") &&
            ui_feed_directory_outline_type_is_rss(child)) {

            url = utf8_fix(xmlGetProp(child, BAD_CAST "xmlUrl"));

            if (url != NULL)
                feed_list = g_slist_append(feed_list, url);
        }
        /* Otherwise check if the outline has children */
        else if (!xmlStrcmp(child->name, BAD_CAST "outline") &&
                 child->xmlChildrenNode != NULL) {

            process_outline_tags(child);
        }
        child = child->next;
    }
    depth--;
}

/** Parses the feed directory contents provided in OPML format
  *
  * @param doc the document to parse
  * @param cur the current node
  */
static void opml_parse(feedPtr fp, xmlDocPtr doc, xmlNodePtr cur)
{
    int error = 0;

    g_assert(doc != NULL);
    g_assert(cur != NULL);

    feed_list = NULL;

    feed_set_feed_directory(fp, TRUE);

    ULOG_DEBUG("Parsing opml document");

    if (xmlStrcmp(cur->name, BAD_CAST "opml") &&
        xmlStrcmp(cur->name, BAD_CAST "oml") &&
        xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {
        ULOG_DEBUG
            ("OPML Could not find OPML header. Unable to parse feed directory.");
        xmlFreeDoc(doc);
        error = 1;
    }

    cur = cur->xmlChildrenNode;

    while (cur && xmlIsBlankNode(cur)) {
        cur = cur->next;
    }

    while (cur != NULL) {
        if (!xmlStrcmp(cur->name, BAD_CAST "body"))
            process_outline_tags(cur);
        cur = cur->next;
    }

    app_data->app_ui_data->feed_directory_count = g_slist_length(feed_list);

    feed_list_free();
}

static gboolean opml_format_check(xmlDocPtr doc, xmlNodePtr cur)
{
    g_assert(cur != NULL);

    if (!xmlStrcmp(cur->name, BAD_CAST "opml") ||
        !xmlStrcmp(cur->name, BAD_CAST "oml") ||
        !xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {

        return TRUE;
    }
    return FALSE;
}

/* ---------------------------------------------------------------------------- */
/* initialization                               */
/* ---------------------------------------------------------------------------- */

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

feedHandlerPtr opml_init_feed_handler(void)
{
    feedHandlerPtr fhp = NULL;

    fhp = g_new0(struct feedHandler, 1);

    /* prepare feed handler structure */
    fhp->typeStr = "opml";
    fhp->icon = ICON_RSS_NEWS_FEED;
    fhp->directory = FALSE;
    fhp->feedParser = opml_parse;
    fhp->checkFormat = opml_format_check;
    fhp->merge = FALSE;

    return fhp;
}
