/**
 * @file htmlview.h common interface for browser module implementations
 * and module loading functions
 *
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _HTMLVIEW_H
#define _HTMLVIEW_H

#include <glib.h>
#include <gmodule.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include "gtkhtml/gtkhtml_plugin.h"

struct displayset {
    gchar *headtable;
    gchar *head;
    gchar *body;
    gchar *foot;
    gchar *foottable;
};

/**
 * @{ Common HTML definitions
 *
 * Used styles should be defined in the
 * liferea.css style sheet! For some important HTML literals
 * like the item and feed description headers the styles 
 * are duplicated here just in case the style sheet is 
 * missing. 
 */

/*#define EMPTY     "<html><body></body></html>" */
#define EMPTY		""

/*#define HTML_NEWLINE  "<br>" */
#define HTML_NEWLINE	"\n"

/* RSS feed/item output definitions (some are used by OCS too!) */

#define HEAD_START		""
#define HEAD_LINE		"<span class=\"osso_rss_feed_reader_headline\">%s</span>"
#define HEAD_LINE_MARKED		"<span class=\"osso_rss_feed_reader_marked_headline\">%s</span>"
#define HEAD_END		""

#define FEED_FOOT_TABLE_START	""
#define FEED_FOOT_FIRSTTD	""
#define FEED_FOOT_NEXTTD	""
#define FEED_FOOT_LASTTD	""
#define FEED_FOOT_TABLE_END	""

#define ITEM_ACTIONS "\n<span class=\"osso_rss_feed_reader_item_actions\" itemid=\"%ld\"/>"
#define ITEM_END "<span class=\"osso_rss_feed_reader_item_boundary\"/>"

#define FEED_FOOT_WRITE(buffer, key, value)	if(NULL != value) { \
							addToHTMLBuffer(&(buffer), FEED_FOOT_FIRSTTD); \
							addToHTMLBuffer(&(buffer), (gchar *)key); \
							addToHTMLBuffer(&(buffer), FEED_FOOT_NEXTTD); \
							addToHTMLBuffer(&(buffer), (gchar *)value); \
							addToHTMLBuffer(&(buffer), FEED_FOOT_LASTTD); \
						}

#define	IMG_START	"<img class=\"feed\" src=\""
#define IMG_END		"\"><br>"

/* OCS direntry output definitions */

#define FORMAT_START	"<table cellspacing=\"0\" class=\"ocsformats\"><tr><td class=\"ocslink\">"
#define FORMAT_LINK	"<b>Format: </b>"
#define FORMAT_LANGUAGE		"</td></tr><tr><td class=\"ocslanguage\">Language: "
#define FORMAT_UPDATEPERIOD	"</td></tr><tr><td class=\"ocsinterval\">Update Period: "
#define FORMAT_UPDATEFREQUENCY	"</td></tr><tr><td class=\"ocsfrequency\">Update Frequency: "
#define FORMAT_CONTENTTYPE	"</td></tr><tr><td class=\"ocscontenttype\">Content Type: "
#define FORMAT_END	"</td></tr></table>"

/* condensed mode shading */

/* FIXME: check the correct color etc */
#define SHADED_START	"<span foreground=\"blue\">"
#define SHADED_END	"</span>"
#define UNSHADED_START	""
#define UNSHADED_END	""

/* HTTP and parsing error text */

#define UPDATE_ERROR_START	"<table cellspacing=\"0\" class=\"httperror\"><tr><td class=\"httperror\">"
#define HTTP_ERROR_TEXT		"The last update of this subscription failed!<br><b>HTTP error code %d: %s</b>"
#define PARSE_ERROR_TEXT	"There were errors while parsing this feed. The following error occured:<br><b>%s</b>"
#define HTTP410_ERROR_TEXT	"This feed is discontinued. It's no longer available. Liferea won't update it anymore but you can still access the cached headlines."
#define UPDATE_ERROR_END	"</td></tr></table>"

/*@} */

#define HTMLVIEW_API_VERSION 2



typedef struct htmlviewPluginInfo_ htmlviewPluginInfo;

struct htmlviewPluginInfo_ {
    unsigned int api_version;
    char *name;

    void (*init) (void);
    void (*deinit) (void);

    GtkWidget *(*create) ();
    void (*write) (GtkWidget * widget, gchar * string, gchar * base,
                   nodePtr node);
    void (*launch) (GtkWidget * widget, const gchar * url);
      gboolean(*launchInsidePossible) (void);
      gfloat(*zoomLevelGet) (GtkWidget * widget);
    void (*zoomLevelSet) (GtkWidget * widget, gfloat zoom);
      gboolean(*scrollPagedown) (GtkWidget * widget);
    void (*setProxy) (gchar * hostname, int port, gchar * username,
                      gchar * password);
    GtkWidget *(*get_buffer) ();
    void (*render_item) (itemPtr ip);
    gchar *(*render_head) (feedPtr fd);
    void (*render_foot) ();
    void (*render_and_add_item) (itemPtr ip);   /*Add item and display update, used in displaying search resutls */
    void (*display_scroll_to_news_item) (itemPtr ip);   /*TODO: when applet headline clicked */
    /* another one for scrolll to item, scroll to top */
};

htmlviewPluginInfo *htmlview_plugin_getinfo(void);

# define DECLARE_HTMLVIEW_PLUGIN(plugininfo) \
        G_MODULE_EXPORT htmlviewPluginInfo* htmlview_plugin_getinfo(void) { \
                return &plugininfo; \
        }


/** list type to provide a list of available modules for the preferences dialog */
struct browserModule {
    gchar *description;
    gchar *libname;
};

/** 
 * Function to set up the html view widget for the three
 * and two pane view. 
 */
GtkWidget *ui_htmlview_new();

/** loads a emtpy HTML page */
void ui_htmlview_clear(GtkWidget * htmlview);

/**
 * Function to add HTML source footer to create a valid HTML source.
 *
 * @param buffer	pointer to buffer to add the HTML to
 */
void ui_htmlview_finish_output(gchar ** buffer);

/**
 * Method to display the passed HTML source to the HTML widget.
 *
 * @param htmlview  The htmlview widget to be set
 * @param string	HTML source
 * @param base		base url for resolving relative links
 */
void ui_htmlview_write(GtkWidget * htmlview, gchar * string, gchar * base,
                       nodePtr node);

/**
 * Launches the specified URL in the configured browser or
 * in case of Mozilla inside the HTML widget.
 *
 * @param url			URL to launch
 * @param force_external	to force loading in external browser
 */
void ui_htmlview_launch_URL(gchar * url, gboolean force_external);

/**
 * Function to determine the currently selected URL.
 * The string must be freed by the caller.
 *
 * @return currently selected URL string.  
 */
gchar *ui_htmlview_get_selected_URL(void);

/**
 * Function scrolls down the item views scrolled window.
 *
 * @return FALSE if the scrolled window vertical scroll position is at
 * the maximum and TRUE if the vertical adjustment was increased.
 */
gboolean ui_htmlview_scroll(void);

/**
 * Function alerts the htmlview that the selected proxy has
 * changed.
 *
 * @param hostname proxy hostname, or NULL to disable the proxy
 * @param port proxy port
 * @param username proxy authentication username
 * @param password proxy authentication password
 */
void ui_htmlview_set_proxy(gchar * hostname, int port, gchar * username,
                           gchar * password);

/**
 * Function sends a text item to e-mail application for mailing.
 * @param item pointer to the text buffer
 */
int send_text_via_email(const gchar * post);

/**
 * Function sends RSS item to e-mail application for mailing.
 * @param item pointer to RSS news item to mail
 */
int send_item(itemPtr item, gboolean bluetooth);

/** Gets the GtkTextView widget for displaying html output
  *
  * @return the GtkTextView widget for displaying html output
  */
GtkWidget *ui_htmlview_get_buffer();

/** Renders an item to the html view
  *
  * @param itemPtr the item to render
  */
void ui_htmlview_render_item(itemPtr);
#endif
