/**
 * @file feed_types.c cash related feed handling
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "feed_types.h"
#include "conf.h"
#include "htmlview.h"
#include "vfolder.h"
#include "metadata.h"
#include "debug_new.h"

/* function to create a new feed structure */
feedPtr feed_new(void)
{
    feedPtr fp;

    fp = g_new0(struct feed, 1);

    /* we don't allocate a request structure this is done
     * during cache loading or first update! */

    fp->updateInterval = -1;
    fp->defaultInterval = -1;
    /*    Currently done where needed. */
    fp->addedTime = time(NULL);
    //    get_corrected_time(&(fp->addedTime));
    fp->type = FST_FEED;
    fp->cacheLimit = CACHE_DEFAULT;
    fp->newestPost = 0;
    fp->feedRead = 0;
    fp->new_subscription = FALSE;
    fp->feed_directory = FALSE;
    fp->tmpdata =
        g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_free);
    fp->downloadImages = TRUE;

    return fp;
}

/* ---------------------------------------------------------------------------- */
/* feed attributes encapsulation                        */
/* ---------------------------------------------------------------------------- */

void feed_set_id(feedPtr fp, const gchar * id)
{
    g_assert(fp != NULL);
    if (NULL != fp->id)
        g_free(fp->id);
    /*TODO: an appropriate remove_white_spaces is preferrable
     * this is just a hack to some feeds that have '\n' at the beginning
     * of the guid or their souce. Weird though
     */
    //g_strstrip(id);
    fp->id = g_strdup(id);
}
const gchar *feed_get_id(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->id;
}

void feed_set_type(feedPtr fp, gint type)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    fp->type = type;
    conf_feedlist_schedule_save(FALSE);
}

gint feed_get_type(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->type;
}

gpointer feed_get_favicon(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->icon;
}

void feed_increase_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->unreadCount++;
}

void feed_decrease_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->unreadCount--;
}

gint feed_get_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->unreadCount;
}

void feed_increase_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->newCount++;
}

void feed_decrease_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->newCount--;
}

gint feed_get_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->newCount;
}

void feed_increase_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->markCount++;
}

void feed_decrease_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->markCount--;
}

gint feed_get_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->markCount;
}

gint feed_get_default_update_interval(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->defaultInterval;
}

void feed_set_default_update_interval(feedPtr fp, gint interval)
{
    g_assert(fp != NULL);
    fp->defaultInterval = interval;
}

gint feed_get_update_interval(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->updateInterval;
}

void feed_set_update_interval(feedPtr fp, gint interval)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    fp->updateInterval = interval;

    conf_feedlist_schedule_save(FALSE);
}

const feedHandlerPtr feed_get_fhp(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->fhp;
}

void feed_reset_update_counter(feedPtr fp)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    g_get_current_time(&fp->lastPoll);
    //get_corrected_time(&(fp->lastPoll));
    /*fp->lastPoll.tv_sec = get_current_time(); */
    conf_feedlist_schedule_save(FALSE);
    debug2(DEBUG_CONF, "Reseting last poll counter for %s to %ld.\n",
           fp->title, fp->lastPoll.tv_sec);
}

void feed_set_available(feedPtr fp, gboolean available)
{
    g_assert(fp != NULL);
    fp->available = available;
}

gboolean feed_get_available(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->available;
}

void feed_set_download_images(feedPtr fp, gboolean download)
{
    g_assert(fp != NULL);
    fp->downloadImages = download;
}

gboolean feed_get_download_images(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->downloadImages;
}

void feed_set_discontinued(feedPtr fp, gboolean discontinued)
{
    g_assert(fp != NULL);
    fp->discontinued = discontinued;
}

gboolean feed_get_discontinued(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->discontinued;
}

void feed_set_nonremovable(feedPtr fp, gboolean nonremovable)
{
    g_assert(fp != NULL);
    fp->nonremovable = nonremovable;
}

gboolean feed_get_nonremovable(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->nonremovable;
}

/* Returns a HTML string describing the last retrieval error 
 * of this feed. Should only be called when feed_get_available
 * returns FALSE. Caller must free returned string! */
gchar *feed_get_error_description(feedPtr fp)
{
    gchar *tmp1 = NULL;

    if (feed_get_discontinued(fp)) {
        addToHTMLBufferFast(&tmp1, UPDATE_ERROR_START);
        addToHTMLBufferFast(&tmp1, HTTP410_ERROR_TEXT);
        addToHTMLBufferFast(&tmp1, UPDATE_ERROR_END);
    }
    addToHTMLBuffer(&tmp1, fp->errorDescription);
    return tmp1;
}

const time_t feed_get_time(feedPtr fp)
{
    return (fp != NULL ? fp->time : 0);
}
void feed_set_time(feedPtr fp, const time_t t)
{
    g_assert(fp != NULL);
    fp->time = t;
}

const gchar *feed_get_title(feedPtr fp)
{
    g_assert(fp != NULL);

    if (NULL != fp->title)
        return fp->title;
    else if (NULL != fp->source)
        return fp->source;
    else
        return NULL;
}

void feed_set_title(feedPtr fp, const gchar * title)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    g_free(fp->title);
    if (title != NULL)
        fp->title = g_strstrip(g_strdup(title));
    else
        fp->title = NULL;
    conf_feedlist_schedule_save(FALSE);
}


const gchar *feed_get_description(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->description;
}
void feed_set_description(feedPtr fp, const gchar * description)
{
    g_assert(fp != NULL);
    g_free(fp->description);
    if (description != NULL) {
        fp->description = g_strdup(description);
    } else
        fp->description = NULL;
}

const gchar *feed_get_source(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->source;
}
const gchar *feed_get_filter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->filtercmd;
}

void feed_set_source(feedPtr fp, const gchar * source)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    g_free(fp->source);

    fp->source = g_strdup(source);
    conf_feedlist_schedule_save(FALSE);
}

void feed_set_filter(feedPtr fp, const gchar * filter)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    g_free(fp->filtercmd);

    fp->filtercmd = g_strdup(filter);
    conf_feedlist_schedule_save(FALSE);
}

const gchar *feed_get_html_url(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->htmlUrl;
};
void feed_set_html_url(feedPtr fp, const gchar * htmlUrl)
{
    g_assert(fp != NULL);

    g_free(fp->htmlUrl);
    if (htmlUrl != NULL)
        fp->htmlUrl = g_strdup(htmlUrl);
    else
        fp->htmlUrl = NULL;
}

const time_t feed_get_added(feedPtr fp)
{
    return (fp != NULL ? fp->addedTime : 0);
}
void feed_set_added(feedPtr fp, const time_t t)
{
    g_assert(fp != NULL);
    if (t == 0)
        fp->addedTime = time(NULL);
    //get_corrected_time(&fp->addedTime);
    else
        fp->addedTime = t;
}

time_t feed_get_lastmodified(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->lastModified;
};
void feed_set_lastmodified(feedPtr fp, time_t lastmodified)
{
    g_assert(fp != NULL);

    fp->lastModified = lastmodified;
}

const gchar *feed_get_etag(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->etag;
};
void feed_set_etag(feedPtr fp, const gchar * etag)
{
    g_assert(fp != NULL);

    g_free(fp->etag);
    if (etag != NULL)
        fp->etag = g_strdup(etag);
    else
        fp->etag = NULL;
}

const gchar *feed_get_image_url(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->imageUrl;
};
void feed_set_image_url(feedPtr fp, const gchar * imageUrl)
{
    g_assert(fp != NULL);

    g_free(fp->imageUrl);
    if (imageUrl != NULL)
        fp->imageUrl = g_strdup(imageUrl);
    else
        fp->imageUrl = NULL;
}

/* returns feed's list of items, if necessary loads the feed from cache */
GSList *feed_get_item_list(feedPtr fp)
{
    g_assert(fp != NULL);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == fp->type));
    return fp->items;
}

void feed_set_sort_column(feedPtr fp, gint sortColumn, gboolean reversed)
{
    DMSG("Entering %s\n", __FUNCTION__);
    g_assert(fp != NULL);
    fp->sortColumn = sortColumn;
    fp->sortReversed = reversed;
    conf_feedlist_schedule_save(FALSE);
}

/**
 * Method to free all items structures of a feed, does not mean
 * that it removes items from cache! This method is used 
 * for feed unloading.
 */
void feed_clear_item_list(feedPtr fp)
{
    GSList *item = NULL;

    g_assert(fp != NULL);
    item = fp->items;

    while (NULL != item) {
        item_free(item->data);
        item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = NULL;
    /* explicitly not forcing feed saving to allow feed unloading */
}

/** 
 * Method to permanently removing all items from the given feed
 */
void feed_remove_items(feedPtr fp)
{
    GSList *item = NULL;

    g_assert(fp != NULL);
    //for proper removing of vfolder's items, feed has to be loaded 

    feed_load(fp);
    item = fp->items;

    while (NULL != item) {
        vfolder_remove_item(item->data);    /* remove item copies */
        remove_links_from_cache(item->data);
        item_free(item->data);  /* remove the item */
        item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = NULL;
    fp->needsCacheSave = TRUE;  /* force feed saving to make it permanent */
    feed_unload(fp);
}

/** 
 * Method to permanently remove unmarked items from feed
 */
void feed_remove_unmarked(feedPtr fp, gboolean save_unread)
{
    GSList *item = NULL;
    GSList *newlist = NULL;
    //    gboolean load_displayed_feed = FALSE;

    g_assert(fp != NULL);

    item = fp->items;

    /*    if((nodePtr)fp == displayed_node)
     * {
     * //        load_displayed_feed = TRUE;
     * load_displayed_feed = FALSE;
     * ui_itemlist_clear();
     * } */

    while (NULL != item) {
        if (item_get_mark((itemPtr) item->data)) {
            newlist = g_slist_append(newlist, item->data);
        } else if (save_unread && !item_get_read_status((itemPtr) item->data)) {
            newlist = g_slist_append(newlist, item->data);
        } else {
            vfolder_remove_item(item->data);    /* remove item copies */
            remove_links_from_cache(item->data);
            item_free(item->data);  /* remove the item */
        }
        item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = newlist;
    fp->needsCacheSave = TRUE;  /* force feed saving to make it permanent */

    /*    if(load_displayed_feed)
     * ui_feedlist_load_selected((nodePtr)fp); */
}

void feed_mark_all_items_read(feedPtr fp)
{
    GSList *item = NULL;

    feed_load(fp);
    item = fp->items;
    while (NULL != item) {
        item_set_read((itemPtr) item->data);
        item = g_slist_next(item);
    }
    feed_unload(fp);
}

time_t feed_get_newest_post(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->newestPost;
}

void feed_set_newest_post(feedPtr fp, time_t newestPost)
{
    g_assert(fp != NULL);

    fp->newestPost = newestPost;
}

void feed_set_feed_read(feedPtr fp, time_t time)
{
    GTimeVal current_time;

    g_assert(fp != NULL);

    if (time == -1) {
        g_get_current_time(&current_time);

        fp->feedRead = current_time.tv_sec;
    } else
        fp->feedRead = time;
    fp->needsCacheSave = TRUE;
}

time_t feed_get_feed_read(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->feedRead;
}

/**
 * To be used by parser implementation to merge a new orderd list of
 * items to a feed. Ensures properly ordered joint item list. The
 * passed GList is free'd afterwards!
 */
void feed_add_items(feedPtr fp, GList * items)
{
    GList *iter = NULL;

    g_assert(NULL != fp);

    if (NULL == items) {        /* Nothing to do - hopefully not an error */
        /*ULOG_WARN("Feed %s attempt to add NULL list.", fp->title); */
        return;
    }

    /* The parser implementations read the items from the
     * feed from top to bottom. Adding them directly would
     * mean to reverse there order. */
    iter = g_list_last(items);
    while (iter != NULL) {
        feed_add_item(fp, ((itemPtr) iter->data));
        iter = g_list_previous(iter);
    }

    g_list_free(items);
}

void feed_add_items_delete_old(feedPtr fp, GList * items)
{
    GList *iter = NULL;
    GSList *gs_iter = NULL;
    time_t newestPost = 0;
    itemPtr ip = NULL;
    gboolean item_added = FALSE;

    g_assert(NULL != fp);

    if (NULL == items) {        /* Nothing to do - hopefully not an error */
        /*ULOG_WARN("Feed %s attempt to add NULL list.", fp->title); */
        return;
    }

    ULOG_DEBUG("Adding new feed items ...");
    /* The parser implementations read the items from the
     * feed from top to bottom. Adding them directly would
     * mean to reverse there order. */
    iter = g_list_last(items);
    while (iter != NULL) {
       gboolean result = feed_add_item_ignore_old(fp, ((itemPtr) iter->data));
       if(!result) {
           iter->data = NULL;
       }

        if (result &&
            !item_added)
            item_added = TRUE;
        iter = g_list_previous(iter);
    }

    //We don't do this... this could be from older version?
    //The automatic deletion function is changed.
    /*ULOG_DEBUG("Removing unmarked ...");
     * if(item_added)
     * feed_remove_unmarked(fp, TRUE); */

    ULOG_DEBUG("Calculating newest post ...");
    gs_iter = fp->items;

    ip = NULL;
    while (gs_iter != NULL) {
        ip = gs_iter->data;

        if (ip->time > newestPost)
            newestPost = ip->time;
        gs_iter = g_slist_next(gs_iter);
    }
    if (newestPost > feed_get_newest_post(fp))
        feed_set_newest_post(fp, newestPost);

    ULOG_DEBUG("Freeing items ...");
    g_list_free(items);
    ULOG_DEBUG("Done ...");
}

/**
 * Can be used to add a single item to a feed. But it's better to
 * use feed_add_items() to keep the item order of parsed feeds.
 * Should be used for vfolders only.
 *
 * Adds an item to the feed. Realizes the item merging logic based on
 * the item id's. 
 *
 * Note: Does not do merging for vfolders. Would be hazardous!
 * Note: This version adds vfolder results automatically to display.
 */
void feed_add_item(feedPtr fp, itemPtr new_ip)
{
    GSList *old_items = NULL;
    itemPtr old_ip = NULL;
    gboolean found = FALSE, equal = FALSE;

    g_assert(NULL != fp);
    g_assert(NULL != new_ip);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));

    if (FST_VFOLDER != feed_get_type(fp)) {
        /* determine if we should add it... */
        debug1(DEBUG_VERBOSE, "check new item for merging: \"%s\"",
               item_get_title(new_ip));

        /* compare to every existing item in this feed */
        found = FALSE;
        old_items = feed_get_item_list(fp);
        while (NULL != old_items) {
            old_ip = old_items->data;

            /* try to compare the two items */

            /* trivial case: one item has item the other doesn't -> they can't be equal */
            if (((item_get_id(old_ip) == NULL)
                 && (item_get_id(new_ip) != NULL))
                || ((item_get_id(old_ip) != NULL)
                    && (item_get_id(new_ip) == NULL))) {
                /* cannot be equal (different ids) so compare to 
                 * next old item */
                ULOG_DEBUG("Comparing %s vs %s, id mismatch",
                           item_get_title(old_ip), item_get_title(new_ip));
                old_items = g_slist_next(old_items);
                continue;
            }

            /* just for the case there are no ids: compare titles and HTML descriptions */
            equal = TRUE;

            if (((item_get_title(old_ip) != NULL)
                 && (item_get_title(new_ip) != NULL))
                && (0 !=
                    strcmp(item_get_title(old_ip), item_get_title(new_ip))))
                equal = FALSE;

            if (((item_get_description(old_ip) != NULL)
                 && (item_get_description(new_ip) != NULL))
                && (0 !=
                    strcmp(item_get_description(old_ip),
                           item_get_description(new_ip))))
                equal = FALSE;

            /* best case: they both have ids (position important: id check is useless 
             * without knowing if the items are different!) */
            if (NULL != item_get_id(old_ip)) {
                if (0 == strcmp(item_get_id(old_ip), item_get_id(new_ip))) {
                    found = TRUE;
                    break;
                }
            }

            if (equal) {
                found = TRUE;
                break;
            }

            old_items = g_slist_next(old_items);
        }

        if (!found) {
            if (0 == new_ip->nr) {
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            /* ensure that the feed last item nr is at maximum */
            if (new_ip->nr > fp->lastItemNr)
                fp->lastItemNr = new_ip->nr;

            /* ensure that the item nr's are unique */
            if (NULL != feed_lookup_item_by_nr(fp, new_ip->nr)) {
                g_warning
                    ("The item number to be added is not unique! Feed (%s) Item (%s) (%ld)\n",
                     fp->id, new_ip->title, new_ip->nr);
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            if (FALSE == item_get_read_status(new_ip))
                feed_increase_unread_counter(fp);
            if (TRUE == item_get_mark(new_ip))
                feed_increase_mark_counter(fp);
            if (TRUE == new_ip->newStatus) {
                fp->newCount++;
            }
            fp->items = g_slist_prepend(fp->items, (gpointer) new_ip);
            new_ip->fp = fp;

            /* check if the item matches any vfolder rules */
            vfolder_check_item(new_ip);

            debug0(DEBUG_VERBOSE, "-> item added to feed itemlist");

        } else {
            /* if the item was found but has other contents -> update contents */
            if (!equal) {
                /* no item_set_new_status() - we don't treat changed items as new items! */
                item_set_title(old_ip, item_get_title(new_ip));
                item_set_description(old_ip, item_get_description(new_ip));
                item_set_time(old_ip, item_get_time(new_ip));
                item_set_unread(old_ip);    /* FIXME: needed? */
                metadata_list_free(old_ip->metadata);
                old_ip->metadata = new_ip->metadata;
                vfolder_update_item(old_ip);
                debug0(DEBUG_VERBOSE,
                       "-> item already existing and was updated");
            } else {
                debug0(DEBUG_VERBOSE, "-> item already exists");
            }
        }                       //while end
    } else {
        /* vfolder/search - just add the item */
        if (FALSE == item_get_read_status(new_ip))
            feed_increase_unread_counter(fp);
        fp->items = g_slist_append(fp->items, (gpointer) new_ip);
        new_ip->fp = fp;
        /* item number should already be set by item_copy() */

        //ui_itemlist_search_append(new_ip);
    }
}

void feed_set_parse_errors(feedPtr fp, gboolean parse_errors)
{
    g_assert(fp != NULL);

    fp->err = parse_errors;
}
