/**
 * @file feed.h common feed handling
 * 
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _FEED_H
#define _FEED_H

#include <glib.h>
#include "item.h"
#include "folder.h"
#include "feed_types.h"

/* ------------------------------------------------------------ */
/* feed creation/modification interface */
/* ------------------------------------------------------------ */

/** 
 * Initializes feed parsing handlers. Should be called 
 * only once on program startup.
 */
void feed_init(void);

/**
 * General feed source parsing function. Parses the passed feed source
 * and tries to determine the source type. If the type is HTML and 
 * autodiscover is TRUE the function tries to find a feed, tries to
 * download it and parse the feed's source instead of the passed source.
 *
 * @param fp		the feed structure to be filled
 * @param data		the feed source
 * @param dataLength the length of the 'data' string
 * @param autodiscover	TRUE if auto discovery should be possible
 */
feedHandlerPtr feed_parse(feedPtr fp, gchar * data, size_t dataLength,
                          gboolean autodiscover, struct request *nreq);

void feed_merge(feedPtr old_fp, feedPtr new_fp);
void feed_remove(feedPtr fp);
void feed_schedule_update(feedPtr fp, gint flags);

void feed_remove_item(feedPtr fp, itemPtr ip);

/** 
 * To lookup an item given by it's unique item nr 
 */
itemPtr feed_lookup_item_by_nr(feedPtr fp, gulong nr);

/** 
 * To lookup an item given by it's unique item nr 
 */
itemPtr feed_lookup_item_by_nr_extended(feedPtr fp, gulong nr,
                                        gchar * orig_fp_id);

/**
 * Mark a feeds update requests as canceled.
 */
void feed_free_requests(nodePtr ptr);

/**
 * Removes all feed caches and saves opml file. 
 * savechanges needs to be TRUE for opml to be saved.
 * this is added because in some cases opml needs to be saved 
 * first to ensure there's enough space on the device and only
 * then data can be removed from the internal structures
 * If savechanges is FALSE, opml is not saved but all other stuff
 * is done. Dirty hack i know, shoot me or ask for refactoring =)
 *
 * */

gboolean feed_free(feedPtr fp, gboolean savechanges);

/**
 * This is a timeout callback to check for feed update results.
 * If there is a result pending its data is parsed and merged
 * against the feed it belongs to.
 */
void feed_process_update_result(struct request *request);

/** Returns a HTML rendering of a feed. The returned string must be
  * freed
  *
  *  @param fp the feed to get the HTML rendering for
  */
gchar *feed_render(feedPtr fp);

/** Adds an item to a feed if it is newer than any items before, which
  * means that no feed gets downloaded twice
  *
  * @param fp the feed to add an item to
  * @param new_ip the new item to add
  * @return TRUE if an item has been added
  */
gboolean feed_add_item_ignore_old(feedPtr fp, itemPtr new_ip);


/** This is set when a new subscription is made and an internet connection has been
  * established
  *
  * @param fp the feed to set the new subscription for
  * @param new_subscription the value to set
  */
void feed_set_new_subscription(feedPtr fp, gboolean new_subscription);

/** Gets the value of the new_subscription field
  *
  * @param fp the feed to get the new subscription field for
  */
gboolean feed_get_new_subscription(feedPtr fp);

/** This is set if the feed is a feed directory
  *
  * @param fp the feed to set as a feed directory
  * @param feed_directory the value to set
  */
void feed_set_feed_directory(feedPtr fp, gboolean feed_directory);

/** Gets the value of the feed_directory field
  *
  * @param fp the feed to get the value for
  */
gboolean feed_get_feed_directory(feedPtr fp);

/** Gets the parse errors field
  *
  * @param fp the feed to get the parse errors for
  * @return TRUE if there were parse errors, FALSE otherwise
  */
gboolean feed_get_parse_errors(feedPtr fp);

void feed_download_images(feedPtr fp);

extern gboolean conf_keep_feeds_in_memory;

void feed_parse_new_download(struct request *nreq);

#endif
