/*
 * Copyright (C) 2009-2010 Marco Barisione <marco@barisione.org>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <string.h>

#include "merge.h"


typedef struct
{
    OssoABookContact *rc;
    gchar            *old_value;
} FBContact;

struct _MergeCandidate
{
    OssoABookContact *mc;
    GList            *fb_contacts;

    gint              ref_count;
};

static MergeCandidate *
merge_candidate_new (OssoABookContact *mc)
{
    MergeCandidate *candidate;

    candidate = g_new0 (MergeCandidate, 1);
    candidate->ref_count = 1;
    candidate->mc = g_object_ref (mc);

    return candidate;
}

MergeCandidate *
merge_candidate_ref (MergeCandidate *candidate)
{
    candidate->ref_count++;

    return candidate;
}

void
merge_candidate_unref (MergeCandidate *candidate)
{
    candidate->ref_count--;

    if (candidate->ref_count > 0)
        return;

    g_object_unref (candidate->mc);

    while (candidate->fb_contacts) {
        FBContact *fb = candidate->fb_contacts->data;

        g_object_unref (fb->rc);
        g_free (fb->old_value);
        g_free (fb);

        candidate->fb_contacts = g_list_delete_link (
                candidate->fb_contacts, candidate->fb_contacts);
    }

    g_free (candidate);
}

static void
merge_candidate_add_fb_contact (MergeCandidate   *candidate,
                                OssoABookContact *rc,
                                const gchar      *old_value)
{
    FBContact *fb_contact;

    fb_contact = g_new0 (FBContact, 1);
    fb_contact->old_value = g_strdup (old_value);
    fb_contact->rc = g_object_ref (rc);

    candidate->fb_contacts = g_list_prepend (
            candidate->fb_contacts, fb_contact);
}

static void
merge_candidate_add_roster_contact (MergeCandidate   *candidate,
                                    const gchar      *old_user_name)
{
    OssoABookAggregator *aggregator;
    gchar *new_user_name;
    GList *roster_contacts;
    OssoABookContact *rc;
    const gchar *bound_name;
    const gchar *vcard_field;

    aggregator = OSSO_ABOOK_AGGREGATOR (osso_abook_aggregator_get_default (NULL));

    new_user_name = g_strdup (old_user_name);
    new_user_name[0] = '-';

    /* We cannot use osso_abook_aggregator_find_contacts_for_im_contact()
     * as it finds the master contacts for the user name and not the
     * roster ones */
    roster_contacts = osso_abook_aggregator_list_roster_contacts (aggregator);
    while (roster_contacts) {
        rc = roster_contacts->data;
        bound_name = osso_abook_contact_get_bound_name (rc);
        vcard_field = osso_abook_contact_get_vcard_field (rc);
        if (strcmp (bound_name, new_user_name) == 0 &&
            strcmp (vcard_field, EVC_X_JABBER) == 0) {
            merge_candidate_add_fb_contact (candidate, rc, old_user_name);
        }

        roster_contacts = g_list_delete_link (roster_contacts, roster_contacts);
    }

    g_free (new_user_name);
}

static void
merge_candidate_do_merge (MergeCandidate *candidate,
                          EBookCallback   cb,
                          gpointer        user_data)
{
    GList *l;
    FBContact *fb_contact;
    const gchar *bound_name;
    EVCardAttribute *attr;

    for (l = candidate->fb_contacts; l; l = l->next) {
        fb_contact = l->data;

        /* Remove the old value */
        osso_abook_contact_remove_value (E_CONTACT (candidate->mc),
                EVC_X_JABBER, fb_contact->old_value);

        /* Add the new one */
        attr = e_vcard_attribute_new (NULL, EVC_X_JABBER);
        bound_name = osso_abook_contact_get_bound_name (fb_contact->rc);
        e_vcard_attribute_add_value (attr, bound_name);
        e_vcard_add_attribute (E_VCARD (candidate->mc), attr);

        /* Merge the IM and master contacts */
        osso_abook_contact_accept (fb_contact->rc, candidate->mc, NULL);
    }

    osso_abook_contact_async_commit (candidate->mc, NULL, cb, user_data);
}

typedef struct
{
    GList *candidates;

    gint merged_count;
    gint failed_count;
    gint total_count;

    MergeCb  updated_cb;
    MergeCb  finished_cb;
    gpointer user_data;
} MergeClosure;

static void
continue_merge (MergeClosure *closure);

static void
master_committed_cb (EBook       *book,
                     EBookStatus  status,
                     gpointer     user_data)
{
    MergeClosure *closure;

    closure = user_data;

    if (status == E_BOOK_ERROR_OK)
        closure->merged_count++;
    else
        closure->failed_count++;

    merge_candidate_unref (closure->candidates->data);
    closure->candidates = g_list_delete_link (
            closure->candidates, closure->candidates);

    if (closure->updated_cb)
        closure->updated_cb (closure->merged_count, closure->failed_count,
                closure->total_count, closure->user_data);

    continue_merge (closure);
}

static void
continue_merge (MergeClosure *closure)
{
    if (closure->candidates == NULL) {
        if (closure->finished_cb)
            closure->finished_cb (closure->merged_count,
                    closure->failed_count, closure->total_count,
                    closure->user_data);
        g_free (closure);
    }
    else {
        merge_candidate_do_merge (closure->candidates->data,
                master_committed_cb, closure);
    }
}

void
merge_candidates_do_merge (GList    *candidates,
                           MergeCb   updated_cb,
                           MergeCb   finished_cb,
                           gpointer  user_data)
{
    MergeClosure *closure;

    closure = g_new0 (MergeClosure, 1);
    closure->updated_cb = updated_cb;
    closure->finished_cb = finished_cb;
    closure->user_data = user_data;
    closure->total_count = g_list_length (candidates);
    closure->candidates = g_list_copy (candidates);
    g_list_foreach (closure->candidates, (GFunc) merge_candidate_ref, NULL);

    continue_merge (closure);
}

void
merge_candidates_free (GList *candidates)
{
    g_list_foreach (candidates, (GFunc) merge_candidate_unref, NULL);
    g_list_free (candidates);
}

static void
add_candidate_if_match (GHashTable       *candidates,
                        OssoABookContact *mc,
                        EVCardAttribute  *attr)
{
    GList *values;
    const gchar *user_name;
    GRegex *regex;
    GMatchInfo *match_info;
    MergeCandidate *candidate;

    if (strcmp (e_vcard_attribute_get_name (attr), EVC_X_JABBER) != 0)
        return;

    values = e_vcard_attribute_get_values (attr);
    if (g_list_length (values) != 1) {
        g_warning ("Multiple values where a single value was expected");
        return;
    }

    user_name = values->data;

    regex = g_regex_new ("u\\d+@chat\\.facebook\\.com", 0, 0, NULL);
    g_regex_match (regex, user_name, 0, &match_info);
    if (g_match_info_matches (match_info)) {
        candidate = g_hash_table_lookup (candidates, mc);
        if (!candidate) {
            candidate = merge_candidate_new (mc);
            g_hash_table_insert (candidates, mc, candidate);
        }
        merge_candidate_add_roster_contact (candidate, user_name);
    }

    g_match_info_free (match_info);
    g_regex_unref (regex);
}

GList *
merge_candidates_get (void)
{
    OssoABookAggregator *aggregator;
    GHashTable *candidates;
    GList *master_contacts;
    OssoABookContact *mc;
    GList *attrs;
    gpointer value;
    MergeCandidate *candidate;
    GHashTableIter iter;
    GList *ret = NULL;

    aggregator = OSSO_ABOOK_AGGREGATOR (osso_abook_aggregator_get_default (NULL));

    candidates = g_hash_table_new (g_direct_hash, g_direct_equal);

    master_contacts = osso_abook_aggregator_list_master_contacts (aggregator);
    while (master_contacts) {
        mc = master_contacts->data;

        attrs = osso_abook_contact_get_attributes (E_CONTACT (mc),
                EVC_X_JABBER);
        while (attrs) {
            add_candidate_if_match (candidates, mc, attrs->data);
            attrs = g_list_delete_link (attrs, attrs);
        }

        master_contacts = g_list_delete_link (master_contacts, master_contacts);
    }

    g_hash_table_iter_init (&iter, candidates);
    while (g_hash_table_iter_next (&iter, NULL, &value)) {
        candidate = value;
        if (candidate->fb_contacts != NULL)
            ret = g_list_prepend (ret, candidate);
        else
            /* This can happen if the facebook contact was removed in
             * the meantime */
            merge_candidate_unref (candidate);
    }

    g_hash_table_unref (candidates);

    return ret;
}
