/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
/** @mainpage Microfeed Documentation
 *
 * Microfeed is a specification and a reference implementation of client-server architecture
 * providing access to various information sources that have a feed-type interface.
 * Examples of those feed sources include micro-blogging services, such as Jaiku, Twitter,
 * Laconi.ca, and FriendFeed. By utilizing Microfeed architecture, the client application can
 * focus on user interface, while the actual feed fetching in done in the background
 * independently. The communication between the local server publishing information about
 * the feeds and the client application displaying that information to an user is done with
 * the DBus messaging following the publisher-subscriber principle.
 *
 * The local server is called provider. There can be - and usually are - many providers,
 * one for each micro-blogging or similar service, and thus a provider can focus solely
 * on polling one service. A provider may be instantiated and configured several ways at the
 * same time, for example, with different service URLs or authentication information.
 * One configuration forms a publisher, and thus a publisher is a configured instance of
 * a provider that provides the access to the service.
 *
 * The client application is acting as a subscriber. It subscribes the feeds it is interested.
 * A publisher sends a notification each time an item is added into, changed in, or removed from
 * the feed. It also notifies the changed status of the feed itself. One publisher can serve
 * multiple subscribers at the same time.
 *
 * If you are implementing a publisher, you need the following modules:
 * @link MicrofeedPublisher MicrofeedPublisher@endlink,
 * @link MicrofeedFeed MicrofeedFeed@endlink,
 * @link MicrofeedItem MicrofeedItem@endlink, and
 * @link MicrofeedConfiguration MicrofeedConfiguration@endlink. It is strongly adviced that you use also
 * @link MicrofeedProvider MicrofeedProvider@endlink. You may also find useful the following
 * modules:
 * @link MicrofeedMain MicrofeedMain@endlink,
 * @link MicrofeedHttp MicrofeedHttp@endlink, and
 * @link MicrofeedJson MicrofeedJson@endlink.
 * 
 *
 * If you are implementing a subscriber, you need the following modules:
 * @link MicrofeedSubscriber MicrofeedSubsrciber@endlink and
 * @link MicrofeedItem MicrofeedItem@endlink. It is highly propable that you need also
 * @link MicrofeedConfiguration MicrofeedConfiguration@endlink. You may also find useful
 * the following modules:
 * @link MicrofeedMain MicrofeedMain@endlink.
 *
 * The following modules are used internally in the implementation, but can be utilized
 * also outside:
 * @link MicrofeedDatabase MicrofeedDatabase@endlink,
 * @link MicrofeedStore MicrofeedStore@endlink, and
 * @link MicrofeedThread MicrofeedThread@endlink.
 */

#ifndef MICROFEEDPROTOCOL_H
#define MICROFEEDPROTOCOL_H

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedProtocol
 *
 * Protocol constants as preprocessor macros.
 *
 * @{
 */


#define MICROFEED_DBUS_INTERFACE_PREFIX "org.microfeed."
#define MICROFEED_DBUS_INTERFACE_PUBLISHER MICROFEED_DBUS_INTERFACE_PREFIX "Publisher"
#define MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION MICROFEED_DBUS_INTERFACE_PUBLISHER ".ToDestination"
#define MICROFEED_DBUS_INTERFACE_SUBSCRIBER MICROFEED_DBUS_INTERFACE_PREFIX "Subscriber"
#define MICROFEED_DBUS_INTERFACE_ERROR MICROFEED_DBUS_INTERFACE_PREFIX "Error"
#define MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION MICROFEED_DBUS_INTERFACE_ERROR ".ToDestination"
#define MICROFEED_DBUS_OBJECT_PATH_PREFIX "/org/microfeed/"
#define MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER MICROFEED_DBUS_OBJECT_PATH_PREFIX "publisher/"

#define MICROFEED_ERROR_UNKNOWN "UnknownError"
#define MICROFEED_ERROR_DBUS_MESSAGE_FAILED "DBusMessageFailed"
#define MICROFEED_ERROR_PROVIDER_CLOSED_CONNECTION "ProviderClosedConnection"
#define MICROFEED_ERROR_INVALID_PUBLISHER_IDENTIFIER "InvalidPublisherIdentifier"
#define MICROFEED_ERROR_INVALID_FEED_URI "InvalidFeedUri"
#define MICROFEED_ERROR_INVALID_ARGUMENTS "InvalidArguments"
#define MICROFEED_ERROR_MISSING_PROPERTY "MissingProperty"
#define MICROFEED_ERROR_FEED_NOT_SUBSCRIBED "FeedNotSubscribed"
#define MICROFEED_ERROR_FEED_ALREADY_SUBSCRIBED "FeedAlreadySubscribed"
#define MICROFEED_ERROR_PUBLISHER_ALREADY_EXISTS "PublisherAlreadyExists"
#define MICROFEED_ERROR_ITEM_ALREADY_EXISTS "ItemAlreadyExists"
#define MICROFEED_ERROR_NO_SUCH_PROVIDER "NoSuchProvider"
#define MICROFEED_ERROR_NO_SUCH_PUBLISHER "NoSuchPublisher"
#define MICROFEED_ERROR_NO_SUCH_FEED "NoSuchFeed"
#define MICROFEED_ERROR_NO_SUCH_ITEM "NoSuchItem"
#define MICROFEED_ERROR_NO_SUCH_PROPERTY "NoSuchProperty"
#define MICROFEED_ERROR_NO_SUCH_DATA "NoSuchData"
#define MICROFEED_ERROR_DATA_NOT_SUPPORTED "DataNotSupported"
#define MICROFEED_ERROR_CANNOT_MODIFY_ITEM "CannotModifyItem"
#define MICROFEED_ERROR_CONNECTION_FAILED "ConnectionFailed"
#define MICROFEED_ERROR_LOGIN_FAILED "LoginFailed"
#define MICROFEED_ERROR_COMMUNICATION_FAILED "CommunicationFailed"
#define MICROFEED_ERROR_FROM_SERVICE "FromService"

#define MICROFEED_METHOD_NAME_ADD_ITEM "AddItem"
#define MICROFEED_METHOD_NAME_MODIFY_ITEM "ModifyItem"
#define MICROFEED_METHOD_NAME_REMOVE_ITEM "RemoveItem"
#define MICROFEED_METHOD_NAME_SUBSCRIBE_FEED "SubscribeFeed"
#define MICROFEED_METHOD_NAME_UNSUBSCRIBE_FEED "UnsubscribeFeed"
#define MICROFEED_METHOD_NAME_UPDATE_FEED "UpdateFeed"
#define MICROFEED_METHOD_NAME_REPUBLISH_ITEMS "RepublishItems"
#define MICROFEED_METHOD_NAME_STORE_DATA "StoreData"
#define MICROFEED_METHOD_NAME_CREATE_PUBLISHER "CreatePublisher"
#define MICROFEED_METHOD_NAME_DESTROY_PUBLISHER "DestroyPublisher"
#define MICROFEED_METHOD_NAME_MARK_ITEM "MarkItem"
#define MICROFEED_METHOD_NAME_UNMARK_ITEM "UnmarkItem"
#define MICROFEED_METHOD_NAME_PING "Ping"
#define MICROFEED_METHOD_NAME_READ_ITEM "ReadItem"

#define MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED "FeedUpdateStarted"
#define MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED "FeedUpdateEnded"
#define MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED "FeedRepublishingStarted"
#define MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED "FeedRepublishingEnded"
#define MICROFEED_SIGNAL_NAME_ITEM_ADDED "ItemAdded"
#define MICROFEED_SIGNAL_NAME_ITEM_CHANGED "ItemChanged"
#define MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED "ItemRepublished"
#define MICROFEED_SIGNAL_NAME_ITEM_REMOVED "ItemRemoved"
#define MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED "ItemStatusChanged"
#define MICROFEED_SIGNAL_NAME_DATA_STORED "DataStored"

#define MICROFEED_FEED_URI_OVERVIEW ":Overview"
#define MICROFEED_FEED_URI_SETTINGS ":Settings"
#define MICROFEED_FEED_URI_CONTACTS ":Contacts"
#define MICROFEED_FEED_URI_USER_PREFIX ":User/"
#define MICROFEED_FEED_URI_COMMENTS_PREFIX ":Comments/"

#define MICROFEED_ITEM_UID_FEED_METADATA ":Metadata"

#define MICROFEED_ITEM_PERMISSION_NAME_MODIFY ":modify"
#define MICROFEED_ITEM_PERMISSION_NAME_DELETE ":delete"
#define MICROFEED_ITEM_PERMISSION_NAME_REPLY ":reply"

#define MICROFEED_FEED_PERMISSION_NAME_ADD ":add"

#define MICROFEED_ITEM_PROPERTY_NAME_CONTENT_IMAGE "content.image"
#define MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT "content.text"
#define MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE "user.image"
#define MICROFEED_ITEM_PROPERTY_NAME_USER_NAME "user.name"
#define MICROFEED_ITEM_PROPERTY_NAME_USER_NICK "user.nick"
#define MICROFEED_ITEM_PROPERTY_NAME_USER_FEED "user.feed"
#define MICROFEED_ITEM_PROPERTY_NAME_USER_URL "user.url"
#define MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM "referred.item"
#define MICROFEED_ITEM_PROPERTY_NAME_REFERRED_FEED "referred.feed"
#define MICROFEED_ITEM_PROPERTY_NAME_COMMENTS_COUNT "comments.count"
#define MICROFEED_ITEM_PROPERTY_NAME_COMMENTS_FEED "comments.feed"

#define MICROFEED_ITEM_PROPERTY_NAME_SETTING_LENGTH "setting.length"
#define MICROFEED_ITEM_PROPERTY_NAME_SETTING_TEXT "setting.text"
#define MICROFEED_ITEM_PROPERTY_NAME_SETTING_TYPE "setting.type"
#define MICROFEED_ITEM_PROPERTY_NAME_SETTING_UNIT "setting.unit"
#define MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE "setting.value"
#define MICROFEED_ITEM_PROPERTY_NAME_FEED_NAME "feed.name"
#define MICROFEED_ITEM_PROPERTY_NAME_FEED_PERMISSION "feed.permission"

#define MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR '-'
#define MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_STRING "-"

/**
 * @}
 * @}
 */

#endif
