/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDFILE_H
#define MICROFEEDFILE_H

#define _GNU_SOURCE

#include <stdint.h>
#include <sys/types.h>

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedFile
 *
 * @{
 */

/**
 * Opaque data type representing a file.
 */
typedef struct _MicrofeedFile MicrofeedFile;
typedef uint32_t MicrofeedFileIndex;

#define MICROFEED_FILE_INDEX_INVALID ((MicrofeedFileIndex)-1)

MicrofeedFile* microfeed_file_new(const char* path, int create);
void microfeed_file_free(MicrofeedFile* file);

#define microfeed_file_allocate_block(f,s,t) ((t*)microfeed_file_allocate_block_impl((f), (s)))
void* microfeed_file_allocate_block_impl(MicrofeedFile* file, size_t size);
#define microfeed_file_resize_block(f,b,s,t) ((t*)microfeed_file_resize_block_impl((f), (b), (s)))
void* microfeed_file_resize_block_impl(MicrofeedFile* file, void* block, size_t size);
void microfeed_file_free_block(MicrofeedFile* file, void* block);
void microfeed_file_free_index(MicrofeedFile* file, MicrofeedFileIndex index);
MicrofeedFileIndex microfeed_file_get_index(MicrofeedFile* file, void* block);
#define microfeed_file_get_block(f,i,t) ((t*)microfeed_file_get_block_impl((f), (i)))
void* microfeed_file_get_block_impl(MicrofeedFile* file, MicrofeedFileIndex index);
size_t microfeed_file_get_index_size(MicrofeedFile* file, MicrofeedFileIndex index);
void microfeed_file_print_statistics(MicrofeedFile* file);
size_t microfeed_file_block_get_size(void* block);

/**
 * @}
 * @}
 */

#endif
