/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 * Copyright (C) <2006> Nokia Corporation (contact <stefan.kost@nokia.com>)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __GST_AVI_DEMUX_H__
#define __GST_AVI_DEMUX_H__

#include <gst/gst.h>

#include "avi-ids.h"
#include "gst/riff/riff-ids.h"
#include "gst/riff/riff-read.h"
#include <gst/base/gstadapter.h>

G_BEGIN_DECLS

#define GST_TYPE_AVI_DEMUX \
  (gst_avi_demux_get_type ())
#define GST_AVI_DEMUX(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj), GST_TYPE_AVI_DEMUX, GstAviDemux))
#define GST_AVI_DEMUX_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST ((klass), GST_TYPE_AVI_DEMUX, GstAviDemuxClass))
#define GST_IS_AVI_DEMUX(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GST_TYPE_AVI_DEMUX))
#define GST_IS_AVI_DEMUX_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE ((klass), GST_TYPE_AVI_DEMUX))

#define GST_AVI_DEMUX_MAX_STREAMS       16

/*
 * Saves memory because it does not interpolate between index frames and
 * create seeking entries "in-between" the index frames.
 * This way the index structure in the memory will consist only of the
 * actual index frames found in the AVI header.
 * The memory saving is quite notable with big AVI clips (from several Megs
 * to some tens of kilobytes).
 *
 * The big downside is that this lead to a different code path in avidemux right
 * now. Original avidemux is playing index-entry by index-entry (see 
 * gst_avi_demux_process_next_entry()). Progess gets tracked via
 * GstAviDemux->current_entry. The gst_avi_index_entry->offset points directly
 * to the data and gst_avi_index_entry->size bytes can be pulled and send to the
 * decoder. The timestamp and the duration of the buffer map 1:1 to the
 * index-entry.
 *
 * In the memory optimized version the progress is tracked via 
 * avi_stream_context->current_pos (see gst_avi_demux_read_chunk() ).
 * The difference is that multiple index entries will be played and those index
 * entries aren't in the index table. So the current_pos =
 * gst_avi_index_entry->offset - 8. 8 is the chuck header (id + size).
 * For each buffer, timestamp and duration have to be calculated from position
 * queries.
 *
 * FIXME: wouldn't it be better to have a flag 'avi->multi-entry'. Then instead
 * of _process_next_entry() it would use _process_next_entries(). This would
 * work similar to _read_chunk(), but read and concat all entries spanned by the
 * current index entry. This means we can get rid of managing current_pos in
 * streams and the buffer fields can again be taken from the index-entry
 * (timestamp and duration)
 * Ultimately this should make the changes less cluttered over the source and
 * increase the likelyhood to become accepted upstream.
 *
 * FIXME: make this a gobject property if accepted
 */
#define MEMOPT 1

#define CHUNKID_TO_STREAMNR(chunkid) \
  ((((chunkid) & 0xff) - '0') * 10 + \
   (((chunkid) >> 8) & 0xff) - '0')

/* one index entry (104 bytes) */
typedef struct {
  gint           index_nr;
  gint           stream_nr;
  guint64        ts, dur;
  guint32        flags;
  guint64        offset;
  gint           size;
  guint64        bytes_before;
  guint32        frames_before;
} gst_avi_index_entry;

typedef struct {
  /* index of this streamcontext */
  guint          num;

  /* pad*/
  GstPad        *pad;

  /* stream info and headers */
  gst_riff_strh *strh;
  union {
    gst_riff_strf_vids *vids;
    gst_riff_strf_auds *auds;
    gst_riff_strf_iavs *iavs;
    gpointer     data;
  } strf;
  GstBuffer     *extradata, *initdata;
  gchar         *name;

  /* current position (byte, frame, time) and other status vars */
  guint          current_frame;
  guint64        current_byte;
#ifdef MEMOPT
  guint64        current_pos;
#endif
  GstFlowReturn  last_flow;
  gboolean       discont;

  /* stream length */
  guint64        total_bytes;
  guint32        total_frames;
  /* stream length according to index */
  GstClockTime   idx_duration;
  /* stream length according to header */
  GstClockTime   hdr_duration;
  /* stream length based on header/index */
  GstClockTime   duration;

  /* VBR indicator */
  gboolean       is_vbr;

  gboolean       superindex;
  guint64       *indexes;

  GstTagList	*taglist;
} avi_stream_context;

typedef enum {
  GST_AVI_DEMUX_START,
  GST_AVI_DEMUX_HEADER,
  GST_AVI_DEMUX_MOVI,
} GstAviDemuxState;

typedef enum {
  GST_AVI_DEMUX_HEADER_TAG_LIST,
  GST_AVI_DEMUX_HEADER_AVIH,
  GST_AVI_DEMUX_HEADER_ELEMENTS,
  GST_AVI_DEMUX_HEADER_INFO,
  GST_AVI_DEMUX_HEADER_JUNK,
  GST_AVI_DEMUX_HEADER_DATA
} GstAviDemuxHeaderState;

typedef struct _GstAviDemux {
  GstElement     parent;

  /* pads */
  GstPad        *sinkpad;

  /* AVI decoding state */
  GstAviDemuxState state;
  GstAviDemuxHeaderState header_state;
  guint64        offset;

  /* index */
  gst_avi_index_entry *index_entries;
  guint          index_size;
  guint64        index_offset;
  guint          current_entry;

  /* streams */
  guint          num_streams;
  guint          num_v_streams;
  guint          num_a_streams;
#ifdef MEMOPT
  guint          current_stream;
  gboolean       local_without_index;
  gint64         total_length;
#endif

  avi_stream_context stream[GST_AVI_DEMUX_MAX_STREAMS];

  /* for streaming mode */
  gboolean      streaming;
  gboolean      have_eos;
  GstAdapter    *adapter;

  /* some stream info for length */
  gst_riff_avih *avih;

  /* segment in TIME */
  GstSegment     segment;
  gboolean       segment_running;

  /* pending tags/events */
  GstEvent      *seek_event;
  GstTagList	*globaltags;
  gboolean	got_tags;

} GstAviDemux;

typedef struct _GstAviDemuxClass {
  GstElementClass parent_class;
} GstAviDemuxClass;

GType           gst_avi_demux_get_type          (void);

G_END_DECLS

#endif /* __GST_AVI_DEMUX_H__ */
