# -*- coding: utf-8 -*-
'''
Common Config
2008-2012(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

try: #most distros have a separate gconf
  import gconf
except: #but Maemo doesn't
  from gnome import gconf

class Config():

  def __init__(self, settings_path='/'):
    self.__config = {}
    self.__gconf_path = settings_path
    self.__gconf_client = gconf.client_get_default()
    self.__gconf_types = {gconf.VALUE_LIST: list, gconf.VALUE_PAIR: tuple, gconf.VALUE_STRING: str, gconf.VALUE_INT: int, gconf.VALUE_BOOL: bool, gconf.VALUE_FLOAT: long}
    self.__types = dict(zip(self.__gconf_types.values(), self.__gconf_types.keys()))
    self.__gconf_value_methods = {gconf.VALUE_BOOL: 'bool', gconf.VALUE_INT: 'int', gconf.VALUE_STRING: 'string', gconf.VALUE_FLOAT: 'float', gconf.VALUE_LIST: 'list', gconf.VALUE_SCHEMA: 'schema'}
    self.__options = {}
    self.load()
    #print self.__config

  def get_keys(self):
    return self.__config.keys()

  def get_values(self):
    return self.__config.values()

  def get_option(self, key):
    try:
      return self.__options[key]
    except KeyError:
      return None

  def get_options(self):
    return self.__options

  def get_entry(self, key):
    try:
      return self.__config[key]['name'], self.__config[key]['type'], self.__config[key]['value']
    except KeyError:
      return None, None, None

  def get_value(self, key):
    try:
      return self.__config[key]['value']
    except KeyError:
      return None

  def add_entry(self, key, key_value=None, save=False, callback=None):
    if not key in self.__config:
      if key_value and not type(key_value) in self.__types:
        key_value = str(key_value)
      self.__config[key] = {'type': type(key_value), 'value': key_value, 'name': key.split('/')[-1]}
      if save:
        self.save_key(key, callback)
      elif callback:
        callback(key, None, key_value)
    return self.__config[key]

  def set_value(self, key, key_value, save=False, callback=None):
    try:
      old_value = self.__config[key]['value']
      self.__config[key]['value'] = self.__config[key]['type'](key_value)
      if save:
        self.save_key(key, callback, old_value)
      elif callback:
        callback(key, old_value, key_value)
    except KeyError:
      self.add_entry(key, key_value, save, callback)

  def set_option(self, key, value):
    self.__options[key] = value

  def set_options(self, options = {}):
    self.__options = options

  def clear_options(self):
    self.__options.clear()

  def save(self, callback=None):
    for key in self.__config:
      self.save_key(key, callback)
    if self.__gconf_client:
      self.__gconf_client.suggest_sync()

  def save_key(self, key, callback=None, old_value=None):
    try:
      self.__set_gvalue('/'.join((self.__gconf_path, key)),
                        self.__config[key]['value'])
      if callback:
        callback(key, old_value, self.__config[key]['value'])
    except KeyError:
      pass

  def load(self):
    if not self.__gconf_client:
      return
    self.__config.clear()
    self.__parse_dir(self.__gconf_path)

  def __parse_dir(self, g_dir):
    gconf_entries = self.__gconf_client.all_entries(g_dir)
    for entry in gconf_entries:
      self.__parse_entry(entry)
    gconf_dirs = self.__gconf_client.all_dirs(g_dir)
    for gconf_dir in gconf_dirs:
      self.__parse_dir(gconf_dir)

  def __parse_entry(self, entry):
    g_path = entry.get_key()
    self.add_entry(self.__get_rpath(g_path), self.__get_gvalue(g_path))

  def __get_type(self, g_type):
    try:
      return self.__gconf_types[g_type]
    except KeyError:
      return str

  def __get_gvalue(self, g_path):
    g_value = self.__gconf_client.get(g_path)
    g_type = g_value.type
    value = None
    if g_type == gconf.VALUE_LIST:
      g_list_type = g_value.get_list_type()
      g_list_values = g_value.get_list(g_list_type)
      list_type = self.__get_type(g_list_type)
      value = [self.__get_primitive_gvalue(g_list_value) for g_list_value in g_list_values]
    elif g_type == gconf.VALUE_PAIR:
      pair_car = self.__get_primitive_gvalue(g_value.get_car())
      pair_cdr = self.__get_primitive_gvalue(g_value.get_cdr())
      value = (pair_car, pair_cdr)
    elif g_type == gconf.VALUE_SCHEMA:
      pass
    else:
      value = self.__get_primitive_gvalue(g_value)
    return value

  def __get_primitive_gvalue(self, g_value):
    g_type = g_value.type
    return self.__get_type(g_type)(getattr(g_value, 'get_%s' % (self.__gconf_value_methods[g_type]))())

  def __set_gvalue(self, g_path, value):
    g_value = self.__gconf_client.get(g_path)
    g_type = None
    if g_value:
      g_type = g_value.type
    else:
      g_type = self.__types[type(self.__config[self.__get_rpath(g_path)]['value'])]
    kargs = []
    if g_type == gconf.VALUE_LIST:
      kargs = [g_value.get_list_type(), value]
    elif g_type == gconf.VALUE_PAIR:
      kargs = [g_value.get_car().type, g_value.get_cdr().type, value[0], value[1]]
    elif g_type == gconf.VALUE_SCHEMA:
      kargs = [value]
    else:
      kargs = [value]
    getattr(self.__gconf_client, 'set_%s' % (self.__gconf_value_methods[g_type]))(g_path, *kargs)

  def __get_rpath(self, path):
    return path.replace(('%s/' % self.__gconf_path), '', 1)
