/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "tweet.h"

/*!
 * \brief Class Constructor
 * \param entry object with all the tweet's informations
 */
Tweet::Tweet(Entry entry, QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
    _entry(entry),
    _avatar(QImage(":images/resources/avatar.png"))
{
    setMinimumSize(454, 74);

    _paintImage = false;

    _message = new QGraphicsTextItem(this);
    _message->setDefaultTextColor(QColor(0, 0, 0));

    createTweetMessage();

    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(contains(event->pos()))
        emit createTweetInfo(this);
}

/*!
 * \brief Writes The whole tweet message
 * \param entry
 */
void Tweet::createTweetMessage(void)
{
    _favorited = new QGraphicsPixmapItem(this);
    if(_entry.favorited) {
        _favorited->setPixmap(QPixmap::fromImage(QImage(":images/resources/fave_ico.png")));
    }

    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    if(_entry.retweeted_status->id == 0) {
        messageHtml += "<span style=\" color:#0f95da; font-size:21px\">" + _entry.userInfo.screenName + "</span>";
        messageHtml += "<span style=\" color:#757575; font-size:17px\"> &nbsp;" + _entry.userInfo.name + "<br></span>";
    } else {
        messageHtml += "<span style=\" color:#0f95da; font-size:21px\">" + _entry.retweeted_status->userInfo.screenName + "</span>";
        messageHtml += "<span style=\" color:#757575; font-size:17px\"> &nbsp;" + _entry.retweeted_status->userInfo.name + "</span>";
    }

    if(_entry.retweeted_status->id != 0) {
        messageHtml += "&nbsp;<img src=\":images/resources/rt_timeline.png\"/>";
        messageHtml += "<span style=\" color:#757575; font-size:17px\"> by ";
        messageHtml += _entry.userInfo.screenName + "<br></span>";
    }

    if(_entry.retweeted_status->id == 0) {
        messageHtml += "<span style=\" color:#000000;\">" + _entry.text + "</span></p>";
    } else {
        messageHtml += "<span style=\" color:#000000;\">" + _entry.retweeted_status->text + "</span></p>";
    }

    QString messageInfo = "<p align=\"right\"; style=\" padding-top:5px; margin-top:5px; font-size:14px; font-family:'Nokia Sans';\"> ";
    messageInfo += "<span style=\" color=#757575;\">" + _entry.parsedTime;
    messageInfo += " via " + _entry.source + "</span></p>";

    _message->setHtml(messageHtml + messageInfo);

    if(_message->document()->lineCount() < 2) {
        messageHtml.replace("</p>","<br></p>");
        _message->setHtml(messageHtml + messageInfo);
    }

    if (_entry.hasInReplyToStatusId) {
        messageInfo.replace("</span></p>"," in reply to " + _entry.inReplyToScreenName + "</span></p>");
        _message->setHtml(messageHtml + messageInfo);
    }

    if(_entry.retweeted_status->id == 0) {
        _url = _entry.userInfo.imageUrl;
    } else {
        _url = _entry.retweeted_status->userInfo.imageUrl;
    }
    _download(_url);
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void Tweet::_download(QString url)
{
    _imageReady(url);
    ImageDownload::instance()->imageGet(url);
    connect(ImageDownload::instance(),SIGNAL(imageReadyForUrl(QString)), SLOT(_imageReady(QString)));
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void Tweet::_imageReady(QString url)
{
    if(url != _url)
        return;

    QPixmap pm;
    QPixmapCache::find( url, pm );

    if ( pm.isNull() )
        return;

    _avatar = pm.toImage();

    emit imageDownloaded();
    update();
}

/*!
 * \brief Paints the Tweet white rectangle and the user avatar
 * \param event Unused
 */
void Tweet::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    // Transparent Rect
    painter->setPen(Qt::NoPen);

    if(_entry.isOwn)
        painter->setBrush(QColor(255, 240, 204));
    else
        painter->setBrush(QColor(246, 253, 255));

    painter->drawRoundedRect(0, 0, rect().width() - 10, rect().height(), 2, 2);
    painter->setRenderHint(QPainter::Antialiasing);

    // Avatar Rect
    painter->drawImage(QRect(13, 13, 48, 48), _avatar);
}

/*!
 * \brief Method to get the avatar's image from tweet
 * \return The avatar image
 */
QImage Tweet::getImage(void)
{
    return _avatar;
}

/*!
 * \brief Method to get the Tweet message in html format
 * \return The message in html format
 */
QTextDocument* Tweet::getTweetDocument()
{
    return _message->document();
}

Entry Tweet::getEntry(void)
{
    return _entry;
}

void Tweet::_changeOrientation(void)
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    if(screenGeometry.width() > screenGeometry.height()) {
        _message->setTextWidth(702);
    } else if(screenGeometry.width() < screenGeometry.height()) {
        _message->setTextWidth(382);
    }
    setMinimumSize(-1, -1);
    setMaximumSize(-1, -1);

    setMinimumSize(screenGeometry.width() - 5, _message->document()->size().height());
    setMaximumSize(minimumSize());

    _favorited->setPos(screenGeometry.width() - _favorited->boundingRect().width() - 20, 5);
    _message->setPos(74,0);

    emit resizeScene();
}
