# -*- coding: utf-8 -*-
import sqlite3
import ConfigParser
import os


class SQLConnection():
    # Borg singleton object
    __singleton = {}


    @staticmethod
    def dbtype():
        return "sqlite"


    def __init__(self, dbfile = None, schemafile = None, blobdir = None):
        self.new = False
        if dbfile != None:
            if os.path.isfile(dbfile) == False:
                self.new = True
                
        if self.__singleton.has_key("sqlite_conn") != True:
            if dbfile == None or schemafile == None or blobdir == None:
                raise Exception("database connection not initialized")
            self.__singleton["sqlite_conn"] = sqlite3.connect(dbfile)
            self.__singleton["sqlite_schema"] = {}
            try:
                os.mkdir(blobdir)
            except:
                pass
            self.__singleton["sqlite_blobdir"] = blobdir
            config = ConfigParser.ConfigParser()
            config.read(schemafile)
            for table,schema in config.items("schema"):
                self.__singleton["sqlite_schema"][table] = {}
                field_strs = schema.split(",")
                for field_str in field_strs:
                    field, sep, desc = field_str.strip().partition(" ")
                    self.__singleton["sqlite_schema"][table][field] = desc
            self.__singleton["autocommit_counter"] = 0
            self.__singleton["autocommit_limit"] = 0

        self.conn = self.__singleton["sqlite_conn"]
        self.schema_object = self.__singleton["sqlite_schema"]
        self.blobdir = self.__singleton["sqlite_blobdir"]


    def cursor(self):
        return self.conn.cursor()


    def set_commit_limit(self, limit):
        self.__singleton["autocommit_counter"] = 0
        self.__singleton["autocommit_limit"] = limit
        if self.__singleton["autocommit_limit"] < 1:
            self.conn.commit()
            

    def commit(self, force = False):
        self.__singleton["autocommit_counter"] += 1
        if self.__singleton["autocommit_counter"] >= self.__singleton["autocommit_limit"] or force == True:
            self.conn.commit()
            self.__singleton["autocommit_counter"] = 0


    def schema(self, table):
        if self.schema_object.has_key(table) != True:
            raise Exception("invalid table given %s" % (table))
        schema = {}
        for field, desc in self.schema_object[table].items():
            schema[field] = None
        return schema


    def create_database(self):
        for table, fields in self.__singleton["sqlite_schema"].items():
            field_descs = []
            for field, desc in fields.items():
                field_descs.append("%s %s" % (field, desc))
            query = "create table %s (%s)" % (table, ", ".join(field_descs))
            try:
                self.conn.execute(query)
            except:
                for field, desc in fields.items():
                    query = "alter table %s add %s %s" % (table, field, desc)
                    try:
                        self.conn.execute(query)
                    except:
                        pass
        

class SQLObject():
    def __init__(self, table, p1 = None):
        self._conn = SQLConnection()

        self._fields = self._conn.schema(table)
        
        self._new = False
        self._table = table
        if isinstance(p1, tuple):
            p2 = self._fields.keys()
            for i in range(len(p1)):
                self._fields[p2[i]] = p1[i]
        # new object
        else:
            self._new = True


    def __eq__(self, other):
        if type(self) == type(other):
            return True
        return False
        
        
    def __getattr__(self, name):
        if name[0] == "_":
            return self.__dict__[name]
        else:
            if self.__dict__["_fields"].has_key(name):
                return self.__dict__["_fields"][name]
            else:
                return self.__dict__[name]


    def __setattr__(self, name, value):
        if name[0] == "_":
            self.__dict__[name] = value
        else:
            if self.__dict__["_fields"].has_key(name):
                self.__dict__["_fields"][name] = value
            else:
                self.__dict__[name] = value


    def get_by(self, field, value):
        fields = self._fields.keys()
        q = SQLQuery(self._table)
        q.add_constraint(field, "=", value)
        r = q.execute()
        if len(r) < 1:
            raise Exception("object not found from %s with %s = %s" % (self._table, field, value))
        self._fields = r[0]._fields
        self._new = False


    def create(self):
        if self._new == False:
            raise Exception("this is not new object, cannot create")
        fields = []
        query_params = []
        values_ex = []
        for k, v in self._fields.items():
            if v == None:
                continue
            fields.append(str(k))
            query_params.append(str(v))
            values_ex.append("?")
        if len(fields) < 1:
            raise Exception("no values to be set")
        query = "insert into %s(%s) values (%s)" % (self._table, ", ".join(fields), ", ".join(values_ex))
        cursor = self._conn.cursor()
        cursor.execute(query, tuple(query_params))
        self.id = cursor.lastrowid
        
        self._conn.commit()
        self._new = False


    def update(self):
        if self._new == True:
            raise Exception("this is new object, cannot update")
        fields = []
        query_params = []
        for k, v in self._fields.items():
            if v == None:
                continue
            fields.append(str(k)+" = ?")
            query_params.append(str(v))
        if len(fields) < 1:
            raise Exception("no values to be set")
        query_params.append(self.id)
        query = "update %s set %s where id = ?" % (self._table, ", ".join(fields))
        cursor = self._conn.cursor()
        cursor.execute(query, tuple(query_params))
        self._conn.commit()


    def delete(self):
        if self._new == True:
            raise Exception("this is new object, cannot delete")
        query = "delete from %s where id = %s" % (self._table, self.id)
        cursor = self._conn.cursor()
        cursor.execute(query)
        self._conn.commit()


    def blobWrite(self, name, data):
        filename = "%s/%s_%s_%s" % (self._conn.blobdir, self._table, self.id, name)
        f = open(filename, "wb")
        f.write(data)
        f.close()


    def blobPath(self, name):
        filename = "%s/%s_%s_%s" % (self._conn.blobdir, self._table, self.id, name)
        if os.path.isfile(filename):
            return filename
        return None
        

class SQLQuery():
    def __init__(self, table):
        self._conn = SQLConnection()

        self.table = table
        self.fields = self._conn.schema(table).keys()
        self.constraints = []
        self.orders = []
        self.limit = None
        

    def add_constraint(self, prop, const, value):
        self.constraints.append([ str(prop), str(const), str(value) ])

    def add_order(self, field, order = "ASC"):
        self.orders.append("%s %s" % (field, order))

    def set_limit(self, limit):
        self.limit = int(limit)
        
    def execute(self):
        if self.fields == None:
            query = "select * from %s" % (self.table)
        else:
            query = "select %s from %s" % (", ".join(self.fields), self.table)
            
        query_params = []
        if len(self.constraints) > 0:
            query += " where "
            for c in self.constraints:
                if len(query_params) > 0:
                    query += " and"
                query_params.append(c[2])
                query += " %s %s ?" % (c[0], c[1])
                
        if len(self.orders) > 0:
            query += " order by " + ", ".join(self.orders)
        if self.limit != None:
            query += " limit %s" % (self.limit)
            
        cursor = self._conn.cursor()
        cursor.execute(query, tuple(query_params))
        results = []
        while True:
            o = cursor.fetchone()
            if o == None:
                break
            results.append(SQLObject(self.table, o))
        return results


    def count(self):
        query = "select count(*) from %s" % (self.table)
        query_params = []
        if len(self.constraints) > 0:
            query += " where "
            for c in self.constraints:
                if len(query_params) > 0:
                    query += " and"
                query_params.append(c[2])
                query += " %s %s ?" % (c[0], c[1])
        cursor = self._conn.cursor()
        cursor.execute(query, tuple(query_params))
        r = cursor.fetchone()
        return r[0]

        