#    Openvpn Applet
#    Copyright (C) 2008 - 2010  Mikko Vartiainen <mvartiainen@gmail.com>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
import os

if os.path.exists("/tmp/openvpn-applet.debug"):
    enable_logging = True
else:
    enable_logging = False

if enable_logging:
    fd = open('/tmp/openvpn-applet.log','w+')
else:
    fd = None

app_version = '0.6.0'
about_name = 'OpenVPN Applet'
about_text = 'Applet for controlling OpenVPN connections'
about_authors = 'Mikko Vartiainen'
about_website = 'http://openvpn-applet.garage.maemo.org/'
about_bugtracker = 'http://bugs.maemo.org/enter_bug.cgi?product=openvpn'
about_donate = 'https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=XB5F5CMCPY9SE&lc=FI&item_name=Openvpn%20Applet&currency_code=EUR&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted'
donate_wishlist_url = ''
donate_device_url = ''


import sys
sys.path.insert(0, '/usr/lib/openvpn-applet')

import glob
import string
from subprocess import Popen, PIPE, call
import gtk
import hildondesktop
import hildon
import gobject
from sets import Set
import fileinput
import re
from UserDict import UserDict
import socket
import time
import conic

def copy_file(file):
    log("Copy: ", file)
    for f in file:
        if f != None:
            call(['sudo','/usr/libexec/openvpn-applet-helper.py', 'copy', str(f), str(file[f])])

def delete_files(file):
    log("Delete: ", file)
    for f in file:
        if f != None:
            call(['sudo','/usr/libexec/openvpn-applet-helper.py', 'delete', f])
    

def pid_exists(pid):
    try:
        os.kill(pid, 0)
        return True
    except OSError, err:
        if err.errno == 1:
            return True
        else:
            return False
    
def openvpn_running(pid):
    p = Popen("pidof openvpn", shell=True, stdout=PIPE)
    pipe = p.stdout
    s = pipe.read().split()
    p.wait()
    if s.count(str(pid)):
        return True
    else:
        return False
        
class vpn():
    def __init__(self, conf_file, conf_dir='/etc/openvpn', auth_callback_fb=None):
        self.conf_file = conf_file
        self.name = string.split(os.path.basename(self.conf_file),'.')[0]
        self.pid_file = '/var/run/openvpn.'+self.name+'.pid'
        self.conf_dir = conf_dir
        self.stopcmd = '/etc/init.d/openvpn'
        self.startcmd = '/usr/sbin/openvpn'
        self.socket_name = '/tmp/'+self.name+'.socket'
        self.socket = None
        self.auth_callback_fb = auth_callback_fb

    def __socket_connect(self):
        if self.socket == None:
            self.socket = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            try:
                self.socket.connect(self.socket_name)
            except socket.error, (value,message):
                log("socket.connect() failed:",value, message)
                self.socket = None
                raise socket.error(value,message);
                return False
            return True
        return False

    def __socket_disconnect(self):
        if self.socket != None:
            self.socket.close()
            self.socket = None

    def __admin_command(self, command, get_response = True, disconnect = True, wait_for_ack = True):
        data = ''
        if self.socket == None: 
            if self.__socket_connect() == False:
                return data
        bytes_sent = self.socket.send(command)
        log("Command: "+command,bytes_sent)
        if (get_response):
            data = self.__admin_recv(wait_for_ack)
            log("Response:", data[-3:])

        if disconnect:
            self.__socket_disconnect()
        return data

    def __admin_recv(self, wait_for_ack=False):
        if self.socket == None: return

        data = ''
        while True:
            chunk = self.socket.recv(4096)
            #log(chunk)
            data = data+chunk
            lines = data.split("\r\n")
            if wait_for_ack:
                if data.find("SUCCESS") != -1: break
                if data.find("ERROR") != -1: break
                if data.find("END") != -1: break
                if data.find(">FATAL") != -1: break
            else:
                if lines[-1] == '': break
        return lines

    def get_log(self):
        data = "\n".join(self.__admin_command("log all\n")[1:])
        while True:
            mindex = data.find("MANAGEMENT")
            if mindex == -1: break
            end = data.find("\n",mindex)
            if end == -1: end = len(data)
            start = data.rfind("\n",0,mindex)
            if start >= mindex or start == -1: start = 0
            data = data[:start+1]+data[end+1:]
    
        return data[:-4]

    def get_statistics(self):
        data = self.__admin_command("status\n")
        return "\n".join(data[1:-2])

    def get_state(self):
        data = self.__admin_command("state all\n")
        return "\n".join(data[1:-2])
        
    def get_status(self):
        pid = self.get_pid()
                
        if pid != None and pid_exists(pid) and openvpn_running(pid):
            #self.pid = pid
            return True
        else:
            return False
        
    def get_pid(self):
        try:
            f = open(self.pid_file)
        except IOError:
            return None
        
        pid = int(f.readline())    
        return int(pid)
        
    def start(self,user=None,passwd=None,enable_log=False):
        call(['sudo','/usr/libexec/add_default_route.sh'])
        command =["sudo",self.startcmd,"--daemon","--writepid",self.pid_file,"--config",self.conf_file,"--persist-key", "--management",self.socket_name,"unix","--management-hold","--management-query-passwords", "--cd", self.conf_dir]
        if enable_log:
            command.extend(['--verb','6'])
        log(command)
        process = Popen(command, shell=False, stdin=PIPE, stdout=None, cwd=self.conf_dir)
        input = process.stdin
        input.close()

        for i in range(1,10):
            try:
                self.__admin_command("hold off\n")
            except socket.error:
                time.sleep(0.2)
            else:
                break
        response = self.__admin_command("hold release\n", disconnect=False)
        auth_entered = []
        for i in range(2):
            response.extend(self.__admin_command("test 1\n", disconnect=False, wait_for_ack=False))
            response.extend(self.__admin_command("test 1\n", disconnect=True, wait_for_ack=False))
            for line in response:
                if line[:9] == ">PASSWORD":
                    if len(line.split()) > 3:
                        (a, auth_name,auth_name2, auth_type) = line.split()
                        auth_name = auth_name+" "+auth_name2
                    else:
                        (a, auth_name, auth_type) = line.split()
                    auth_name = auth_name[1:-1]
                    ask_username = False

                    if auth_type == 'username/password':
                        ask_username = True
                        log("auth-user-pass", auth_name)
                    if auth_type == 'password':
                        ask_username = False
                        log("askpass", auth_name)

                    try:
                        auth_entered.index(auth_name)
                    except ValueError:
                        (username,password,response_ok) = self.auth_callback_fb(ask_username, "OpenVPN Authentication: "+self.name+" "+auth_name)
                        auth_entered.append(auth_name)
                        if response_ok:
                            if ask_username: self.__admin_command("username \""+auth_name+"\" \""+username+"\"\n")
                            self.__admin_command("password \""+auth_name+"\" \""+password+"\"\n")
                        else:
                            self.stop()

        return process.returncode
        
    def stop(self):
        log("stop")
        self.__admin_command("signal SIGTERM\n")

    def restart(self):
        log("restart")
        self.__admin_command("signal SIGUSR1\n")

    def get_conf_file(self):
        return self.conf_file        

    def get_name(self):
        return self.name

class vpn_control():
    def read_vpn_conf(self, dir, ext):
        vpn_conf = []
        for e in ext:
            l = glob.glob(dir+'/*.'+e)
            if len(l) > 0: vpn_conf.extend(l)
        return vpn_conf        
        
    def __init__(self):
        self.update()
    
    def __str__(self):
        s = ''
        for i,v in enumerate(self.vpn_list):
            if v.get_status() != True: cmd = 'start'
            else: cmd = 'stop'
            s = s+str(i+1)+'. '+v.name+' - '+cmd+'\n'
        return s   

    def update(self):
        self.vpn_list = []
        for v in self.read_vpn_conf('/etc/openvpn',['conf','ovpn']):
            log(v)
            self.vpn_list.append(vpn(v, auth_callback_fb=user_pass_dlg))
        for v in self.read_vpn_conf('/home/user/.openvpn-applet',['conf','ovpn']):
            log(v)
            self.vpn_list.append(vpn(v,'/home/user/.openvpn-applet',auth_callback_fb=user_pass_dlg))

    def start(self, index,user=None,passwd=None,enable_log=False):
        self.vpn_list[index].start(user,passwd)
        
    def stop(self, index):
        self.vpn_list[index].stop()

    def restart_connected(self):
        log("restart_connected")
        for vpn in self.vpn_list:
            if vpn.get_status():
                vpn.restart()
        
    def status(self, index):
        if len(self.vpn_list) > 0:
            return self.vpn_list[index].get_status()
        else:
            return -1

    def get_vpn_count(self):
        return len(self.vpn_list)

    def get_active_connections_number(self):
        active = 0
        for i,v in enumerate(self.vpn_list):
            if v.get_status(): active = active + 1
        return active

    def get_vpn_name(self, number):
        return self.vpn_list[number].get_name()

    def get_conf_file(self, number):
        return self.vpn_list[number].get_conf_file()

    
class openvpn_applet(hildondesktop.StatusMenuItem):
    def __init__(self):
        hildondesktop.StatusMenuItem.__init__(self)

        self.connection = conic.Connection()
        self.connection.set_property("automatic-connection-events", True)
        self.connection.connect("connection-event", self.connection_cb)

        self.vc = None
        self.banner = None
        self.timer = None
        self.timeout = 0
        #self.connection_text = "no connections"
        self.button_title_text = "OpenVPN"

        self.button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        #self.button.set_text(self.button_title_text, self.connection_text)
        self.button.set_style(hildon.BUTTON_STYLE_PICKER)
        self.button.set_alignment(0.2,0.5,1,1)
        image = gtk.image_new_from_icon_name("openvpn-applet", gtk.ICON_SIZE_BUTTON)
        self.button.set_image(image)
        self.button.set_image_position(gtk.POS_LEFT)

        #self.button = gtk.Button("OpenVPN");
        self.button.connect("clicked", self.button_clicked_event)
        
        self.status_area_icon = gtk.gdk.pixbuf_new_from_file_at_size("/usr/share/icons/hicolor/48x48/apps/openvpn-applet.png", 18, 18)

        self.add(self.button)
        self.show_all()
        self.update_list()
        self.update_button()
        #self.set_settings(True)
        #self.connect("show-settings", self.open_settings)
     
    def update_list(self,source=None):
        if self.vc == None:
            self.vc = vpn_control()
        else:
            self.vc.update()

    def get_vpn_list(self):
        return self.combo.get_model()
        
    def button_clicked_event(self,widget):
        dlg = SelectConnectionDialog(parent=None,vpn_manager=self.vc)            
        dlg.connect("response",self.manage_connection)
        dlg.show_all()
        self.update_button()
    
    def update_button(self):
        log("update_button")
        active = self.vc.get_active_connections_number()
        if active == 1:
            self.button.set_text(self.button_title_text, str(active) + ' active connection')
            self.set_status_area_icon(self.status_area_icon)
        elif active > 1:
            self.button.set_text(self.button_title_text, str(active) + ' active connections')
            self.set_status_area_icon(self.status_area_icon)
        else:
            self.button.set_text(self.button_title_text, 'no active connections')
            self.set_status_area_icon(None)

    def update_button_timer(self):
        self.timeout = self.timeout - 1
        self.update_button()
        if self.timeout > 0:
            self.timer = gobject.timeout_add(500, self.update_button_timer)
        elif self.vc.get_active_connections_number() > 0:
            self.timer = gobject.timeout_add(30000, self.update_button_timer)
        else:
            self.timer = None
            return False

    def settings_event(self, widget, data=None):
        widget = gtk.MenuItem("OpenVPN setup")
        widget.connect("activate", self.open_settings)
        return widget
        
    def open_settings(self, widget=None, data=None):
        #self.set_options_visible(False)
        dlg = OpenvpnAppletSettings(vpn_manager=self.vc)
        dlg.run()
        dlg.destroy()
        #self.set_options_visible(True)
        #self.combo.set_active(0)
        self.update_button()

    def manage_connection(self, widget, vpn_number):
        if vpn_number == gtk.RESPONSE_HELP:
            self.open_settings()
            return

        if vpn_number >= 0  and vpn_number < self.vc.get_vpn_count():
            if self.vc.status(vpn_number) == False:
                self.banner = hildon.hildon_banner_show_information(self, "", "Starting Openvpn Connection")
                self.vc.start(vpn_number,None,None)
            else:
                self.banner = hildon.hildon_banner_show_information(self,"", "Stopping Openvpn Connection")
                self.vc.stop(vpn_number)
            self.timeout = 10
            self.timer = gobject.timeout_add(500, self.update_button_timer)
        elif vpn_number == self.vc.get_vpn_count():
            log("open settings")
            self.open_settings(self)

    def connection_cb(self, connection, event):
        log("connection_cb",connection, event)
        
        status = event.get_status()
        error = event.get_error()
        iap_id = event.get_iap_id()
        bearer = event.get_bearer_type()
        
        if status == conic.STATUS_CONNECTED:
            log("CONNECTED",iap_id, bearer, status, error)
            if bearer == "GPRS" and self.vc != None and self.vc.get_active_connections_number() > 0:
                log("add_default_route", self.vc.get_active_connections_number())
                call(['sudo','/usr/libexec/add_default_route.sh'])
            if self.vc != None:
                self.vc.restart_connected()
        elif status == conic.STATUS_DISCONNECTED:
            log("(DISCONNECTED",iap_id, bearer, status, error)
        elif status == conic.STATUS_DISCONNECTING:
            log("(DISCONNECTING",iap_id, bearer, status, error)

def user_pass_dlg(ask_passwd, message):
    if ask_passwd:
        dlg = hildon.LoginDialog(gtk.Window())
        dlg.set_message(message)
        response = dlg.run()
        user_pass_resp = (dlg.get_username(),dlg.get_password())
        dlg.destroy()
    else:
        dlg = hildon.GetPasswordDialog(gtk.Window(),False)
        dlg.set_message(message)
        response = dlg.run()
        user_pass_resp = (None,dlg.get_password())
        dlg.destroy()

    if response == gtk.RESPONSE_OK:
        user_pass_resp = (user_pass_resp[0],user_pass_resp[1],True)
    else:
        user_pass_resp = (user_pass_resp[0],user_pass_resp[1],False)

    return user_pass_resp

class SelectConnectionDialog(gtk.Dialog):
    def __init__(self,title="Select OpenVPN Connection", parent=None, flags=0, buttons=None,vpn_manager=None):
        gtk.Dialog.__init__(self,title, parent, flags, buttons)
        self.vpn_manager = vpn_manager
        vbox = gtk.VBox()
        pannable_area = hildon.PannableArea()

        for vpn_number,v in enumerate(vpn_manager.vpn_list):
            hbox = gtk.HBox()
            button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
            if v.get_status():
                stats_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
                stats_button.set_text("Status",'')
                stats_button.connect("clicked", self.stats_button_clicked_event, vpn_number)
                hbox.pack_start(stats_button)
                button.set_style(hildon.BUTTON_STYLE_PICKER)
                button.set_text("Disconnect from",v.get_name())
            else:
                button.set_style(hildon.BUTTON_STYLE_NORMAL)
                button.set_text("Connect to",v.get_name())
            hbox.pack_start(button)
            vbox.pack_start(hbox)
            button.connect("clicked", self.button_clicked_event, vpn_number)

        button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        button.set_text("Manage Connections","")
        button.connect("clicked", self.button_clicked_event, gtk.RESPONSE_HELP)
        vbox.pack_start(button)

        pannable_area.add_with_viewport(vbox)
        self.vbox.pack_start(pannable_area)
        self.action_area.show_all()
        self.vbox.show_all()
        self.set_size_request(800,vbox.size_request()[1]+10)
        #self.show_all()

    def button_clicked_event(self, widget, vpn_number):
        self.response(vpn_number)
        self.destroy()

    def stats_button_clicked_event(self, widget, vpn_number):
        dlg = OpenvpnAppletTestConnection(parent=self, title=self.vpn_manager.get_vpn_name(vpn_number), vpn_conn=self.vpn_manager.vpn_list[vpn_number])            
        dlg.run()        
        dlg.destroy()

class OpenvpnAppletSettings(gtk.Dialog):
    def __init__(self,title="OpenVPN connections", parent=None, flags=0, buttons=None, vpn_manager=None, applet=None):
        gtk.Dialog.__init__(self,title, parent, flags, buttons)

        self.vpn_manager = vpn_manager
        self.selection = None
        self.update_selection()

        self.new_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.new_button.set_label("New")
        self.new_button.connect("clicked", self.open_new_dialog)
        self.delete_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.delete_button.set_label("Delete")
        self.delete_button.connect("clicked", self.delete_clicked_event)
        self.test_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.test_button.set_label("Test")
        self.test_button.connect("clicked", self.test_clicked_event)
        self.about_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.about_button.set_label("About")
        self.about_button.connect("clicked", self.about_clicked_event)

        self.selection.set_size_request(-1, 155)
        self.action_area.pack_start(self.new_button)
        self.action_area.pack_start(self.delete_button)
        self.action_area.pack_start(self.test_button)
        self.action_area.pack_start(self.about_button)
        self.add_button("Close", gtk.RESPONSE_CLOSE)
        self.action_area.show_all()
        self.vbox.show_all()

    def update_selection(self):
        self.vpn_manager.update()
        if self.selection != None:
	    self.vbox.remove(self.selection)
            self.selection.destroy()

        self.selection = hildon.TouchSelector(text = True)
        self.vbox.pack_start(self.selection)   
        for i in range(self.vpn_manager.get_vpn_count()):
            self.selection.insert_text(i,self.vpn_manager.get_vpn_name(i))

	self.vbox.show_all()
 
    def open_new_dialog(self, widget, data=None):
        dlg = OpenvpnAppletNewConnection()
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            files_to_overwrite = {}
            files_to_overwrite_message = []
            files_to_copy = {}
            all_files = dlg.get_filenames()
            for f in all_files:
                target_file = os.path.join('/etc/openvpn', os.path.basename(f))
                if os.path.isfile(target_file):
                    files_to_overwrite[f] = all_files[f]
                    files_to_overwrite_message.append(target_file)
                else:
                    files_to_copy[f] =  all_files[f]
            if len(files_to_overwrite) > 0:
                log("Files to overwrite:", files_to_overwrite)
                mstr = ""
                for f in files_to_overwrite_message:
                    mstr = mstr+"\n"+f
                mdlg = gtk.MessageDialog(dlg,0,gtk.MESSAGE_QUESTION,gtk.BUTTONS_NONE,"Overwrite files?\n"+mstr)
                mdlg.add_buttons("Overwrite", gtk.RESPONSE_ACCEPT,"Don't Overwrite", gtk.RESPONSE_NONE)
                response = mdlg.run()
                mdlg.destroy()
                if response == gtk.RESPONSE_ACCEPT:
                    files_to_copy = all_files
            else:
                files_to_copy = all_files
                        
            copy_file(files_to_copy)
            self.update_selection()
        dlg.destroy()

    def delete_clicked_event(self, widget, data=None):
        log("delete_clicked_event")
        file_links = FileLinks("/etc/openvpn", ["conf", "ovpn"])
        vpn_number = self.selection.get_active(0)
        if vpn_number >= 0:
            log("Item selected")
            filename = self.vpn_manager.get_conf_file(vpn_number)
            log(filename)
            log(file_links[filename])
            mstr = filename
            for f in file_links[filename]:
                mstr += "\n"+str(f)
            dlg = gtk.MessageDialog(self,0,gtk.MESSAGE_QUESTION,gtk.BUTTONS_NONE,"Deleting files:\n"+mstr)
            dlg.add_buttons("Delete", gtk.RESPONSE_ACCEPT,"Cancel", gtk.RESPONSE_NONE)
            response = dlg.run()
            dlg.destroy()
            if response == gtk.RESPONSE_ACCEPT:
                delete_files(file_links[filename])
                delete_files([filename])
        else:
            log("No selection")

        self.update_selection()
        
    def test_clicked_event(self,widget,data=None):
        vpn_number = self.selection.get_active(0)
        if vpn_number >= 0:
            log("Item selected", vpn_number)
            vpn = self.vpn_manager.vpn_list[vpn_number]
            if vpn.get_status():
                vpn.stop()
            vpn.start(None,None,True)

            dlg = OpenvpnAppletTestConnection(parent=self, vpn_conn=vpn)            
            dlg.run()        
            dlg.destroy()
            vpn.stop()

    def about_clicked_event(self,widget,data=None):
        from about import HeAboutDialog

        response = HeAboutDialog.present(self,
                about_name,
                'openvpn-applet',
                app_version,
                about_text,
                '(c) 2008-2010 ' + about_authors,
                about_website,
                about_bugtracker,
                about_donate)

        if response == HeAboutDialog.RESPONSE_WEBSITE or response == HeAboutDialog.RESPONSE_BUGTRACKER or response == HeAboutDialog.RESPONSE_DONATE:
            self.destroy()


class OpenvpnAppletTestConnection(gtk.Dialog):
    def __init__(self,title="Test Connection", parent=None, flags=0, buttons=None,vpn_conn=None):
        gtk.Dialog.__init__(self,title, parent, flags, buttons)
        self.vpn_conn = vpn_conn        

        self.timer = None
        self.update_func = self.vpn_conn.get_state
        
        self.textbuffer = gtk.TextBuffer()
        self.textview = gtk.TextView(self.textbuffer) 
        self.textview.set_size_request(800, 480)
        self.textview.set_editable(False)
        self.textview.set_cursor_visible(False)
        
        scrolledwindow = gtk.ScrolledWindow()
        scrolledwindow.set_policy(gtk.POLICY_AUTOMATIC,gtk.POLICY_AUTOMATIC)
        scrolledwindow.add(self.textview)
        
        self.vbox.pack_start(scrolledwindow)
                
        save_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        save_button.set_text("Save","")
        save_button.connect("clicked", self.save_button_clicked)
        
        log_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        log_button.set_text("View Log","")
        log_button.connect("clicked", self.log_button_clicked)

        status_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        status_button.set_text("Statistics","")
        status_button.connect("clicked", self.status_button_clicked)

        state_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        state_button.set_text("Status","")
        state_button.connect("clicked", self.state_button_clicked)

        self.action_area.pack_start(log_button)
        self.action_area.pack_start(status_button)
        self.action_area.pack_start(state_button)
        self.action_area.pack_start(save_button)
        self.add_button(gtk.STOCK_CLOSE,gtk.RESPONSE_CLOSE)
        
        self.update_text()
        self.show_all()
        self.timer = gobject.timeout_add(5000, self.update_text)
    
    def update_text(self):
        log("update_text")
        if self.timer == 'stop':
            return False
        if self.vpn_conn != None:
            new_text = self.update_func()
            old_text = self.textbuffer.get_text(self.textbuffer.get_start_iter(),self.textbuffer.get_end_iter(),True)
            if len(new_text) > len(old_text):
                self.textbuffer.insert(self.textbuffer.get_end_iter(), new_text[len(old_text):])
            elif len(new_text) < len(old_text):
                self.textbuffer.set_text(new_text)
                self.textview.scroll_to_iter(self.textbuffer.get_end_iter(),0.0,False,0,0)
        return True
    
    def log_button_clicked(self, widget, data=None):
        self.update_func = self.vpn_conn.get_log
        self.textbuffer.set_text("")
        self.update_text()

    def status_button_clicked(self, widget, data=None):
        self.update_func = self.vpn_conn.get_statistics
        self.textbuffer.set_text("")
        self.update_text()

    def state_button_clicked(self, widget, data=None):
        self.update_func = self.vpn_conn.get_state
        self.textbuffer.set_text("")
        self.update_text()

    def save_button_clicked(self, widget, data=None):
        dlg = hildon.FileChooserDialog(self,gtk.FILE_CHOOSER_ACTION_SAVE,  hildon.FileSystemModel())
        dlg.set_current_folder(os.path.expanduser("~/MyDocs"))
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            filename = dlg.get_filename()
            fd = open(filename,'w+')
            print >>fd,self.textbuffer.get_text(self.textbuffer.get_start_iter(),self.textbuffer.get_end_iter())
            fd.close()
            
        dlg.destroy()
        
    
    def destroy(self):
        self.timer = 'stop'
        super(OpenvpnAppletTestConnection,self).destroy()
    
class OpenvpnAppletNewConnection(gtk.Dialog):
    def __init__(self,title="New Connection", parent=None, flags=0, buttons=None):
        gtk.Dialog.__init__(self,title, parent, flags, buttons)

        self.extra_files = []
        self.browse_button = {}

        self.add_button("Import",gtk.RESPONSE_OK)
        self.add_button("Cancel",gtk.RESPONSE_CANCEL)
        description_keys = ["Configuration File", "Key File", "Cert File", "Ca File", "Secret File", "PKCS12 File"]
        description_values = ["conf", "key", "cert", "ca", "secret", "pkcs12"]
        descriptions = {}
        for i,key in enumerate(description_keys):
            descriptions[key] = description_values[i]
        log(descriptions)
        table = gtk.Table(len(descriptions), 2)
        vbox2 = gtk.VBox()
        pannable_area = hildon.PannableArea()
        for i,d in enumerate(description_keys):
            log(d)
            hbox = gtk.HBox()
            self.browse_button[descriptions[d]] = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL,d, "None")
            self.browse_button[descriptions[d]].set_alignment(0.0,0.5,1,1)
            self.browse_button[descriptions[d]].set_style(hildon.BUTTON_STYLE_PICKER)
            hbox.pack_start(self.browse_button[descriptions[d]])           
            self.browse_button[descriptions[d]].connect("clicked", self.browse_button_event, None)   
            table.attach(hbox,1,2,i,i+1)   
            vbox2.pack_start(table)

        pannable_area.add_with_viewport(vbox2)
        self.vbox.pack_start(pannable_area)
        self.set_size_request(800,480)
        self.vbox.show_all()

    def get_filenames(self):
        files = {}
        for i,key in enumerate(self.browse_button):
            if self.browse_button[key].get_value() != "None":
                fname = self.browse_button[key].get_value()
                perm = '600'
                for ext in ["conf", "ovpn"]:
                    if re.search("\."+ext+"$",fname):
                        perm = '644'
                files[self.browse_button[key].get_value()] = perm
            
            for file in self.extra_files:
                files[file] = '755'
            
        return files
    
    def browse_button_event(self, widget, data=None):
        dlg = hildon.FileChooserDialog(self, gtk.FILE_CHOOSER_ACTION_OPEN, hildon.FileSystemModel())
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            widget.set_value(dlg.get_filename())
            filelinks = FileLinks(dlg.get_filename())
            filelinks.filter(['/home','/media'])
            log("Referenced files in",dlg.get_filename(),filelinks)
            for f in filelinks[dlg.get_filename()]:
                log("Keyword",f,filelinks.get_matchword(f))
                try:
                    self.browse_button[filelinks.get_matchword(f)].set_value(f)
                    log("Keyword file:",filelinks.get_matchword(f),f)
                except KeyError:
                    self.extra_files.append(f)
                    log("Extra file:",f)
                
            if len(filelinks.get_not_found()) > 0:
                log("Files not found:",filelinks.get_not_found())
                mstr = ""
                for f in filelinks.get_not_found():
                    if isinstance(f,str):
                        mstr = mstr+"\n"+f
                mdlg = gtk.MessageDialog(self,0,gtk.MESSAGE_INFO,gtk.BUTTONS_OK,"Files not found:\n"+mstr)
                response = mdlg.run()
                mdlg.destroy() 
        else:
            widget.set_value("None")    
        dlg.destroy()
        

class FileLinks(UserDict):
    def __init__(self, path, filetypes=None):
        UserDict.__init__(self)
        self.path = path
        self.keywords = ["secret","up","down","ca","cert","key","pkcs12","iproute","status","dh","tls-verify","ipchange","client-connect","route-up","client-disconnect","learn-address","auth-user-pass-verify","askpass","auth-user-pass","dh"]
        self.filetypes = filetypes
        self.matchword = {}
        self.files_not_found = []
        if os.path.isdir(path):
            self.type = 'directory'
        else:
            self.type = 'file'
        
        self.update()
        self.filter(['/etc/openvpn/maemo-update-resolvconf'], 'blacklist')
        
    def update(self):
        log("function: update")
        self.data = {}
        self.references = {}
        if self.type == 'directory':
            for root, dirs, files in os.walk(self.path):
                log(root, files)
                for name in files:
                    log("name: ", name)
                    for type in self.filetypes:
                        if re.search(".*\."+type+"$",name):
                            self.data[os.path.join(root, name)] = self.__scan(os.path.join(root, name), root)
        else:
            self.data[self.path] = self.__scan(self.path, os.path.dirname(self.path))
        
    def __scan(self,file,root):
        log("function: __scan", file, root)
        fileset = Set()
        for line in fileinput.input(file):
            parts = line.split()
            #check for keywords
            if len(parts) > 0:
                for word in self.keywords:
                    if parts[0] == word:
                        log("MATCH: ", word)
                        #get filename
                        if len(parts) > 1:
                            filename = parts[1]
                        else:
                            filename = None
                        #check path
                        if filename != None and os.path.exists(filename):
                            #add to return Set
                            fileset.add(filename)
                            self.matchword[filename] = word
                        elif filename !=None and os.path.exists(os.path.join(root, filename)):
                            fileset.add(os.path.join(root, filename))
                            self.matchword[os.path.join(root, filename)] = word
                        elif filename !=None:
                            self.files_not_found.append(filename)
                            filename = None
                    
        #add entries in return Set to reference count
        for f in list(fileset):
            try:
                self.references[f] += 1
            except KeyError:
                self.references[f]  = 1
        return list(fileset)

    def __getitem__(self,item):
        items = self.data[item]
        return_items = []
        for it in items:
            if self.references[it] == 1:
                return_items.append(it)
        return return_items

    def get_matchword(self,file):
        try:
            response = self.matchword[file]     
        except KeyError:
            response = None
        return response
            
    def get_not_found(self):
        if self.files_not_found == [None]:
            self.files_not_found = []
        return self.files_not_found
        
    def filter(self,wordlist,mode='whitelist'):
        log("function: filter", wordlist, mode)
        for key in self.data:
            passed_files = []
            for file in self.data[key]:
                if os.path.isabs(file):
                    for wl in wordlist:
                        if mode == 'whitelist':
                            if re.search('^'+wl,file):
                                log("Passed:", file)
                                passed_files.append(file)
                            else:
                                log("Dropped:", file)
                        else: #'blacklist'
                            if not re.search('^'+wl,file):
                                passed_files.append(file)
                                log("Passed:", file)
                            else:
                                log("Dropped:", file)
            self.data[key] = passed_files
                        
    
hd_plugin_type = openvpn_applet

def log(*input):
    if enable_logging:
        print input
        print >>fd,input
        fd.flush()

# The code below is just for testing purposes.
# It allows to run the widget as a standalone process.
if __name__ == "__main__":
    import gobject
    gobject.type_register(hd_plugin_type)
    obj = gobject.new(hd_plugin_type, plugin_id="plugin_id")
    obj.show_all()
    gtk.main()


#if __name__ == '__main__':
#    vc = vpn_control()
#    print vc
#    sel = int(raw_input("command: "))-1
#    if vc.status(sel) != 0:
#        vc.start(sel)
#    else:
#        vc.stop(sel)
