'''
Advanced Power Abstract Module and its subclasses
2008-2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

from advpowcommon.util.common import *
from advpowcommon.util.execn import *
from advpowcommon.util.meminfo import *

import time, os.path

class AbstractModule():

  def __init__(self):
    self.__name = self.__class__.__name__
    self.__available = True
    self.interval = 1000
    self.value = None
    self.async = False
    self.pre_init()
    self.__available = self.__available and self.check_available()
    if self.__available:
      self.update()

  def pre_init(self):
    raise NotImplementedError("AbstractModule.pre_init() must be implemented")

  def check_available(self):
    raise NotImplementedError("AbstractModule.check_available() must be implemented")

  def update(self):
    raise NotImplementedError("AbstractModule.update() must be implemented")

  def process(self):
    raise NotImplementedError("AbstractModule.process() must be implemented")

  def is_available(self):
    return self.__available

  def get_name(self):
    return self.__name

  def get_updated(self):
    self.update()
    return self.value

class AModule(AbstractModule):

  def __init__(self):
    AbstractModule.__init__(self)

  def check_available(self):
    return True

  def update(self):
    self.value = self.process()

class AMWithFileSource(AbstractModule):

  def __init__(self):
    self.file = ''
    AbstractModule.__init__(self)

  def check_available(self):
    return os.path.exists(self.file) and os.path.isfile(self.file)

  def update(self):
    lines = get_file_lines(self.file)
    if lines:
      self.value = self.process(lines)

class AMWithShellSource(AbstractModule):

  def __init__(self):
    self.command = ''
    AbstractModule.__init__(self)

  def check_available(self):
    return True

  def update(self):
    lines = get_shell_command_output(self.command).rsplit('\n')
    lines = [l for l in lines if l]
    if lines:
      self.value = self.process(lines)

class AMWithCommonFileSource(AMWithFileSource):

  last_update = {}
  all_fields = set()

  def __init__(self):
    self.min_interval = 5
    self.file_content = []
    AMWithFileSource.__init__(self)

  def pre_init(self):
    self.fields_free = []
    self.fields_total = []
    self.pre_init_cfs()
    self.register()

  def pre_init_cfs(self):
    raise NotImplementedError("AMWithCommonFileSource.pre_init_cfs() must be implemented")

  def update(self):
    ctime = time.time()
    if ctime - self.last_update[self.file]['time'] >= self.min_interval:
      self.last_update[self.file]['time'] = ctime
      lines = get_file_lines(self.file)
      self.last_update[self.file]['content'] = [l for l in lines if l.split(':')[0] in self.all_fields]
    if self.last_update[self.file]['content']:
      self.value = self.process(self.last_update[self.file]['content'])

  def register(self):
    if not self.file in self.last_update:
      self.last_update[self.file] = {'time': 0, 'content': []}
    self.all_fields.update(self.fields_free)
    self.all_fields.update(self.fields_total)
    self.last_update[self.file]['time'] = 0
    self.update()

class AMMeminfo(AMWithCommonFileSource):

  def __init__(self):
    AMWithCommonFileSource.__init__(self)

  def pre_init_cfs(self):
    self.file = '/proc/meminfo'
    self.value = 0
    self.pre_init_meminfo()

  def pre_init_meminfo(self):
    raise NotImplementedError("AMMeminfo.pre_init_meminfo() must be implemented")

  def process(self, lines):
    free_value = process_meminfo(lines, self.fields_free)
    total_value = process_meminfo(lines, self.fields_total)
    used_value = total_value - free_value
    return [free_value, used_value, total_value]
