#include "QvernoteView.h"
#include "QvernoteAPI.h"
#include "QNewNotebookDialog.h"
#include "QNotebookListItemWidget.h"
#include "QvernoteTypes.h"
#include "QvernoteSettings.h"
#include "qvernotelogindialog.h"
#include "QNoteSearchDialog.h"
#include "QDeleteNotebookView.h"
#include <QGridLayout>
#include <QMessageBox>
#include <QDesktopWidget>
#include <QDebug>

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

using namespace qvernote;
using namespace qvernote::api;

QvernoteView::QvernoteView(shared_ptr<QvernoteAPI> hEvernote, QWidget *parent)
    : QvernoteWindow(parent), m_hEvernote(hEvernote)
{
	QvernoteSettings* settings = QvernoteSettings::Instance();
	ui.setupUi(this);

	ui.actionUse_SSL->setChecked(settings->getUseSsl());

	QvernoteSettings::DisplayOrientation orientation = settings->getDisplayOrientation();

	QActionGroup* displaySetupGroup = new QActionGroup(this);
	displaySetupGroup->setExclusive(true);

	QAction* actDoPortrait = new QAction(tr("Portrait"), displaySetupGroup);
	actDoPortrait->setCheckable(true);
	actDoPortrait->setChecked(orientation == QvernoteSettings::DO_PORTRAIT);
	QAction* actDoLandscape = new QAction(tr("Landscape"), displaySetupGroup);
	actDoLandscape->setCheckable(true);
	actDoLandscape->setChecked(orientation == QvernoteSettings::DO_LANDSCAPE);
	QAction* actDoAuto = new QAction(tr("Auto"), displaySetupGroup);
	actDoAuto->setCheckable(true);
	actDoAuto->setChecked(orientation == QvernoteSettings::DO_AUTO);

	ui.menubar->addActions(displaySetupGroup->actions());

	QObject::connect(ui.lvNotebookList, SIGNAL(itemClicked(QListWidgetItem *)),  this, SLOT(loadNotebook(QListWidgetItem* )));
	QObject::connect(ui.pbNewNotebook, SIGNAL(clicked()), this, SLOT(createNewNotebook()));
	QObject::connect(ui.actionReload, SIGNAL(triggered(bool)), this, SLOT(reloadNotebookList()));
	QObject::connect(ui.actionUse_SSL, SIGNAL(toggled(bool)), this, SLOT(configureSslProtocol(bool)));
	QObject::connect(ui.actionAccount, SIGNAL(triggered(bool)), this, SLOT(accountConfig()));
	QObject::connect(ui.actionSearch, SIGNAL(triggered(bool)), this, SLOT(openSearchDialog()));
	QObject::connect(ui.actionDelete, SIGNAL(triggered(bool)), this, SLOT(openDeleteNotebookWindow()));

	QObject::connect(actDoPortrait, SIGNAL(toggled(bool)), this, SLOT(setPortraitOrientation(bool)));
	QObject::connect(actDoLandscape, SIGNAL(toggled(bool)), this, SLOT(setLandscapeOrientation(bool)));
	QObject::connect(actDoAuto, SIGNAL(toggled(bool)), this, SLOT(setAutoOrientation(bool)));

	setProgressIndicator(true);
	displayInformationBox("Initializing notebook store. Please wait...");
}

QvernoteView::~QvernoteView()
{

}

void QvernoteView::initView()
{
	m_pBackgroundJob->start();
	displayInformationBox("Loading notebooks. Please wait...");
}

void QvernoteView::loadNotebookList()
{
	Notebook notebook;
	bool hasNextNotebook = false;
	int noteCount = 0;
	setProgressIndicator(true);

	ui.lvNotebookList->clear();
	ui.lvNotebookList->setVisible(false);

	for(hasNextNotebook = m_hEvernote->getFirstNotebook(notebook); hasNextNotebook; hasNextNotebook = m_hEvernote->getNextNotebook(notebook))
	{
		//Notebook* newNotebook = new Notebook(notebook);
		QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
		iw->setSizeHint(QSize(0, 70));
		noteCount += m_hEvernote->getNotebookContentsCount(notebook);
		ui.lvNotebookList->setItemWidget(iw, new QNotebookListItemWidget(notebook, m_hEvernote, -1, false, ui.lvNotebookList));

	}

	/* Add special "ALL" notebook */
	Notebook specialAllNotebook;
	specialAllNotebook.name = "All notebooks";
	QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
	iw->setSizeHint(QSize(0, 70));
	ui.lvNotebookList->setItemWidget(iw,
			new QNotebookListItemWidget(specialAllNotebook,
					m_hEvernote,
					noteCount,
					false,
					ui.lvNotebookList));

	ui.lvNotebookList->setVisible(true);
	setProgressIndicator(false);
}

void QvernoteView::loadNotebook(QListWidgetItem* p_notebookListItem)
{
	QNotebookListItemWidget* p = (QNotebookListItemWidget*)p_notebookListItem->listWidget()->itemWidget(p_notebookListItem);
	qDebug() << "Loading notebook " << p->getNotebook().name.c_str();

	m_pNotesView = new QNotesView(m_hEvernote, p->getNotebook(), this);
	m_pNotesView->show();
	//QTimer *timer = new QTimer( this );
	//connect( timer, SIGNAL(timeout()), m_pNotesView, SLOT(reloadNotes()) );
	//timer->start( 1000 );
	//QTimer::singleShot(500, m_pNotesView, SLOT(reloadNotes()));
	//m_pNotesView->reloadNotes();
}

void QvernoteView::createNewNotebook()
{
	QNewNotebookDialog* pDlg = new QNewNotebookDialog(m_hEvernote, this);
	pDlg->show();
}

void QvernoteView::reloadNotebookList()
{
	initView();
}

void QvernoteView::deleteSelectedNotebook(QNotebookListItemWidget* notebookWidget)
{
	if(m_hEvernote->deleteNotebook(notebookWidget->getNotebook()) == false)
	{
		displayError("Failed to delete notebook", QString(m_hEvernote->getLastErrorString().c_str()));
	}
}

void QvernoteView::configureSslProtocol(bool checked)
{
	QvernoteSettings::Instance()->setUseSsl(checked);
	setProgressIndicator(true);
	m_hEvernote->reinitNoteStore();
	reloadNotebookList();
}

void QvernoteView::accountConfig()
{
	QvernoteLoginDialog ld;
	QvernoteSettings* settings = QvernoteSettings::Instance();

	if(ld.exec() == QDialog::Rejected)
		return;

	while(true)
	{
		if(m_hEvernote->Authenticate(
							settings->getUsername().toStdString(),
							settings->getPassword().toStdString()))
		{
			break;;
		}

		if(ld.exec() == QDialog::Rejected)
			return;

	}

	if(m_hEvernote->initNoteStore() == true)
	{
		reloadNotebookList();
	}
}

void QvernoteView::openSearchDialog()
{
	QNoteSearchDialog* pDlg = new QNoteSearchDialog(this);
	pDlg->show();
}

void QvernoteView::searchNote(QString searchString)
{
	qDebug() << "Searching" << searchString;
	setProgressIndicator(true);

	if(m_hEvernote->searchNotes(searchString.toStdString(), 10) == false)
	{
		displayError("Note search failed", QString(m_hEvernote->getLastErrorString().c_str()));
		return;
	}

	setProgressIndicator(false);

	m_pNotesView = new QNotesView(m_hEvernote, "Search results for " + searchString, this);
	m_pNotesView->show();
}

void QvernoteView::orientationChanged()
{
	qDebug() << "Orientation changed";

	 QRect screenGeometry = QApplication::desktop()->screenGeometry();
	 if (screenGeometry.width() > screenGeometry.height())
	 {
		 // Landscape

	 }
	 else
	 {
		 // Portrait
	 }
}

bool QvernoteView::queryNotebooks()
{
	return m_hEvernote->loadNotebookList();
}

void QvernoteView::openDeleteNotebookWindow()
{
	QDeleteNotebookView* pWnd = new QDeleteNotebookView(m_hEvernote, this);
	pWnd->show();
}
