/*
 * QvernoteAPI.cpp
 *
 *  Created on: May 12, 2010
 *      Author: alexr
 */


#include "QvernoteAPI.h"
#include <QDebug>
#include <transport/THttpClient.h>
#include <transport/TSSLSocket.h>
#include <protocol/TBinaryProtocol.h>
#include <Thrift.h>
#include <boost/shared_ptr.hpp>
#include "QvernoteSettings.h"

using namespace std;
using namespace boost;
using namespace qvernote;
using namespace qvernote::api;

QvernoteAPI::QvernoteAPI() {
	try {
		initUserStore();
	} catch(TException& e) {
		throw e;
	}
}

QvernoteAPI::~QvernoteAPI() {
	// TODO Auto-generated destructor stub
}

void QvernoteAPI::initUserStore() {
	shared_ptr<TSSLSocketFactory> sslSocketFactory(new TSSLSocketFactory());
	shared_ptr<TSocket> sslSocket = sslSocketFactory->createSocket(EVERNOTE_HOST, 443);
	shared_ptr<TBufferedTransport> bufferedTransport(new TBufferedTransport(sslSocket));
	userStoreHttpClient = shared_ptr<THttpClient>(new THttpClient(bufferedTransport, EVERNOTE_HOST, EDAM_USER_STORE_PATH));

	//shared_ptr<THttpClient> userStoreHttpClient(new THttpClient(EVERNOTE_HOST, 80, EDAM_USER_STORE_PATH));

	userStoreHttpClient->open();
	shared_ptr<TProtocol> iprot(new TBinaryProtocol(userStoreHttpClient));
	m_UserStoreClient = shared_ptr<UserStoreClient>(new UserStoreClient(iprot));
}

void QvernoteAPI::reInitUserStore() {
	if(userStoreHttpClient)
		userStoreHttpClient->close();
	initUserStore();
}

bool QvernoteAPI::initNoteStore() {
	string noteStorePath = string(EDAM_NOTE_STORE_PATH) + string("/") + getShardId();
	//shared_ptr<TTransport> noteStoreHttpClient;
	QvernoteSettings* settings = QvernoteSettings::Instance();

	try {
		if(settings->getUseSsl() == true)
		{
			shared_ptr<TSSLSocketFactory> sslSocketFactory(new TSSLSocketFactory());
			shared_ptr<TSocket> sslSocket = sslSocketFactory->createSocket(EVERNOTE_HOST, 443);
			shared_ptr<TBufferedTransport> bufferedTransport(new TBufferedTransport(sslSocket));
			noteStoreHttpClient = shared_ptr<TTransport>(new THttpClient(bufferedTransport, EVERNOTE_HOST, noteStorePath));
		}
		else
		{
			noteStoreHttpClient = shared_ptr<TTransport>(new THttpClient(EVERNOTE_HOST, 80, noteStorePath));
		}

		noteStoreHttpClient->open();
		shared_ptr<TProtocol> noteStoreProtocol(new TBinaryProtocol(noteStoreHttpClient));
		m_NoteStoreClient = shared_ptr<NoteStoreClient>(new NoteStoreClient(noteStoreProtocol));
	} catch(TTransportException& e) {
		qDebug() << e.what();
		setError(e.what(), e.getType());
		return false;
	}

	//loadNotebookList();
	return true;
}

bool QvernoteAPI::reinitNoteStore() {
	//m_NotebookList.clear();
	if(noteStoreHttpClient)
		noteStoreHttpClient->close();
	return initNoteStore();
}

bool QvernoteAPI::checkVersion() {
	UserStoreConstants usc;

	return m_UserStoreClient->checkVersion(EDAM_CLIENT_NAME, usc.EDAM_VERSION_MAJOR, usc.EDAM_VERSION_MINOR);
}

bool QvernoteAPI::Authenticate(const string& userName, const string& password) {
	m_AuthenticationResult = shared_ptr<AuthenticationResult>(new AuthenticationResult());

	try {
		reInitUserStore();
		m_UserStoreClient->authenticate(
				*m_AuthenticationResult,
				userName,
				password,
				EDAM_CONSUMER_KEY,
				EDAM_CONSUMER_SECRET);
	} catch(EDAMUserException& e) {
		qDebug() << e.parameter.c_str();
		setError(e.parameter, e.errorCode);
		return false;
	} catch(TTransportException& te) {
		qDebug() << __FUNCTION__ << te.what();
		setError(te.what(), te.getType());
		return false;
	}

	return true;
}

bool	QvernoteAPI::loadNotes(int maxNotes, const Notebook& notebook) {
	NoteFilter filter;

	if(notebook.guid.size() > 0)
	{
		filter.notebookGuid = notebook.guid;
		filter.__isset.notebookGuid = true;
	}
	try {
		reinitNoteStore();
		m_NoteStoreClient->findNotes(m_NoteList, getAuthenticationToken(), filter, 0, maxNotes);
	} catch(EDAMUserException& e) {
		qDebug() << e.what();
		setError(e.parameter, e.errorCode);
		return false;
	} catch(TTransportException& te) {
		setError(te.what(), te.getType());
		return false;
	}

	return true;
}

bool	QvernoteAPI::getFirstNote(Note& note) {
	if(m_NoteList.notes.size() == 0)
		return false;

	m_NoteIterator = m_NoteList.notes.begin();

	if((*m_NoteIterator).content.size() != 0)
	{
		note = (*m_NoteIterator);
		return true;
	}

	try {
		reinitNoteStore();
		m_NoteStoreClient->getNote(/*note*/(*m_NoteIterator), getAuthenticationToken(), (*m_NoteIterator).guid, true, true, false, false);
		reinitNoteStore();
		m_NoteStoreClient->getNoteTagNames(/*note.tagNames*/(*m_NoteIterator).tagNames, getAuthenticationToken(), (*m_NoteIterator).guid);
	} catch(EDAMUserException& e) {
		setError(e.parameter, e.errorCode);
		return false;
	}

	note = (*m_NoteIterator);
	return true;
}

bool	QvernoteAPI::getNextNote(Note& note) {
	m_NoteIterator++;

	if(m_NoteIterator != m_NoteList.notes.end()) {
		if((*m_NoteIterator).content.size() != 0)
		{
			note = (*m_NoteIterator);
			return true;
		}

		try {
			reinitNoteStore();
			m_NoteStoreClient->getNote(/*note*/(*m_NoteIterator), getAuthenticationToken(), (*m_NoteIterator).guid, true, true, false, false);
			reinitNoteStore();
			m_NoteStoreClient->getNoteTagNames(/*note.tagNames*/(*m_NoteIterator).tagNames, getAuthenticationToken(), (*m_NoteIterator).guid);
		} catch(EDAMUserException& e) {
			setError(e.parameter, e.errorCode);
			return false;
		}

		note = (*m_NoteIterator);
		return true;
	}

	return false;
}

bool	QvernoteAPI::createNewNote(Note& newNote) {
	Note createdNote;
	newNote.__isset.notebookGuid = true;
	newNote.__isset.title = true;
	newNote.__isset.content = true;
	newNote.__isset.tagGuids = true;

	try {
		reinitNoteStore();
		m_NoteStoreClient->createNote(createdNote, getAuthenticationToken(), newNote);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}

	m_NoteList.notes.insert(m_NoteList.notes.begin(), createdNote);
	return true;
}

bool 	QvernoteAPI::updateExistingNote(Note& existingNote) {
	Note updatedNote;
	existingNote.__isset.guid = true;
	existingNote.__isset.title = true;
	existingNote.__isset.content = true;
	existingNote.__isset.tagGuids = true;

	try {
		reinitNoteStore();
		m_NoteStoreClient->updateNote(updatedNote, getAuthenticationToken(), existingNote);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}

	vector<Note>::iterator pos;
	if(searchNoteList(pos, existingNote.guid)) {
		m_NoteList.notes.erase(pos);
		m_NoteList.notes.insert(pos, updatedNote);
	}
	return true;
}

bool	QvernoteAPI::deleteNote(Note& existingNote) {
	try {
		reinitNoteStore();
		m_NoteStoreClient->deleteNote(getAuthenticationToken(), existingNote.guid);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}

	vector<Note>::iterator pos;
	if(searchNoteList(pos, existingNote.guid)) {
		m_NoteList.notes.erase(pos);
	}
	return true;
}

bool 	QvernoteAPI::searchNotes(const string& searchString, int maxResults) {
	NoteFilter filter;
	filter.words = searchString;
	filter.__isset.words = true;
	try {
		reinitNoteStore();
		m_NoteStoreClient->findNotes(m_NoteList, getAuthenticationToken(), filter, 0, maxResults);
	} catch(EDAMUserException& e) {
		qDebug() << e.what();
		setError(e.parameter, e.errorCode);
		return false;
	}

	return true;
}

bool	QvernoteAPI::searchNoteList(vector<Note>::iterator& pos, Guid noteGuid) {
	for(pos = m_NoteList.notes.begin(); pos < m_NoteList.notes.end(); pos++) {
		if((*pos).guid == noteGuid)
			return true;
	}

	return false;
}

string QvernoteAPI::getDefaultNotebookName() {
	vector<Notebook>::iterator it;

	for(it = m_NotebookList.begin(); it < m_NotebookList.end(); it++) {
		if((*it).defaultNotebook == true) {
			return (*it).name;
		}
	}

	return "";
}

bool QvernoteAPI::loadNotebookList() {
	qDebug() << "Getting list of notebooks";
	NoteFilter filter;
	m_NotebookList.clear();
	m_NoteCounts.notebookCounts.clear();
	m_NoteCounts.tagCounts.clear();
	m_TagsPerNotebook.clear();

	try {
		reinitNoteStore();
		m_NoteStoreClient->listNotebooks(m_NotebookList, getAuthenticationToken());
		reinitNoteStore();
		m_NoteStoreClient->findNoteCounts(m_NoteCounts, getAuthenticationToken(), filter, true);

	} catch(EDAMUserException& e) {
		setError(e.parameter, e.errorCode);
		return false;
	} catch(TTransportException& te) {
		setError(te.what(), te.getType());
		return false;
	}


	return true;
}

bool QvernoteAPI::getNotebookByName(Notebook& notebook, string notebookName) {
	vector<Notebook>::iterator it;

	for(it = m_NotebookList.begin(); it < m_NotebookList.end(); it++) {
		if((*it).name == notebookName) {
			notebook = (*it);
			return true;
		}
	}

	return false;
}

bool QvernoteAPI::getFirstNotebook(Notebook& notebook) {
	if(m_NotebookList.size() == 0)
		return false;

	m_NotebookIterator = m_NotebookList.begin();

	notebook = (*m_NotebookIterator);

	if(m_TagsPerNotebook[notebook.name].empty())
	{
		try {
			reinitNoteStore();
			m_NoteStoreClient->listTagsByNotebook(m_TagsPerNotebook[notebook.name], getAuthenticationToken(), notebook.guid);
		} catch(EDAMUserException& e) {
			setError(e.parameter, e.errorCode);
			return false;
		}
	}

	return true;
}

bool QvernoteAPI::getNextNotebook(Notebook& notebook) {
	m_NotebookIterator++;
	if(m_NotebookIterator != m_NotebookList.end()) {
		notebook = (*m_NotebookIterator);

		if(m_TagsPerNotebook[notebook.name].empty())
		{
			try {
				reinitNoteStore();
				m_NoteStoreClient->listTagsByNotebook(m_TagsPerNotebook[notebook.name], getAuthenticationToken(), notebook.guid);
			} catch(EDAMUserException& e) {
				setError(e.parameter, e.errorCode);
				return false;
			}
		}

		return true;
	}

	return false;
}

bool QvernoteAPI::createNewNotebook(const string& notebookName, bool isDefault, bool isPublished) {
	Notebook newNotebook;
	Notebook result;
	newNotebook.name = notebookName;
	newNotebook.defaultNotebook = isDefault;
	newNotebook.published = isPublished;
	newNotebook.__isset.name = true;
	try {
		reinitNoteStore();
		m_NoteStoreClient->createNotebook(result, getAuthenticationToken(), newNotebook);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}


	return true;
}

bool QvernoteAPI::deleteNotebook(Notebook& notebook) {
	try {
		reinitNoteStore();
		m_NoteStoreClient->expungeNotebook(getAuthenticationToken(), notebook.guid);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}


	return true;
}

int	 QvernoteAPI::getNotebookCount() {
	return m_NotebookList.size();
}

int  QvernoteAPI::getNotebookContentsCount(Notebook& notebook) {
	return m_NoteCounts.notebookCounts[notebook.guid];
}

int QvernoteAPI::getTrashCount() {
	return m_NoteCounts.trashCount;
}

bool QvernoteAPI::getFirstNotebookTag(const string& notebookName, Tag& tag) {
	if(m_TagsPerNotebook[notebookName].empty())
		return false;

	m_TagsPerNotebookIterator = m_TagsPerNotebook[notebookName].begin();

	tag = (*m_TagsPerNotebookIterator);
	return true;
}

bool QvernoteAPI::getNextNotebookTag(const string& notebookName, Tag& tag) {
	m_TagsPerNotebookIterator++;

	if(m_TagsPerNotebookIterator != m_TagsPerNotebook[notebookName].end()) {
		tag = (*m_TagsPerNotebookIterator);
		return true;
	}

	return false;
}

bool QvernoteAPI::addNewTag(const string& newTagName, Tag& result) {
	Tag newTag;

	newTag.name = newTagName;
	newTag.__isset.name = true;

	try {
		reinitNoteStore();
		m_NoteStoreClient->createTag(result, getAuthenticationToken(), newTag);
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}

	m_TagList.push_back(result);

	return true;
}

bool QvernoteAPI::loadTagList() {
	try {
		reinitNoteStore();
		m_NoteStoreClient->listTags(m_TagList, getAuthenticationToken());
	} catch(EDAMUserException& e) {
		qDebug() << e.errorCode << QString(((string)e.parameter).c_str());
		setError(e.parameter, e.errorCode);
		return false;
	}

	return true;
}

bool QvernoteAPI::getFirstTag(Tag& tag) {
	m_TagIterator = m_TagList.begin();

	tag = *m_TagIterator;
	m_TagIterator++;
	return true;
}

bool QvernoteAPI::getNextTag(Tag& tag) {
	if(m_TagIterator == m_TagList.end())
		return false;

	tag = *m_TagIterator;
	m_TagIterator++;
	return true;
}

bool QvernoteAPI::addExistingTag(const Tag& tag, Note& note) {
	note.tagGuids.push_back(tag.guid);
	note.tagNames.push_back(tag.name);
	return true;
}

bool QvernoteAPI::removeExistingTag(const Tag& tag, Note& note) {
	vector<Guid>::iterator itGuid = find(note.tagGuids.begin(), note.tagGuids.end(), tag.guid);
	if(itGuid != note.tagGuids.end())
		note.tagGuids.erase(itGuid);
	else
		return false;

	vector<string>::iterator itStr = find(note.tagNames.begin(), note.tagNames.end(), tag.name);
	if(itStr != note.tagNames.end())
		note.tagNames.erase(itStr);
	else
		return false;

	return true;
}

