#include "ytvparser.h"
#include "location.h"
#include <QXmlStreamAttributes>
#include <QDebug>

ytvParser::ytvParser()
{
}


bool ytvParser::start()
{

    while (!atEnd()) {
        readNext();

        if (isStartElement()) {
            if (name() == "MTRXML" && attributes().value("version") == "1.0")
                readMTRXML();
            else
                raiseError(QObject::tr("Information in invalid format"));
        }
    }

    clear();
    return !error();
}

void ytvParser::readUnknownElement()
{ //If compiled with 4.6 it can be removed ;)
    Q_ASSERT(isStartElement());

//    qDebug() << "UnKnown!!" << name();

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement())
            readUnknownElement();
    }
}

void ytvParser::readMTRXML() {

    Q_ASSERT(isStartElement() && name() == "MTRXML");

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "GEOCODE")
                readGeoCode();
            else if (name() == "REVERSE")
                readReverse();
            else if (name() == "ROUTE")
                readRoute();
            else
                readUnknownElement();
        }
    }

}


void ytvParser::readGeoCode() {

    Q_ASSERT(isStartElement() && name() == "GEOCODE");

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "LOC")
                readLocation();
            else
                readUnknownElement();
        }
    }
}

void ytvParser::readReverse() {

    Q_ASSERT(isStartElement() && name() == "REVERSE");

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "LOC")
                readLocation();
            else
                readUnknownElement();
        }
    }
}

Route ytvParser::readRoute() {

    Q_ASSERT(isStartElement() && name() == "ROUTE");

    qDebug() << "!!ROUTE";

    Route reading;

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "LENGTH") {
                QTime time = readLengthTime();
                reading.setTime(time);

            }  else if (name() == "WALK") {
                reading.addTransport( readWalk() );

            } else if (name() == "LINE") {
                reading.addTransport( readLine() );

            } else
                readUnknownElement();
        }
    }

    routeList.append(reading);
    return reading;
}

void ytvParser::readLocation() {

    Q_ASSERT(isStartElement() && name() == "LOC");

    QXmlStreamAttributes att = attributes();

    QString name = att.value( "name1" ).toString();
    int number = att.value( "number" ).toString().toInt();
    QString city = att.value( "city" ).toString();
    int code = att.value( "code" ).toString().toInt();
    QString address = att.value( "address" ).toString();
    int type = att.value( "type" ).toString().toInt();
    QString category = att.value( "category" ).toString();

    int x = att.value( "x" ).toString().toInt();
    int y = att.value( "y" ).toString().toInt();
    float longitude = att.value( "long" ).toString().toFloat();
    float lattitude = att.value( "lat" ).toString().toFloat();

    Location loc(name, number, city, code, address, type, category, x, y, longitude, lattitude);

    locationList.push_back(loc);

//    qDebug() << "Location" << name;
    readNext();
}

QTime ytvParser::readLengthTime() {

    Q_ASSERT(isStartElement() && name() == "LENGTH");

    QXmlStreamAttributes att = attributes();

    float minutes = att.value( "time" ).toString().toFloat();

//    qDebug() << "Lenght" << minutes;

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            readUnknownElement();
        }
    }

    QTime ret;
    ret = ret.addSecs(qRound(minutes)*60);
//    qDebug() << ret << minutes;
    return ret;
}

int ytvParser::readLengthDistance() {

    Q_ASSERT(isStartElement() && name() == "LENGTH");

    QXmlStreamAttributes att = attributes();

    float distance = att.value( "dist" ).toString().toFloat();

//    qDebug() << "Lenght" << minutes;

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            readUnknownElement();
        }
    }

    return distance;
}

Stop ytvParser::readPoint() {

    Q_ASSERT(isStartElement() && name() == "POINT");

    QXmlStreamAttributes att = attributes();

    QString pointName = att.value( "uid" ).toString();
    int x = att.value( "x" ).toString().toInt();
    int y = att.value( "y" ).toString().toInt();

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "ARRIVAL") {
                QDateTime time = readArrival();
                //Currently we dont use it
            } else if (name() == "DEPARTURE") {
                readUnknownElement();

            } else
                readUnknownElement();
        }
    }


//    qDebug() << "Point" << pointName;

    return Stop(pointName,QDateTime(),QDateTime(),x,y);
}

Stop ytvParser::readStop() {

    Q_ASSERT(isStartElement() && name() == "STOP");

    QXmlStreamAttributes att = attributes();

    int x = att.value( "x" ).toString().toInt();
    int y = att.value( "y" ).toString().toInt();
    QDateTime startTime;
    QDateTime endTime;
    QString stopName;

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "ARRIVAL") {
                startTime = readArrival();
            } else if (name() == "NAME") {
                QString temp = readName();
                if (! temp.isNull() )
                    stopName = temp; //Only the finnish name

            } else if (name() == "DEPARTURE") {
                endTime = readDeparture();
            } else
                readUnknownElement();
        }
    }


//    qDebug() << "Stop" << stopName;

    return Stop(stopName, startTime, endTime, x, y);
}

Transport ytvParser::readWalk() {

    Q_ASSERT(isStartElement() && name() == "WALK");

    Transport reading;

    reading.setType(Transport::Walk);

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "LENGTH") {
                int distance = readLengthDistance();
                reading.setDistance( distance );
                //Currently we dont use it
            } else if (name() == "POINT") {
                readPoint();

            } else if (name() == "STOP") {
                reading.addStop( readStop() );

            }  else
                readUnknownElement();
        }
    }

//    qDebug() << "Walk";

    return reading;
}

Transport ytvParser::readLine() {

    Q_ASSERT(isStartElement() && name() == "LINE");

    QXmlStreamAttributes att = attributes();

    QString code = att.value( "code" ).toString();
    int type = att.value( "type" ).toString().toInt();

    code.remove(0,1); //Don't want the first char
    if (code.at(0) == '0')
        code.remove(0,1);

    QString line = code.section(' ',0,0); //We get the real code of the line

    Transport::TransportType transportType;

    switch (type) {  //This codes are from YTV
        case 1:
        case 3:
        case 4:
        case 5:
        case 8:
        case 9:
        case 11:
        case 21:
        case 23:
        case 24:
            transportType = Transport::Bus;
            break;
        case 2:
            transportType = Transport::Tram;
            if (code.at(0) == '0')
                code.remove(0,1);
            break;
        case 7:
            transportType = Transport::Ferry;
            break;
        case 12:
        case 13:
            transportType = Transport::Train;
            line.remove(0,2);
            break;
        case 22:
        case 25:
            transportType = Transport::Night;
            break;
        case 6:
            transportType = Transport::Metro;
            break;
        default:
            transportType = Transport::Bus;
        }



    Transport reading(line, transportType);

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            if (name() == "LENGTH") {
                QTime time = readLengthTime();
                //Currently we dont use it

            } else if (name() == "STOP") {
                reading.addStop( readStop() );

            }  else
                readUnknownElement();
        }
    }

//    qDebug() << "Line" << line;

    return reading;

}


QString ytvParser::readName() {

    Q_ASSERT(isStartElement() && name() == "NAME");

    QXmlStreamAttributes att = attributes();

    QString name = att.value( "val" ).toString();
    int lang = att.value( "lang" ).toString().toInt();

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            readUnknownElement();
        }
    }

//    qDebug() << "Name" << name;

    if (lang == 1) { //Return only the Finnish name
        return name;
    }
    return QString();


}

QDateTime ytvParser::readArrival() {

    Q_ASSERT(isStartElement() && name() == "ARRIVAL");

    QXmlStreamAttributes att = attributes();

    QString date = att.value( "date" ).toString();
    QString time = att.value( "time" ).toString();

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            readUnknownElement();
        }
    }

    return QDateTime::fromString(date + time, "yyyyMMddhhmm");
}

QDateTime ytvParser::readDeparture() {

    Q_ASSERT(isStartElement() && name() == "DEPARTURE");

    QXmlStreamAttributes att = attributes();

    QString date = att.value( "date" ).toString();
    QString time = att.value( "time" ).toString();

    while (!atEnd()) {
        readNext();

        if (isEndElement())
            break;

        if (isStartElement()) {
            readUnknownElement();
        }
    }

    return QDateTime::fromString(date + time, "yyyyMMddhhmm");
}
