/*
 * Copyright (C) 2010 Kamilla Bremeraunet <kamilla@bremeraunet.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "desktopwidget.h"
#include "ui_desktopwidget.h"

#include <QDebug>
#include <QPainter>
#include <QPaintEvent>
#include <QPalette>
#include <QIcon>

#include <QX11Info>
#include <X11/Xlib.h>
#include <X11/X.h>
#include <X11/Xatom.h>

#include "facebrick.h"
#include "fberror.h"

#include "mainwindow.h"
#include "newsfeedpostview.h"
#include "newsfeed.h"
#include "newsfeeddelegate.h"
#include "newsfeedmodel.h"

DesktopWidget::DesktopWidget(QWidget *parent) :
    QWidget(parent),
    m_ui(new Ui::DesktopWidget)
{
    m_ui->setupUi(this);

    m_ui->refreshButton->setIcon(QIcon::fromTheme("general_refresh"));
    m_ui->upButton->setIcon(QIcon::fromTheme("keyboard_move_up"));
    m_ui->downButton->setIcon(QIcon::fromTheme("keyboard_move_down"));

    m_ui->postsListView->setStyleSheet("background-color: rgba( 255, 255, 255, 0% );");

    setAttribute(Qt::WA_TranslucentBackground);
    setAttribute(Qt::WA_X11NetWmWindowTypeDialog);

    setWindowFlags(windowFlags() | Qt::Dialog);

    Atom window_type = XInternAtom (QX11Info::display(), "_NET_WM_WINDOW_TYPE", False);
    Atom hildonwinType = XInternAtom (QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", False);
    XChangeProperty(QX11Info::display(), winId(), window_type, XA_ATOM, 32,
                    PropModeReplace, (unsigned char *) &hildonwinType, 1);

    m_ui->postsListView->setModel(FaceBrick::instance()->m_newsFeedModel);

    NewsFeedDelegate *delegate = new NewsFeedDelegate(this);
    delegate->setWidgetFontSize(-4);
    m_ui->postsListView->setItemDelegate(delegate);

    connect(m_ui->downButton, SIGNAL(clicked()), this, SLOT(onDownButtonClicked()));
    connect(m_ui->upButton, SIGNAL(clicked()), this, SLOT(onUpButtonClicked()));
    connect(m_ui->refreshButton, SIGNAL(clicked()), this, SLOT(onRefreshButtonClicked()));

    connect(NewsFeed::instance(), SIGNAL(newsFeedLoadingErrorSignal()), this, SLOT(newsFeedRefreshError()));
    connect(NewsFeed::instance(), SIGNAL(newsFeedLoaded()), this, SLOT(newsFeedLoaded()));
    connect(NewsFeed::instance(), SIGNAL(newsFeedLoading()), this, SLOT(newsFeedLoading()));

    // News posts
    connect(m_ui->postsListView, SIGNAL(clicked(QModelIndex)), this, SLOT(newsFeedListClicked(QModelIndex)));

    timerEvent(0);
    startTimer(4000);
}

DesktopWidget::~DesktopWidget()
{
    delete m_ui;
}

void DesktopWidget::onDownButtonClicked()
{
    QModelIndex index = m_ui->postsListView->currentIndex();

    if (index.row() < m_ui->postsListView->model()->rowCount(index) - 1)
    {
        index = index.sibling(index.row() + 1, index.column());

        m_ui->postsListView->scrollTo(index, QAbstractItemView::EnsureVisible);
        m_ui->postsListView->setCurrentIndex(index);
    }    
}

void DesktopWidget::onUpButtonClicked()
{
    QModelIndex index = m_ui->postsListView->currentIndex();

    if (index.row() > 0)
    {
        index = index.sibling(index.row() - 1, index.column());
        qDebug() << "Index: " + QString::number(index.row());
    }

    m_ui->postsListView->scrollTo(index, QAbstractItemView::EnsureVisible);
    m_ui->postsListView->setCurrentIndex(index);
}

void DesktopWidget::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        m_ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void DesktopWidget::paintEvent ( QPaintEvent * event ){
    QPainter painter(this);
    painter.fillRect(event->rect(),QColor ( 0,0,0,200));
}

void DesktopWidget::onRefreshButtonClicked()
{
    NewsFeed::instance()->fetchNewsFeed();
    m_ui->refreshButton->setEnabled(false);
    m_ui->refreshButton->setIcon(QIcon::fromTheme("general_stop"));
}

void DesktopWidget::newsFeedRefreshError()
{
    m_ui->refreshButton->setEnabled(true);
    m_ui->refreshButton->setIcon(QIcon::fromTheme("general_refresh"));
}

void DesktopWidget::newsFeedLoaded()
{
    m_ui->refreshButton->setEnabled(true);
    m_ui->refreshButton->setIcon(QIcon::fromTheme("general_refresh"));
}

void DesktopWidget::newsFeedListClicked(QModelIndex index)
{
    MainWindow::instance()->show();

    NewsFeedPostView *nfpv = new NewsFeedPostView(MainWindow::instance());

    nfpv->setPost(static_cast<NewsFeedPost *>(FaceBrick::instance()->m_newsFeedModel->data(index, NewsFeedModel::PostRole).value<void *>()));
    nfpv->show();
}

void DesktopWidget::newsFeedLoading()
{
    m_ui->refreshButton->setEnabled(false);
}
