/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of Zeesteroids.
 *
 * Zeesteroids is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zeesteroids is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Zeesteroids.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <SDL.h>
#include <SDL_gfxPrimitives.h>
#include <SDL_framerate.h>

#ifdef OSSO
#include <libosso.h>      /* required for screen saver timeout */
#endif

#include <zeemote.h>
#include <zeemote-conf.h>

#define FPS 25

#ifdef OSSO
#define MODE   SDL_SWSURFACE|SDL_FULLSCREEN
#else
#define MODE   SDL_SWSURFACE
#endif

#define WIDTH  800
#define HEIGHT 400
#define DEPTH   16

#define SL0   30             // ship length center -> front
#define SL1  -15             // ship length center -> rear
#define SL2  -20             // ship length center -> rear side
#define SE   -17             // offset center -> engine
#define RE     5             // engine diameter
#define OP    45             // rear angle
#define SHR   35             // shield radius

#define ROT_SPEED  10        // value describing max ship rotation speed
#define MAX_IMPULSE 10000.0  // max ship speed

#define SHIP_FLAG_SHIELD  (1<<0)
#define SHIP_FLAG_THRUST  (1<<1)

#define MAX_SHOTS      10
#define SHOT_LIFESPAN  70
#define RELOAD_TIME     6
#define SR              3    // shot radius
#define SS              5    // shot speed

#define X 0
#define Y 1

/* structure holding info about the players ship */
typedef struct {
  int pos[2];
  int impulse[2];
  int angle;
} ship_t;

/* structure holding info about shots */
typedef struct {
  int pos[2];
  int impulse[2];
  int age;
} shot_t;

void draw_shot(SDL_Surface* screen, int x, int y) {
  filledCircleColor(screen, x, y, SR, 0xffffffff);
}

void process_shots(SDL_Surface* screen, ship_t *ship, zeemote_state_t *state) {
  static shot_t shot[MAX_SHOTS];
  static int last_trigger_state = 0;
  static int reload_timer = 0;
  int i;

  /* shoot if gun is reloaded */
  if(!reload_timer) {
    if(state->buttons & ZEEMOTE_BUTTON_A && 
       state->axis[Y] <= 0 &&                // don't shoot while shields are up
       !last_trigger_state) {
      float a = M_PI * (ship->angle >> ROT_SPEED) / 180.0;

      /* search a free shot slot */
      for(i=0;i<MAX_SHOTS && shot[i].age;i++);
      if(i<MAX_SHOTS) {
	shot[i].age = SHOT_LIFESPAN;

	/* place shot ahead of the ship */
	shot[i].pos[X] = ship->pos[X] + (SHR<<16) *  sin(a);
	shot[i].pos[Y] = ship->pos[Y] + (SHR<<16) * -cos(a);

	/* use ships impulse plus some more */
	shot[i].impulse[X] = ship->impulse[X] + (SS<<16) *  sin(a);
	shot[i].impulse[Y] = ship->impulse[Y] + (SS<<16) * -cos(a);
      }
      reload_timer = RELOAD_TIME;
    }

    last_trigger_state = state->buttons & ZEEMOTE_BUTTON_A;
  } else 
    reload_timer--;

  /* advance all shots and draw them */
  for(i=0;i<MAX_SHOTS;i++) {
    if(shot[i].age) {
      shot[i].pos[X] += shot[i].impulse[X];
      shot[i].pos[Y] += shot[i].impulse[Y];

      /* wrap shots position around the screen */
      if((shot[i].pos[X] >> 16) < -SHR)       shot[i].pos[X] = (WIDTH+SHR)<<16;
      if((shot[i].pos[X] >> 16) > SHR+WIDTH)  shot[i].pos[X] = (-SHR)<<16;
      
      if((shot[i].pos[Y] >> 16) < -SHR)       shot[i].pos[Y] = (HEIGHT+SHR)<<16;
      if((shot[i].pos[Y] >> 16) > SHR+HEIGHT) shot[i].pos[Y] = (-SHR)<<16;

      shot[i].age--;

      draw_shot(screen, shot[i].pos[X] >> 16, shot[i].pos[Y] >> 16);
    }
  }
}

void draw_ship(SDL_Surface* screen, int x, int y, int angle, int flags) {
  int rand = random();

  /* calculate angles of ships body */
  float a = angle * M_PI / 180.0;
  float b0 = a + OP*M_PI/180;
  float b1 = a - OP*M_PI/180;


  if(flags & SHIP_FLAG_THRUST) {
    /* do some random engine flickering */
    int ecol = 0xff0000ff;            // default = red
    if(rand & 1) ecol |= 0x00ff0000;  // sometimes yellow
    if(rand & 2) ecol |= 0x80808000;  // sometimes brighter

    /* draw engine */
    filledCircleColor(screen, x + sin(a)*SE, y - cos(a)*SE, 
		      RE, ecol);
  }

  if(flags & SHIP_FLAG_SHIELD)
    circleColor(screen, x, y, SHR, (rand & 4)?0x808080ff:0xffffffff);

  /* draw ships body */
  filledTrigonColor(screen, 
		    x + sin(a)*SL0,  y - cos(a)*SL0, 
		    x + sin(a)*SL1,  y - cos(a)*SL1,
		    x + sin(b0)*SL2, y - cos(b0)*SL2,
		    0x808080ff);
  filledTrigonColor(screen, 
		    x + sin(a)*SL0,  y - cos(a)*SL0, 
		    x + sin(a)*SL1,  y - cos(a)*SL1,
		    x + sin(b1)*SL2, y - cos(b1)*SL2,
		    0xc0c0c0ff);
}

/* handle the players ship and draw it */
void process_ship(SDL_Surface* screen, zeemote_state_t *state) {
  /* the ships state */
  static ship_t ship = { (WIDTH/2) << 16, (HEIGHT/2) << 16, 0, 0, 0 };

  /* rotate using the sticks x axis (axis 0) */
  ship.angle += state->axis[X];
  
  /* accelerate using y axis (axis 1) in up direction */
  int flags = 0;
  if(state->axis[Y] < 0) {
    float a = M_PI * (ship.angle >> ROT_SPEED) / 180.0;

    ship.impulse[X] += state->axis[Y] * -sin(a);
    ship.impulse[Y] += state->axis[Y] *  cos(a);

    /* get absolute impulse */
    float total_impulse = sqrt((ship.impulse[X]>>8) * (ship.impulse[X]>>8) +
			       (ship.impulse[Y]>>8) * (ship.impulse[Y]>>8));

    /* limit overall impulse */
    if(total_impulse > MAX_IMPULSE) {
      ship.impulse[X] *= MAX_IMPULSE/total_impulse;
      ship.impulse[Y] *= MAX_IMPULSE/total_impulse;
    }

    flags |= SHIP_FLAG_THRUST;
  }

  /* add impulse to ships position */
  ship.pos[X] += ship.impulse[X];
  ship.pos[Y] += ship.impulse[Y];

  /* wrap ships position around the screen */
  if((ship.pos[X] >> 16) < -SHR)       ship.pos[X] = (WIDTH+SHR)<<16;
  if((ship.pos[X] >> 16) > SHR+WIDTH)  ship.pos[X] = (-SHR)<<16;

  if((ship.pos[Y] >> 16) < -SHR)       ship.pos[Y] = (HEIGHT+SHR)<<16;
  if((ship.pos[Y] >> 16) > SHR+HEIGHT) ship.pos[Y] = (-SHR)<<16;

  /* jostick more than halfway "down" is shield */
  if(state->axis[Y]>16000)
    flags |= SHIP_FLAG_SHIELD;

  draw_ship(screen, ship.pos[X] >> 16, ship.pos[Y] >> 16, 
	    ship.angle >> ROT_SPEED, flags);

  process_shots(screen, &ship, state);
}

/* the zeemote is connected. So just run the game */
void DrawGameScreen(SDL_Surface* screen, zeemote_state_t *state) { 
  SDL_FillRect(screen, NULL, 0x00000000);

  if(SDL_MUSTLOCK(screen)) 
    if(SDL_LockSurface(screen) < 0) 
      return;
  
  process_ship(screen, state);

  if(SDL_MUSTLOCK(screen)) 
     SDL_UnlockSurface(screen);
  
  SDL_Flip(screen); 
}

/* the zeemote is not in a usable state. So instead of running the */
/* game we just display a message about the zeemotes state */
void DrawStateScreen(SDL_Surface* screen, zeemote_state_t *state) { 
  
  if(SDL_MUSTLOCK(screen)) {
    if(SDL_LockSurface(screen) < 0) return;
  }
  
  SDL_FillRect(screen, NULL, 0x00000000);

  if(!state) {
    const char msg[] = "No Zeemote configured";
    stringColor(screen, WIDTH/2-strlen(msg)*4, HEIGHT/2-4, msg,
		0xff0000ff);
  } else {
    static const char *state_names[] = {
      "Zeemote in unknown state!",
      "Connecting to Zeemote...",
      "Connection to Zeemote failed!",
      "Connected to Zeemote",
      "Connection to Zeemote lost",
      "Disconnected from Zeemote"
    };

    static const Uint32 colors[] = {
      0xff0000ff, 0xffff00ff, 0xff0000ff, 0x00ff00ff, 
      0xff0000ff, 0xff0000ff };

    stringColor(screen, WIDTH/2-strlen(state_names[state->state])*4, 
		HEIGHT/2-4, 
		state_names[state->state],
		colors[state->state]);
  }

  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  
  SDL_Flip(screen); 
}


int main(int argc, char* argv[]) {
  SDL_Surface *screen;
  SDL_Event event;

#ifdef OSSO
  osso_context_t *osso_context = NULL;
  putenv("SDL_VIDEO_X11_WMCLASS=" APP);
#endif

  if (SDL_Init(SDL_INIT_VIDEO) < 0 ) return 1;

  if (!(screen = SDL_SetVideoMode(WIDTH, HEIGHT, DEPTH, MODE))) {
    SDL_Quit();
    return 1;
  }

  SDL_WM_SetCaption(APP, NULL);

#ifdef OSSO
  SDL_ShowCursor(0);  // hide mouse cursor on maemo

  osso_context = osso_initialize (APP, "1.0", 0, NULL);
  if(osso_context == NULL) {
      fprintf(stderr, "error initiating osso context\n");
  }
#endif

  /* zeemote_get_scan_results_from_gconf() uses gconf which in turn */
  /* uses glib types. If these are used outside a e.g. gtk application */
  /* they need to be initialized explicitely */
  g_type_init();

  /* Get list of configured zeemotes from the conf tool. One could */
  /* alternally call zeemote_scan() which would return the same type */
  /* of list, but instead from the devices currently visible. Also */
  /* zeemote_scan() blocks for about 10 seconds */
  zeemote_scan_result_t *scan_result = 
    zeemote_get_scan_results_from_gconf();

  /* if devices are configured use the first one in the list as this */
  /* is supposed to be a single player game */
  zeemote_t *zeemote = NULL;
  if(scan_result && scan_result->number_of_devices > 0) 
    zeemote = zeemote_connect(&scan_result->device[0].bdaddr);
 
  /* init framerate manager */
  FPSmanager manex;
  SDL_initFramerate(&manex);
  SDL_setFramerate(&manex, FPS);

  int quit = 0;
  while(!quit) {
    
    /* get state of zeemote if a zeemote has been connected */
    zeemote_state_t *state = (!zeemote)?NULL:zeemote_get_state(zeemote);

    /* run the game if the zeemote is connected, otherwise tell the */
    /* user about the zeemotes state */
    if(state && state->state == ZEEMOTE_STATE_CONNECTED)
      DrawGameScreen(screen, state);
    else
      DrawStateScreen(screen, state);

    SDL_framerateDelay( &manex );

    /* quit game with zeemote's power button */
    if(state && state->buttons & ZEEMOTE_BUTTON_D)
      quit = 1;

    while(SDL_PollEvent(&event)) {      
      switch (event.type) {
      case SDL_QUIT:
      case SDL_MOUSEBUTTONDOWN:
      case SDL_KEYDOWN:
	quit = 1;
	break;
      }
    }
  }
  
  SDL_Quit();

#ifdef OSSO
  if (osso_context)
    osso_deinitialize(osso_context);
#endif

  /* finally close the connection to the zeemote */
  if(zeemote)
    zeemote_disconnect(zeemote);

  return 0;
}




