# 
# Copyright 2010 Ruediger Gad
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from ctypes import *
import time
import select
import os

CDLL("libglib-2.0.so.0", RTLD_GLOBAL)
CDLL("libbluetooth.so.3", RTLD_GLOBAL)
gconf = CDLL("libgconf-2.so.4", RTLD_GLOBAL)
gobj = CDLL("libgobject-2.0.so.0", RTLD_GLOBAL)
zeemote = CDLL("libzeemote.so.1")
zeemoteConf = CDLL("libzeemote-conf.so.1")

buttonmap = {
    'A': 0x0001,
    'B': 0x0002,
    'C': 0x0004,
    'D': 0x0008
} 
 
class BdAddrStruct(Structure):
  _fields_ = [("b", c_char * 6)]

class ZeemoteStateStruct(Structure):
  _fields_ = [("state", c_int),
              ("axis", c_int * 4),
              ("buttons", c_ulong),
              ("battery", c_int)]

getState = zeemote.zeemote_get_state
getState.restype = POINTER(ZeemoteStateStruct)

class ZeemoteInfoStruct(Structure):
  _fields_ = [("type", c_int),
              ("num_axes", c_int),
              ("num_buttons", c_int)]

class ZeemoteStruct(Structure):
  pass
ZeemoteStruct._fields_ = [("zeemote_info", ZeemoteInfoStruct),
                          ("zeemote_state", ZeemoteStateStruct * 2),
                          ("refcount", c_int),
                          ("fd", c_int),
                          ("bdaddr", BdAddrStruct),
                          ("next", POINTER(ZeemoteStruct)),
                          ("thread", c_ulong),
                          ("mutex", c_int),
                          ("cond", c_int)]

class ZeemoteDeviceStruct(Structure):
  _fields_ = [("type", c_int),
              ("bdaddr", BdAddrStruct)]

class ZeemoteScanResultStruct(Structure):
  _fields_ = [("number_of_devices", c_int),
              ("device", ZeemoteDeviceStruct)]

class Zeemote():
  def __init__(self, pointer):
    self.p = pointer
    self.poll = select.poll()
    self.poll.register(self.p[0].fd, select.POLLPRI)
  def getState(self):
    return ZeemoteState(self.p)
  def disconnect(self):
    zeemote.zeemote_disconnect(self.p) 
  def waitForData(self):
    zeemote.wait_for_data(self.p)

class ZeemoteState():
  def __init__(self, zeemote):
    self.s = getState(zeemote)[0]
  def getX(self):
    return self.s.axis[0]
  def getY(self):
    return self.s.axis[1]
  def isConnected(self):
    return (self.s.state == 3)
  def getButtonMask(self):
    return self.s.buttons

class ZeeDiscover():
  def __init__(self):
    gobj.g_type_init()

  def getZeemote(self, id):
    getScanResults = zeemoteConf.zeemote_get_scan_results_from_gconf
    getScanResults.restype = POINTER(ZeemoteScanResultStruct)
    result = getScanResults()

    connect = zeemote.zeemote_connect
    connect.restype = POINTER(ZeemoteStruct)
    dev = connect(byref(result[id].device.bdaddr))
 
    state = getState(dev)
    while state[0].state != 3:
      time.sleep(0.2)
      state = getState(dev)
      if state[0].state == 2:
        zeemote.zeemote_disconnect(dev)
        dev = connect(byref(result[0].device.bdaddr))

    return Zeemote(dev)

