#!/usr/bin/python

#
# ZDaemon 0.9.7
# 
# Small portions of this code was borrowed from 
# Anssi Kolehmainen <anssi@aketzu.net>, although
# very little of that code is still left, some
# traces remain (like struct and function names
# and overall layout), all the code interfacing
# with Google Latitude is changed
#

import math

import zlatitude   # Zaps Google Latitude service
import zconfig     # ZapLoc configuration file manager
import zzaloc      # The ZapLoc (meta) service
import zbrowse
import time
import copy

try:
    import dbus
    import dbus.mainloop.glib
    import location
    import gobject
    import conic
    
except:
    pass

import time
import socket
import os
import sys
import getopt
import signal
import pickle

# These values are hardcoded
EPS               = 150   # Latitude samples within this distance are ignored
EPS2              = 50    # General samples within this distance are also ignored
UPDATE_AT_MOST    = 2     # NEVER send out more than every two minutes (only matters if some other app has the GPS on)
ACC_LIMIT         = 100   # Accuracy limit fpr GPS mode        - never accept GPS positions with worse accuracy than this
ACC_LIMIT2        = 9999  # Accuracy limit for cell-tower mode - never accept cell tower locations with worse accuracy than this

FIX_TIMEOUT       = 30    # How many seconds before the GPS is killed

# These values are read from the config file

# For google latitude; 
# even tho we DIDN'T move, update with AT LEAST this frequency (in minutes)
# This is to keep updates from becoming "stale" in the friends list
UPDATE_AT_LEAST   = 60    

# When getting a good GPS fix, we will sleep the GPS for this number of minutes and then re-awaken it
GPS_INTERVAL      = 5     

# 0 = Use wifi and cell only
# 1 = Use wifi, GPS and cell
# 2 = Use wifi and GPS only
# 3 = Use wifi only
GPS_METHOD        = 1  


def get_battery():
    bus = dbus.SystemBus()
    hal_obj = bus.get_object('org.freedesktop.Hal', '/org/freedesktop/Hal/Manager')
    hal = dbus.Interface(hal_obj, 'org.freedesktop.Hal.Manager')
    uids = hal.FindDeviceByCapability('battery')
    dev_obj = bus.get_object('org.freedesktop.Hal', uids[0])

    percent  = dev_obj.GetProperty('battery.charge_level.percentage')
    charging = dev_obj.GetProperty('battery.rechargeable.is_charging')
    
    return (charging, percent)


def switch_to_gps(control):
    control.set_properties(preferred_method   = location.METHOD_USER_SELECTED, #ACWP
                           preferred_interval = location.INTERVAL_5S)

def switch_to_cell(control):
    control.set_properties(preferred_method   = location.METHOD_ACWP,
                           preferred_interval = location.INTERVAL_120S)

zlat       = None
zconf      = None
connection = None
force_connection = False

verbose=0
silent =0
test   =0

wlan_id = None

def connection_cb(connection, event):
    global wlan_id
    # print "connection_cb(%s, %s)" % (connection, event)

    status = event.get_status()
    error  = event.get_error()
    iap_id = event.get_iap_id()               
    bearer = event.get_bearer_type()               

    name = ""
    if verbose:
        try:
            iaps = connection.get_all_iaps()
            for iap in iaps:
                if iap.get_id() == iap_id:
                    name = iap.get_name()
        except:
            pass

    if status == conic.STATUS_CONNECTED:
        if verbose: print time.strftime('%H:%M:%S') + ": CONNECTED (%s, %s, %s, %i, %i)"     % (name, iap_id, bearer, status, error)
        wlan_id = iap_id        
    elif status == conic.STATUS_DISCONNECTED:
        if verbose: print time.strftime('%H:%M:%S') + ": DISCONNECTED (%s, %s, %s, %i, %i)"  % (name, iap_id, bearer, status, error)
        # wlan_id = None
    elif status == conic.STATUS_DISCONNECTING:
        if verbose: print time.strftime('%H:%M:%S') + ": DISCONNECTING (%s, %s, %s, %i, %i)" % (name, iap_id, bearer, status, error)
        wlan_id = None

def connected():
    global wlan_id
    if wlan_id == None:
        if force_connection:
            if verbose: print time.strftime('%H:%M:%S') + ": Connection down - starting it up"
            connection.request_connection(conic.CONNECT_FLAG_NONE)
    
            count = 0
            while wlan_id == None and count < 10:
                count += 1
                if verbose: print time.strftime('%H:%M:%S') + ": Waiting for connection %d...." % count
                time.sleep(2)
    else:
        if verbose: print time.strftime('%H:%M:%S') + ": Connection is up - Good!"
        
    return wlan_id != None
        
def meter_distance( lata, lnga, latb, lngb ):
    dx    = lata - latb
    dy    = lnga - lngb
    dy   *= math.cos(math.radians(lata))
    dist  = math.sqrt(dx*dx+dy*dy)
    dist *= 40000000 / 360
    return dist    

def get_config(conf, name, default=0):
    try:
        return conf[name]
    except:
        return default

is_awake  = False
last_fix  = None
    
class LocationData:
    lat=0
    lng=0
    alt=0
    acc=0
    alt=0
    altacc=0
    head=0
    speed=0;

    # "Last-used" coordinates
    prevlat=0
    prevlng=0
    # Latitude "last-used" coordinates
    prevlatlat=0
    prevlatlng=0
    
    prevupd=0
    prevupd2=0

    last_spot_count      = 0
    last_spot_id         = None

    def push(self):
        global service
        global verbose
        global silent
        global zconf

        if verbose and self.prevupd > 0: print time.strftime('%H:%M:%S') + ": Last used sample %.1f minutes ago" % (( time.time() - self.prevupd ) / 60)

        if GPS_METHOD and time.time() - self.prevupd < UPDATE_AT_MOST * 60:
            if verbose: print time.strftime('%H:%M:%S') + ": Update not posted because not enough time has passed"
            return True

        if self.lat == 0 or self.lng == 0:
            return True

        #  First do Latitude, which we will do more often if we get updates more often

        meters = meter_distance(self.lat, self.lng, self.prevlatlat, self.prevlatlng) 
        if verbose and (self.prevlatlat != 0 or self.prevlatlng != 0): print time.strftime('%H:%M:%S') + ": Last posted Latitude sample %.1f meters away" % meters 
        
        if zlat:
            if meters < EPS and time.time() - self.prevupd < UPDATE_AT_LEAST * 60:
                if verbose: 
                    if UPDATE_AT_LEAST < 99999:
                        print time.strftime('%H:%M:%S') + ": Latitude updates within %g meters and within %d hours - not sending" % ( EPS,  UPDATE_AT_LEAST/60 )
                    else:
                        print time.strftime('%H:%M:%S') + ": Latitude updates within %g meters - not sending" % EPS
            else:
                if meters < EPS:
                    if verbose: print time.strftime('%H:%M:%S') + ": Latitude position re-posted, was within %g meters." % EPS
                    
                if not silent: print "%s: Latitude update %f,%f (acc %.1f)" % (time.strftime('%H:%M:%S'),self.lat,self.lng,self.acc)

                try:
                    if not test: 
                        if not connected():
                            print time.strftime('%H:%M:%S') + ": Not connected to the network, exiting"
                            return True
                        
                        x = zlat.check_in((self.lat, self.lng, self.alt, self.acc), "")
                        try:
                            print time.strftime('%H:%M:%S') + ": " + x[0]
                            self.prevlatlat = self.lat
                            self.prevlatlng = self.lng
                        except:
                            pass
                        
                except:
                    print time.strftime('%H:%M:%S') + ": Error updating location", sys.exc_info()[0]
                    return True
                
                try:
                    self.prevupd = time.time()
                    pickle.dump(self, open("/home/user/.zaploc/lock.dat", "w"))
                except:
                    pass

        # With Latitude done, we want to make the other stuff at a gentler pace...
        # Make sure we hold it "GPS_INTERVAL" times apart, even though we may get here
        # more often (as most UPDATE_AT_MOST interval)
        meters = meter_distance(self.lat, self.lng, self.prevlat, self.prevlng)        
        if verbose and (self.prevlat != 0 or self.prevlng != 0): print time.strftime('%H:%M:%S') + ": Last used sample %.1f meters away" % meters 
        
        if meters < EPS2 and self.last_spot_count < 0:
            if verbose: print time.strftime('%H:%M:%S') + ": We have moved less than %g meters - do nothing" % EPS2
            return True
        
        if time.time() - self.prevupd2 > GPS_INTERVAL * 60 * 0.5: 
            self.prevupd2 = time.time()

            if not silent: print "%s: General location update %f,%f (acc %.1f)" % (time.strftime('%H:%M:%S'),self.lat,self.lng,self.acc)
            
            # Check if any service is even on!
            if get_config(zconf, "foursquare", True) or get_config(zconf, "gowalla", True) or get_config(zconf, "facebook", True):
                if verbose: print time.strftime('%H:%M:%S') + ": Searching for closest spot:"
                
                # Read in the spot database
                zzaloc.get()
    
                closest_item = None
                closest_dist = 5000
    
                # Find the closest spot in the database
                for item in zzaloc.database:
                    if item["auto-check"]:
                        dist = meter_distance(item["lat"], item["lng"], self.lat, self.lng)
                        if dist < closest_dist:
                            closest_dist = dist
                            closest_item = item
                            
                if closest_item != None:
                    if verbose: print time.strftime('%H:%M:%S') + ": The closest AutoCheck spot is %s at %g meters" % ( closest_item["name"], closest_dist )
                    
                    maxdist = closest_item["auto-distance"]
                    
                    if closest_dist < maxdist:
                        if zzaloc.getlastcheckin() == closest_item["id"]:
                            # If we are already checked in to this spot - do nothing
                            if verbose: print time.strftime('%H:%M:%S') + ": We are already checked in at %s so we will do nothing" % closest_item["name"]
                            self.last_spot_count = -1
                        else:
                            if self.last_spot_id == closest_item["id"]:
                                # Increase the count of times we've been here
                                self.last_spot_count += 1
                                if verbose: print time.strftime('%H:%M:%S') + ": We have now been at %s %d times" % ( closest_item["name"], self.last_spot_count)
                                
                                if self.last_spot_count > get_config(zconf,"daemon-autocheck-delay", 2):
                                    self.last_spot_count = -1 # Mark as "we checked in"
                                    if not silent:
                                        print "****************************************************"
                                        print "  We are checking in at %s!" % closest_item["name"]
                                        print "****************************************************"

                                    service = zzaloc.Service()
                                    
                                    if service.is_authorized():
                                        if test:
                                            result = ( "Test mode:" , "No actual checkin performed" )
                                            zzaloc.addcheckin(closest_item)
                                        else:
                                            if not connected():
                                                print time.strftime('%H:%M:%S') + ": Not connected to the network, exiting"
                                                return True
                                            
                                            result = service.check_in(
                                                (closest_item["id"],
                                                 closest_item["lat"],
                                                 closest_item["lng"],
                                                 closest_item["name"]), 
                                                 (self.lat, self.lng ), 
                                                 closest_item["auto-message"], 
                                                 int(closest_item["auto-tweet"]),
                                                 int(closest_item["auto-facebook"])
                                            )
                                        if result != None:					    
                                            zbrowse.notify("You were automatically checked in to " +  closest_item["name"])
                                            for line in result:
                                                zbrowse.notification(line)
                                                time.sleep(3)
                                                if verbose: print time.strftime('%H:%M:%S') + ": " + line
                            else:
                                # Sorry, different spot, reset spot count
                                self.last_spot_count = 0
                                if not silent: print time.strftime('%H:%M:%S') + ": Arrived at %s (but not checked in yet)" % closest_item["name"]

                            # Now remember this for next time
                            self.last_spot_id = closest_item["id"]			
                    else:
                        if verbose: print time.strftime('%H:%M:%S') + ": Spot ignored because it is further than %g meters away" % maxdist
                        self.last_spot_count = 0

                # Throw away the memory for the list of spots
                zzaloc.release()			
            else:
                if verbose: print time.strftime('%H:%M:%S') + ": No services configured - doing nothing."
        else:
            if verbose: print time.strftime('%H:%M:%S') + ": Below GPS interval - no checking nearest spot"
            
        self.prevlat = self.lat
        self.prevlng = self.lng

        try:
            pickle.dump(self, open("/home/user/.zaploc/lock.dat", "w"))
        except:
            pass
        
        return True
  
try:
    curloc = pickle.load(open("/home/user/.zaploc/lock.dat"))
    ago =  time.time() - curloc.prevupd
except:
    curloc = LocationData()

fix_tries = 0

modeinfo = (
    "0 (Cached)",
    "1 (Estimate)",
    "2 (Cell-tower)",
    "3 (GPS)",
    "4 (WiFi)"
    )

def update_loc(mode, lat, lng, acc, alt, altacc, head, speed, data, force = False):
    global verbose
    global acclimit
    global once
    global fix_tries
    global is_awake
    global curloc

    # Ignore cached or country-size measurements
    if mode < 2:
        if verbose: print "%s: Got loc data %f,%f (acc %.1f), Mode %s - IGNORED" % (time.strftime('%H:%M:%S'),lat,lng,acc, modeinfo[mode])
        return

    print "%s: Got loc data %f,%f (acc %.1f), head %.1f, speed %.1f, Mode %s, Try %d" % (time.strftime('%H:%M:%S'),lat,lng,acc, head, speed, modeinfo[mode], fix_tries)
    
    # Skip the NaN's in accuracy
    if acc != acc:
        return

    # Skip no-accuracy values
    if acc <= 0:
        return
    
    if altacc > 32000:
        altacc=0
    
    # If using GPS_METHOD=1, then let a "force" override go to cell tower
    # If using GPS_METHOD=2, ignore force, NEVER agree to a poor quality (non-GPS) location
    if (GPS_METHOD == 1 and not force) or (GPS_METHOD == 2):
        # I don't care about data of low accuracy
        if mode < 3:
            if verbose: print time.strftime('%H:%M:%S') + ": Sample mode %s - IGNORED" % modeinfo[mode]
            return

        # These days this will most likely never happen...
        if not force:
            if acc > ACC_LIMIT: 
                if verbose: print time.strftime('%H:%M:%S') + ": Sample accuracy worse than %g meters - IGNORED" % ACC_LIMIT
                return

            #if fix_tries < 2:
            #    fix_tries += 1
            #    if verbose: print time.strftime('%H:%M:%S') + ": Waiting for try number %d" % fix_tries 
            #    return
            
        #fix_tries = 0
    else:
        # I don't care about data of low accuracy
        if acc > ACC_LIMIT2:
            if verbose: print time.strftime('%H:%M:%S') + ": Sample accuracy worse than %g meters - IGNORED" % ACC_LIMIT2
            return

    curloc.lat=lat
    curloc.lng=lng
    curloc.acc=acc
    curloc.alt=alt
    curloc.altacc=altacc
    curloc.head=head
    curloc.speed=speed
   
    curloc.push()

    if speed < 50 or speed != speed or GPS_METHOD == 0 or mode < 3:
        if GPS_INTERVAL > 0:
            print time.strftime('%H:%M:%S') + ": Sleeping... Zzzzz"
            data.stop()
        """else:
            if GPS_METHOD:
                print time.strftime('%H:%M:%S') + ": Switching to cell tower mode"
                switch_to_cell(data)
            else:
                print time.strftime('%H:%M:%S') + ": Awaiting next cell tower change..." """
            
    is_awake = False

    return

def push_location(control):
    global last_fix
    global is_awake
    global FIX_TIMEOUT
    if is_awake:
        if verbose: print time.strftime('%H:%M:%S') + ": No fix acquired for %d seconds - doing a forced location push!" % FIX_TIMEOUT
        # Re-send the last one again
        if last_fix != None:
            update_loc(last_fix[0], last_fix[4], last_fix[5], last_fix[6]/100, last_fix[7], last_fix[8], last_fix[9], last_fix[11], control, True)
        
        # print time.strftime('%H:%M:%S') + ": Forcing GPS to sleep!"            
        if GPS_INTERVAL > 0:
            if verbose: print time.strftime('%H:%M:%S') + ": Force-Sleeping [*bonk!*].... Zzzzzz" 
            control.stop()
        """ else:
            if GPS_METHOD:
                if verbose: print time.strftime('%H:%M:%S') + ": Switching to cell tower mode"
                switch_to_cell(data)
            else:
                if verbose: print time.strftime('%H:%M:%S') + ": Awaiting next cell tower change..." 
        """
                
        is_awake = False            
        
    else:
        if verbose: print time.strftime('%H:%M:%S') + ": A good location fix was acquired within %g seconds: Good!" % FIX_TIMEOUT
    
    return False


def on_error(control, error, data):
    print time.strftime('%H:%M:%S') + ": location error: %d..." % error

def on_changed(device, data):
    global last_fix, is_awake
    if not device:
        return

    # If we are in "only when cell tower changes" mode...
    """if GPS_INTERVAL == 0 and GPS_METHOD == 0 and not is_awake:
        if verbose: print time.strftime('%H:%M:%S') + ": New cell location came in, switching on GPS again"
        switch_to_gps(data)"""
    
    if device.fix:
        if device.fix[1] & location.GPS_DEVICE_LATLONG_SET:
            last_fix = copy.copy(device.fix)
            update_loc(device.fix[0], device.fix[4], device.fix[5], device.fix[6]/100, device.fix[7], device.fix[8], device.fix[9], device.fix[11], data)

def test_flight_mode():
    try:
        bus = dbus.SystemBus()
        foo = bus.get_object("com.nokia.mce", "/com/nokia/mce/request")
        m   = foo.get_device_mode()
        
        return m == "flight"	
    except:
        return False
            
def restart_gpsd(control, force = False):
    global verbose, is_awake, FIX_TIMEOUT

    if test_flight_mode():
        if verbose: print time.strftime('%H:%M:%S') + ": Device in OFFLINE mode, keeping GPS OFF"	
        return True

    wifi_pos = None
    
    # Are we on a WLAN?
    if wlan_id:
        zzaloc.get()
        for item in zzaloc.database:
            try:
                if item["wifi-id"] == wlan_id:
                    wifi_pos  = (item["lat"], item["lng"], 0, 0 )
                    wifi_name =  item["wifi-name"]
            except:
                pass
            
        zzaloc.release()
    
    if wifi_pos or GPS_METHOD == 3:
        if wifi_pos:
            if verbose: print time.strftime('%H:%M:%S') + ": Using location from WLAN '" + wifi_name + "'"
            update_loc(4, wifi_pos[0], wifi_pos[1], 10, 0, 0, 0, 0, control, False)
        else:
            if verbose: print time.strftime('%H:%M:%S') + ": Not on known WLAN in WiFi only mode - doing nothing."
        
        if GPS_INTERVAL > 0:
            gobject.timeout_add(int(GPS_INTERVAL    * 60000), restart_gpsd, control)
        """else:        
            gobject.timeout_add(int(UPDATE_AT_LEAST * 60000), restart_gpsd, control)"""
        
        return False
    
    if verbose: print time.strftime('%H:%M:%S') + ": Starting GPS..."

    if force: 
        control.start()
    
    if GPS_INTERVAL > 0:
        control.start()

        battery = get_battery()

        factor = 1.0
        
        why = ""
        
        if battery[0] > 0:
            factor = 0.5
            why    = "(on charger - sped up 2x)"
        elif battery[1] < 30:
            if battery[1] < 10:
                factor = 10.0
                why    = "(empty battery!)"
            else:
                factor = 2.5
                why    = "(low battery!)"
                    
        if verbose: 
            print time.strftime('%H:%M:%S') + ": Actual GPS interval %g minutes %s" % ((GPS_INTERVAL * factor), why)
                
        gobject.timeout_add(int(GPS_INTERVAL    * 60000 * factor), restart_gpsd, control)
    """else:
        gobject.timeout_add(int(UPDATE_AT_LEAST * 60000), restart_gpsd, control)
        
        if GPS_METHOD:
            if verbose: print time.strftime('%H:%M:%S') + ": Switching to GPS"
            switch_to_gps(control)
            control.start()"""
        
    is_awake = True

    gobject.timeout_add(int(FIX_TIMEOUT * 1000), push_location, control)
    
    return False

def start_location(control):
    restart_gpsd(control, True)
    return False

last_cell      = None
global_control = None

def cell_signal_handler(*args):
    global last_cell, global_control

    cell = ( args[4], args[3], args[2] )
    print time.strftime('%H:%M:%S') + ": Cell tower: %s,%s,%s" % cell

    if last_cell != cell:
        last_cell = cell
        print time.strftime('%H:%M:%S') + ": Changed - starting GPSD"
        global_control.start()

def usage():
    print "ZapLoc Daemon - Zaps Location Updater Daemon v0.9.7"
    print " -h          This help message"
    print " -d          Run as Daemon always (the normal thing to do)"
    print " -b          Run as Daemon except if config says otherwise"
    print "             (this option is used when booting)"
    print " -t          Terminate Daemon (if running)"
    print " -i  <time>  Set the polling interval to <time> minutes*" 
    print " --cell      Switches to use cell towers only for location (+WiFi)*"
    print " --gps       Switches to use GPS only for location (+WiFi)*"
    print " --both      Preffers GPS location but falls back to Cell Tower if no GPS fix is found (+WiFi)*"
    print " --wifi      ONLY use the currently connected WiFi network as location source*"
    print " -s  Simulate (don't actually send any updates). Useful together w. 'verbose' for testing"
    print " -v  Verbose output. Loads o' crap"
    print ""
    print "     * = Note these options OVERRIDES the configuration but does not SET the configuration,"
    print "         i.e. restarting the program without the option reverts to the configurations value."
    # print " -i  Install Daemon to run at boot"
    # print " -u  Uninstall Daemon from running at boot"

def init(boot_mode):
    global global_control, connection
    
    socket.setdefaulttimeout(20)
    
    loop = gobject.MainLoop()
    control = location.GPSDControl.get_default()
    device = location.GPSDevice()

    connection = conic.Connection()
    connection.set_property("automatic-connection-events", True)
    connection.connect("connection-event", connection_cb)

    
    if GPS_METHOD > 0:
        switch_to_gps(control)
    else:
        switch_to_cell(control)

    control.connect("error-verbose", on_error, loop)
    device.connect("changed", on_changed, control)
    # control.connect("gpsd-stopped", on_stop, loop)

    # If run from boot, we start after an initial timeout
    if boot_mode:
        gobject.timeout_add(int(60000), start_location, control)
    else:
        gobject.timeout_add(int(3000), start_location, control)

    """
    if GPS_INTERVAL == 0:
        global_control = control
        # gobject.timeout_add(int(UPDATE_AT_LEAST * 60000), restart_gpsd, control)
        pass

        
        #dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
        
        #bus = dbus.SystemBus()
        #bus.add_signal_receiver(cell_signal_handler, dbus_interface = "Phone.Net", signal_name = "registration_status_change")
        #bus.add_signal_receiver(cell_signal_handler, dbus_interface = "Phone.Net", signal_name = "cell_info_change")
        
    else:    
        # gobject.timeout_add(int(GPS_INTERVAL * 60000), restart_gpsd, control)
        pass
    """
    
    loop.run()

def test_if_running():
    # os.system("echo `ps | grep \"python.*zaploc-daemon.py\" | grep -v grep | awk '{print $1}'` > pids.txt")

    result = False
    
    try:
        fp  = os.popen("pgrep -f 'zaploc-daemon.py'")
        foo = fp.read()
        fp.close()
        # os.remove("pids.txt")

        ary = foo.split("\n")

        for foo in ary:
            try:
                p = int(foo)
                if p>0 and p != os.getpid():
                    result = True
            except:
                pass
    except:
        pass

    return result
    
    
def daemonize(mode):
    # Kill any already running daemons...
    # os.system("echo `ps | grep \"python.*zaploc-daemon.py\" | grep -v grep | awk '{print $1}'` > pids.txt")

    try:
        fp  = os.popen("pgrep -f 'zaploc-daemon.py'")
        foo = fp.read()
        fp.close()
        # os.remove("pids.txt")

        ary = foo.split("\n")

        for foo in ary:
            try:
                p = int(foo)
                if p>0 and p != os.getpid():
                    os.kill(p, signal.SIGTERM)
                    if mode == 2:
                        print "Daemon was already running as process %d - stopped" % p
            except:
                pass
    except:
        pass

    if mode > 0:
        return
        
    pid = os.fork()
    if (pid == 0):
        os.setsid()
        pid = os.fork()
        if (pid == 0):
            os.umask(0)
        else:
            os._exit(0)
    else:
        os._exit(0)

    os.close(0);
    os.close(1);
    os.close(2);

def main(argv):
    global verbose, service, once, zlat, test, zconf, silent
    global UPDATE_AT_LEAST, GPS_INTERVAL, GPS_METHOD, FIX_TIMEOUT
    global force_connection

    try:
        opts, args = getopt.getopt(argv, "hdvtbsi:", ["help", "verbose", "boot", "gps", "cell", "both", "wifi" ])
    except getopt.GetoptError:
        usage()
        sys.exit(2)

    # Read in the configuration
    zconf = zconfig.get()

    UPDATE_AT_LEAST   = (  99999, 1*60-3, 2*60-3, 6*60-3, 24*60-3)       [get_config(zconf, "daemon-latitude-freq", 1)]
    GPS_INTERVAL      = (      5,     10,     15,     30,      60,   120)[get_config(zconf, "daemon-interval", 1)]
    GPS_METHOD        = get_config(zconf, "daemon-method")
    FIX_TIMEOUT       = get_config(zconf, "daemon-gps-timeout")
    if FIX_TIMEOUT < 10:
        FIX_TIMEOUT   = 30
    force_connection  = get_config(zconf, "daemon-force-connection", True)

    try: 
        if not zconf["daemon-enable"]:
            if verbose: print "Boot-state: Disabled"
    except:
        print "Configuration is damaged - exiting"
        return	    

    if zconf["latitude"] and zconf["daemon-latitude"]:
        zlat = zlatitude.Service()
        if not zlat.is_authorized():
            zlat.authorize()	

    boot_mode = False
            
    for opt,arg in opts:
        if opt in ("-h", "--help"):
            usage()
            sys.exit(0)
        elif opt in ("-d"):
            daemonize(0)
            silent = 1
        elif opt in ("-t"):
            daemonize(1)
            return
        if opt in ("-v"):
            verbose=1
        if opt in ("-s"):
            test=1
        if opt in ("-i"):
            GPS_INTERVAL = float(arg)
            if GPS_INTERVAL <= 0:
                print "Invalid GPS interval: " + arg
                GPS_INTERVAL = 10
        if opt in ("--cell"):
            GPS_METHOD = 0
        if opt in ("--both"):
            GPS_METHOD = 1
        if opt in ("--gps"):
            GPS_METHOD = 2
        if opt in ("--wifi"):
            GPS_METHOD = 3
        if opt in ("-b", "--boot"):
            try: 
                if not zconf["daemon-enable"]:
                    print "Boot mode: Daemon disabled in configuration - not starting it - exiting."
                    return
                boot_mode = True
            except:
                print "Configuration is damaged - exiting"
                return	    
            daemonize(0)
            silent = 1

    print "Starting ZapLoc daemon..."
    print "     Location method: %s" % ( ( "WiFi > Cell-tower", "WiFi > GPS > Cell-tower", "WiFi > GPS", "WiFi" )[GPS_METHOD] )
    print "     %g minutes between samples" % ( GPS_INTERVAL )
            
    # Just test if one is running in the back, and complain if it is
    if not silent:
        daemonize(2)
        
    init(boot_mode)

if __name__ == '__main__': main(sys.argv[1:])
