import pygame
from pygame.locals import *

SHOFF = 2

class TextRectException:
    def __init__(self, message = None):
        self.message = message
    def __str__(self):
        return self.message

def render_textrect(string, font, rect, text_color, justification=0, cursor_pos = -1, shadow = None):
    """Returns a surface containing the passed text string, reformatted
    to fit within the given rect, word-wrapping as necessary. The text
    will be anti-aliased.

    Takes the following arguments:

    string - the text you wish to render. \n begins a new line.
    font - a Font object
    rect - a rectstyle giving the size of the surface requested.
    text_color - a three-byte tuple of the rgb value of the
                 text color. ex (0, 0, 0) = BLACK
    justification - 0 (default) left-justified
                    1 horizontally centered
                    2 right-justified

    Returns the following values:

    Success - a surface object with the text rendered onto it.
    Failure - raises a TextRectException if the text won't fit onto the surface.
    """

    final_lines = []

    requested_lines = string.splitlines()
    
    # Create a series of lines that will fit on the provided
    # rectangle.    
    
    for requested_line in requested_lines:
        if font.size(requested_line)[0] > rect.width:
            words = requested_line.split(' ')
            # if any of our words are too long to fit, return.
            for word in words:
                if font.size(word)[0] >= rect.width:
                    raise TextRectException, "The word " + word + " is too long to fit in the rect passed."
            # Start a new line
            accumulated_line = ""
            for word in words:
                test_line = accumulated_line + word + " "
                # Build the line while the words fit.    
                if font.size(test_line)[0] < rect.width:
                    accumulated_line = test_line 
                else: 
                    final_lines.append(accumulated_line[:-1]) 
                    accumulated_line = word + " " 
            final_lines.append(accumulated_line)
        else: 
            final_lines.append(requested_line) 

    # Let's try to write the text out on the surface.

    surface = pygame.Surface(rect.size, SRCALPHA)    

    if shadow:
        accumulated_height = 0 
        for line in final_lines: 
            # if accumulated_height + font.size(line)[1] >= rect.height:
            #    raise TextRectException, "Once word-wrapped, the text string was too tall to fit in the rect."
    
            if line != "":
                tempsurface = font.render(line, 1, shadow)
                if justification == 0:
                    surface.blit(tempsurface, (SHOFF, accumulated_height+SHOFF))
                elif justification == 1:
                    surface.blit(tempsurface, ((rect.width - tempsurface.get_width()) / 2+SHOFF, accumulated_height+SHOFF))
                elif justification == 2:
                    surface.blit(tempsurface, (rect.width - tempsurface.get_width()+SHOFF, accumulated_height+SHOFF))            
                else:
                    raise TextRectException, "Invalid justification argument: " + str(justification)

            accumulated_height += font.size(line)[1]
                                    
    accumulated_height = 0 
    for line in final_lines: 
        # if accumulated_height + font.size(line)[1] >= rect.height:
        #    raise TextRectException, "Once word-wrapped, the text string was too tall to fit in the rect."

        if line != "":
            tempsurface = font.render(line, 1, text_color)
            if justification == 0:
                surface.blit(tempsurface, (0, accumulated_height))
            elif justification == 1:
                surface.blit(tempsurface, ((rect.width - tempsurface.get_width()) / 2, accumulated_height))
            elif justification == 2:
                surface.blit(tempsurface, (rect.width - tempsurface.get_width(), accumulated_height))            
            else:
                raise TextRectException, "Invalid justification argument: " + str(justification)

        if cursor_pos >= 0 and cursor_pos <= len(line) + 1:
            cursor_x = font.size(line[:cursor_pos])[0]
            r = Rect(cursor_x, accumulated_height, 2, font.size("!")[1])
            pygame.draw.rect(surface, text_color, r )
            
            
        cursor_pos -= len(line)+1
        accumulated_height += font.size(line)[1]

    return surface


if __name__ == '__main__':
    import pygame
    import pygame.font
    from pygame.locals import *

    pygame.init()

    display = pygame.display.set_mode((400, 400))

    pygame.key.set_repeat(150, 40)
    
    my_font = pygame.font.Font(None, 22)

    my_string = "Hi there!\n\n\nI'm a nice bit of wordwrapped text. Won't you be my friend? Honestly, wordwrapping is easy, with David's fancy new render_textrect() function.\nThis is a new line.\n\nThis is another one.\n\n\nAnother line, you lucky dog."

    my_rect = pygame.Rect((40, 40, 300, 300))

    clock = pygame.time.Clock()

    cursor_pos = -1
    
    while True:
        for event in pygame.event.get():
            if event.type == KEYDOWN:
                if event.key == K_LEFT:
                    cursor_pos -= 1
                if event.key == K_RIGHT:
                    cursor_pos += 1
        
        display.fill((100,200,55), my_rect)        
        rendered_text = render_textrect(my_string, my_font, my_rect, (216, 216, 216), 0, cursor_pos)

        if rendered_text:
            display.blit(rendered_text, my_rect.topleft)

        clock.tick(10)
        pygame.display.update()
        

    while not pygame.event.wait().type in (QUIT, KEYDOWN):
        pass
