#!/usr/bin/python

#
#   System imports
#
from random import randint
import threading
import string
import os, sys, pygame, math
from pygame.locals import *

#
#   Zap's Helper Modules
#
import zmaps       # Map display code
import ztext       # Wrapping text + cursor display code
import zconfig     # Configuration file management
import zgps        # Abstraction of the GPS module
import zbrowse     # OS abstraction for stuff like opening webbpages and popping dialogs
import zdaemon     # The Daemon module
import zgui        # The GUI module
import zspin       # The "please wait" spinner module

#
#   The various services we support
#
import zzaloc      # The ZapLoc (meta) service
import zgowalla    # Gowalla service
import zlatitude   # Google Latitude (and in future "places") service
import zfoursquare # FourSquare service
import zfacebook   # Facebook Places service

from   zdaemon   import       meter_distance, daemonize

#
#   Various messages the dialogs send
#
SHOW_ON_WEB  = 1
CONSOLIDATE  = 2
CHECK_IN     = 3
TWITTERON    = 4
FBON         = 5
CFG_SERVICES = 6
CFG_MAPMODE  = 7
CFG_DAEMON   = 8
CFG_DAEMONON = 9
CFG_CREATE_SPOT   = 10
CFG_DELETE_SPOT   = 11
CFG_IMAGE_CHANGED = 12
EDIT_SPOT         = 13
ADJUST_LOCATION   = 14
CFG_AUTOCHECK     = 15

thread_busy   = None
thread_result = []

image_cache = dict()

   

# Make a nifty image for me
def make_adjust_image(image):
    img = image.copy()
    
    y = 0
    while y < img.get_height():
        x = 0
        while x < img.get_width():
            color = img.get_at((x,y))
            
            img.set_at((x,y), ( color[0], color[1], color[2], color[3] * 0.25 ))
            
            x += 1
        y += 1                
    
    return img

def load_cached_image( path, scale ):
    global image_cache
    if path in image_cache:
        return image_cache[path]
    
    try:
        img = pygame.image.load(path).convert_alpha()
        if scale == 2.0:
            img = pygame.transform.scale2x(img)
        elif scale != 1.0:
            img = pygame.transform.rotozoom(img, 0.0, scale)                

        image_cache[path] = img
        return img
    except:
        print "Error loading image '" + path + "'!"
        try:
            return pygame.image.load("gfx/unknown.png").convert_alpha()
        except:
            return pygame.Surface((20,20))
        

def bottom_center_blit(screen, image, pos):
    r = image.get_rect()
    screen.blit(image, (pos[0] - r.width / 2, pos[1] - r.height))

def center_blit(screen, image, pos):
    r = image.get_rect()
    screen.blit(image, (pos[0] - r.width / 2, pos[1] - r.height / 2))        

def new_spot_effect(screen, pxl, time):
    img = pygame.image.load("gfx/glowy.png")
    img = pygame.transform.rotozoom(img, time * 2, 1.0 + time / 10.0)

    center_blit(screen, img, ( pxl[0], pxl[1]-30 ))    
    
def edit_spot(screen, clicked_object, svcs, new = False):
    result = None

    spotidx = -1
    
    if zzaloc.database != None:
        for i in range(len(zzaloc.database)):
            if zzaloc.database[i]["id"] == clicked_object[0]:
                spotidx = i
                break        
            
    if spotidx == -1:
        zbrowse.dialog("Error: Can't find spot in database!")
        return

    spot = zzaloc.database[spotidx]
    
    background = screen.copy()    
    window = pygame.image.load("gfx/consolidate.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = "ZapLoc settings for " + clicked_object[3]
    my_rect = pygame.Rect((100, 50, 600, 100))           
    
    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 1, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass
            
    my_font = pygame.font.Font(None, 30)    
    
    screen.set_clip(None)

    twitter_svcs = []
    facebook_svcs = []
    
    for svc in svcs:
        if svc[0].cfgname() in spot:
            if svc[0].tweetable():
                twitter_svcs.append( svc[0].name())
            if svc[0].facebookable():
                facebook_svcs.append( svc[0].name())
            
    
    pos = - 800

    d = spot["auto-distance"]
    
    if d <= 75:
        ditem = 0
    elif d <= 200:
        ditem = 1
    elif d <= 300:
        ditem = 2
    else:
        ditem = 3    
    
    ui = []        
    ui.append(zgui.Checkbox  (80,   120, "Check in Automatically when within", spot["auto-check"],  True,  CFG_AUTOCHECK))  
    ui.append(zgui.PopupList (500,  130, 100, ( "50 m", "100 m", "200 m", "500 m" ), ditem,  True, -1))  
    ui.append(zgui.Checkbox  (150,   170, "Twitter",    spot["auto-tweet"],     True, -1))  
    ui.append(zgui.Checkbox  (300,   170, "Facebook",   spot["auto-facebook"],  True, -1))  

    if new:
        ui.append(zgui.Button    (550, 250, "Cancel",       True,  CFG_DELETE_SPOT))
        ui.append(zgui.Button    (550, 320, "OK",           True,  CFG_CREATE_SPOT))
    else:
        ui.append(zgui.Button    (550, 250, "Delete",       True,  CFG_DELETE_SPOT))
        ui.append(zgui.Button    (550, 320, "Adjust Loc.",  True,  ADJUST_LOCATION))

    def toggle_buttons():
        on = ui[0].on
        ui[1].set_enabled(on)
        ui[2].set_enabled(on)
        ui[3].set_enabled(on)

    toggle_buttons()
        
    checkin_text = spot["auto-message"] + " "        
    cursor_pos   = len(checkin_text) - 1
    
    my_rect = pygame.Rect((160, 250, 350, 100))        
    
    window.fill((255,255,255), my_rect)
    pygame.draw.rect(window, (100,80,80), my_rect, 2)
    
    my_rect.inflate_ip(-10,-10)
    
    ignorekeys = (K_F1,K_F2,K_F3,K_F4,K_F5,K_F6,K_F7,K_F8,K_F9,K_F10,K_F11,K_F12,K_F13,K_F14,K_F15,K_NUMLOCK,K_CAPSLOCK,K_SCROLLOCK,
                  K_RSHIFT,K_LSHIFT,K_RCTRL,K_LCTRL,K_RALT,K_LALT,K_RMETA,K_LMETA,K_LSUPER,K_RSUPER,K_MODE,K_HELP,K_PRINT,K_SYSREQ,K_BREAK,K_MENU,K_POWER,
                  K_TAB,K_RETURN,K_CLEAR,K_ESCAPE,K_PAUSE )
    
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == KEYDOWN and ui[0].on:
                if event.key == K_LEFT:
                    if cursor_pos > 0:
                        cursor_pos -= 1
                elif event.key == K_RIGHT:
                    if cursor_pos < len(checkin_text):
                        cursor_pos += 1
                elif event.key == K_BACKSPACE:
                    if cursor_pos > 0:
                        checkin_text = checkin_text[:cursor_pos-1] + checkin_text[cursor_pos:]
                        cursor_pos -= 1
                elif event.key == K_DELETE:
                    if cursor_pos < len(checkin_text):
                        checkin_text = checkin_text[:cursor_pos] + checkin_text[cursor_pos+1:]                    
                elif event.key in ignorekeys:
                    pass
                else:
                    if len(checkin_text) < 101:
                        checkin_text = checkin_text[:cursor_pos] + event.unicode + checkin_text[cursor_pos:]                                            
                        cursor_pos += 1
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x

                if event_res == CFG_AUTOCHECK:
                    toggle_buttons()

                result = event_res
                    
                if event_res == ADJUST_LOCATION or event_res == CFG_DELETE_SPOT or event_res == CFG_CREATE_SPOT:
                    working = False

                if event_res == None and not my_rect.collidepoint(event.pos) and not new:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))

        srf = my_font.render("Chars left: %d" % ( 101 - len(checkin_text)), True, (150,120,90))
        screen.blit(srf, (160, my_rect.top + 105 + pos))

        if ui[0].on:
            textcol = (0,0,0)
        else:
            textcol = (140,140,140)
        
        rendered_text = ztext.render_textrect(checkin_text, my_font, my_rect, textcol, 0, cursor_pos)        
        if rendered_text:
            screen.blit(rendered_text, ( my_rect.left, my_rect.top + pos))                    
        
        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6
        if pos > -1:
            pos = 0
                
        clock.tick(30)
        pygame.display.flip()                                   

    spot["auto-check"]     = ui[0].on
    spot["auto-message"]   = checkin_text.strip()
    spot["auto-distance"]  = ( 50, 100, 250, 500 ) [ui[1].selected]
    spot["auto-tweet"]     = ui[2].on
    spot["auto-facebook"]  = ui[3].on
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        rendered_text = ztext.render_textrect(checkin_text, my_font, my_rect, (0,0,0), 0, cursor_pos)        
        if rendered_text:
            screen.blit(rendered_text, ( my_rect.left, my_rect.top + pos))                            
        
        for u in ui:
            u.draw(screen, pos)        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   

    zzaloc.database[spotidx] = spot
    zzaloc.put()
        
    return result        

def consolidate(screen, clicked_object, my_pos, center_pos, svc, svcs, desc):
    result = None
    
    background = screen.copy()    
    window = pygame.image.load("gfx/consolidate.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = "Make ZapLoc from " + clicked_object[3]
    my_rect = pygame.Rect((100, 50, 600, 100))       

    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 1, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass

    screen.set_clip(None)

    pos = - 800

    ui = []

    #
    #  Slide screen in
    #
    
    while pos < -1 and working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x
                        
                if event_res == None:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6

        clock.tick(30)
        pygame.display.flip()                                   

    # 
    #   Fill in the UI
    #
        
    y = 100

    svcresults = []

    i = 0        
    for service in svcs:
        if service[0].cfgname() == "zaploc":
            continue

        # For the active service, search from the objects location
        # but for other services search from screen center
        if service[0].cfgname() == svc.cfgname():
            search_lat = clicked_object[1]
            search_lng = clicked_object[2]
        else:
            search_lat = center_pos[0]
            search_lng = center_pos[1]
                    
        zspin.start(screen)                

        try:
            searchresult = service[0].list_spots((search_lat, search_lng), 100)
        except:
            searchresult = []

        def distance(a):            
            dist = meter_distance(search_lat, search_lng, a[1], a[2])
            return dist
        
        searchresult.sort(key=distance)
                        
        zspin.stop()                                
        
        if len(searchresult) > 0:
            
            items = []
            
            for item in searchresult:
                if len(items) < 8:
                    items.append(item[3])        
                    
            chkbox = zgui.Checkbox      (100, y-10,   service[0].name(), True, True, 100 + i)
            popup  = zgui.PopupList     (350, y, 300, items, 0,                True, 200 + i)
            i += 1
                    
            ui.append(chkbox)
            ui.append(popup)

            svcresults.append( ( service[0], searchresult, chkbox, popup, y ) )
                        
            y += 50

        pygame.time.wait(500)
        
        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)
        
        pygame.display.flip()

    #
    #  Fill in rest of UI
    #

    svcnames = []
    
    for service in svcresults:
        svcnames.append(service[0].name())

    y -= 10
    
    anames = svcnames[:]
    anames.append("ZapLoc")    
    image_popup = zgui.PopupList (106,   y+30, 130, anames, 0 , True, CFG_IMAGE_CHANGED)
    ui.append(zgui.StaticText(100,   y, "Icon"))
    ui.append(image_popup)    

    bnames = svcnames[:]
    
    # Skip this for now - confusing! 
    # bnames.append("- None -")
    
    twitter_popup = zgui.PopupList (350,   y+30, 130,  bnames, 0 , True, 1)
    ui.append(zgui.StaticText(344,   y, "Twitter via"))
    ui.append(twitter_popup)    

    facebook_popup = zgui.PopupList (520,   y+30, 130, bnames, 0 , True, 1)
    ui.append(zgui.StaticText(514,   y, "Facebook via"))
    ui.append(facebook_popup)        
    
    i = load_cached_image(svcresults[0][1][0][4], svcresults[0][1][0][5])
    
    image  = zgui.TwoStateButton(146, y+75, i , i, True, 1)
    ui.append(image)

    create = zgui.Button    (550, 350, "Create >>",          True,  CFG_CREATE_SPOT)
    ui.append(create)            

    avg_lat = 0
    avg_lng = 0
    
    #
    #  The main UI loop
    #
        
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x
                        
                if event_res == None:
                    working = False
                elif event_res >= 200 or event_res == CFG_IMAGE_CHANGED: # spot name changed
                    svcnr = image_popup.selected
                    if svcnr < len(svcresults):
                        resnr = svcresults[svcnr][3].selected                    
                        image.onimg = load_cached_image(svcresults[svcnr][1][resnr][4], svcresults[svcnr][1][resnr][5]) 
                    else:
                        image.onimg = pygame.image.load("gfx/zaplocspot.png")
                        
                    image.set_enabled(True)
                elif event_res >= 100: # checkbox changed
                    svcresults[event_res-100][3].set_enabled(svcresults[event_res-100][2].on)
                    # If any of the checkboxes are selected, we can push "create"
                    any = False
                    for svc in svcresults:
                        if svc[2].on:
                            any = True
                    create.set_enabled(any)                    
                elif event_res == CFG_CREATE_SPOT:
                    result = dict()

                    # svcnr = position_popup.selected                        
                    # if svcnr < len(svcresults):
                    avg_lat = 0
                    avg_lng = 0
                    divisor = 0                    
                    description = ""
                    
                    # Loop over all services, building a list
                    for svc in svcresults:
                        if svc[2].on:
                            sel_item = svc[3].selected
                            result[svc[0].cfgname()] = svc[1][sel_item][0]
                            avg_lat  += svc[1][sel_item][1]
                            avg_lng  += svc[1][sel_item][2]
                            divisor  += 1.0
                            description += svc[0].name() + ": " + svc[1][sel_item][3] + "\n"


                    def posting_truename(item):
                        if item < len(svcresults):
                            return svcresults[item][0].name()
                        else:
                            return ""
                        
                    x = posting_truename(twitter_popup.selected)
                    if x != "":
                        description += "Twitter posts by: " + x + "\n"
                    x = posting_truename(facebook_popup.selected)
                    if x != "":
                        description += "Facebook posts by: " + x + "\n"                                                        
                            
                    avg_lat /= divisor
                    avg_lng /= divisor
                        
                    result["lat"] = avg_lat
                    result["lng"] = avg_lng
                                        
                    result["name"]        = clicked_object[3]
                    result["description"] = description
                    result["link"]        = None

                    def posting_name(item):
                        if item < len(svcresults):
                            return svcresults[item][0].cfgname()
                        else:
                            return ""
                        
                    result["fb_poster"]   = posting_name(facebook_popup.selected)
                    result["tw_poster"]   = posting_name(twitter_popup.selected)
                    
                    result["auto-check"]     = False
                    result["auto-message"]   = "ZapLoc Auto-Checkin"
                    result["auto-tweet"]     = False
                    result["auto-facebook"]  = False
                    result["auto-distance"]  = 50
                    
                    svcnr = image_popup.selected
                    if svcnr < len(svcresults):
                        resnr = svcresults[svcnr][3].selected                    
                        result["image"] = svcresults[svcnr][1][resnr][4]
                        result["scale"] = svcresults[svcnr][1][resnr][5]
                    else:
                        result["image"] = "gfx/zaplocspot.png"
                        result["scale"] = 1.0

                    new_spot = zzaloc.add(result)
                    
                    clicked_object = ( new_spot, clicked_object[1], clicked_object[2], clicked_object[3], clicked_object[4] )
                    
                    result = CFG_CREATE_SPOT
                    
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)                                        
                
        clock.tick(30)
        pygame.display.flip()                                           
        
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        for u in ui:
            u.draw(screen, pos)        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   
        
    if result == CFG_CREATE_SPOT:
        res = edit_spot(screen, clicked_object, svcs, True)
        if res == CFG_CREATE_SPOT:
            result = ( avg_lat, avg_lng )
            
        if res == CFG_DELETE_SPOT:
            zzaloc.remove(new_spot)

    return result
    
    
def config_services(screen, config):
    result = None
    
    background = screen.copy()    
    window = pygame.image.load("gfx/consolidate.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = "Set up Services"
    my_rect = pygame.Rect((200, 50, 400, 100))       

    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 1, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass
            
    my_font = pygame.font.Font(None, 30)    
    
    screen.set_clip(None)

    pos = - 800

    ui = []

    SX = 180
    DX = 250
    
    SY = 100
    DY = 65

    svcs = [
        zgowalla.Service(),
        zfoursquare.Service(),
        zfacebook.Service(),
        zlatitude.Service()
    ]

    zspin.start(screen)
    
    for i in range(len(svcs)):        
        ui.append(zgui.Checkbox(SX, SY+i*DY,    svcs[i].name(),    config[svcs[i].cfgname()],    True,  100 + i))
        ui.append(zgui.Button  (SX+DX, SY+i*DY, "Authorize...",    config[svcs[i].cfgname()],    200 + i))
    
        on  = pygame.image.load(svcs[i].onimg()). convert_alpha()
        off = pygame.image.load(svcs[i].offimg()).convert_alpha()
        ui.append(zgui.TwoStateButton(SX-70, SY+i*DY, on, off, svcs[i].is_authorized() and config[svcs[i].cfgname()],  None))    

        on  = pygame.image.load("gfx/lighton.png"). convert_alpha()
        off = pygame.image.load("gfx/lightoff.png").convert_alpha()
        ui.append(zgui.TwoStateButton(SX+430, SY+i*DY-5, on, off, svcs[i].is_authorized(),  None))    
        
    zspin.stop()
        
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x

                for i in range(len(svcs)):        
                    if event_res == 100 + i:
                        r = ui[i*4].on
                        config[svcs[i].cfgname()] = r
                        ui[i*4+1].set_enabled(r)
                        zspin.start(screen)
                        ui[i*4+2].set_enabled(r and svcs[i].is_authorized())
                        ui[i*4+3].set_enabled(svcs[i].is_authorized())
                        zspin.stop()
                        # We need an update
                        result = True
                        
                    if event_res == 200 + i:
                        svcs[i].is_authorized()
                        zspin.start(screen)
                        svcs[i].authorize()
                        ui[i*4+2].set_enabled(svcs[i].is_authorized())
                        ui[i*4+3].set_enabled(svcs[i].is_authorized())
                        zspin.stop()
                        # We need an update
                        result = True
                        
                if event_res == None:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6
        if pos > -1:
            pos = 0
                
        clock.tick(30)
        pygame.display.flip()                                   
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        for u in ui:
            u.draw(screen, pos)        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   
        
    return result

def config_daemon(screen, config):
    result = None
    
    background = screen.copy()    
    window = pygame.image.load("gfx/consolidate.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = "Background Update Settings"
    my_rect = pygame.Rect((200, 50, 400, 100))           
    
    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 1, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass
            
    my_font = pygame.font.Font(None, 30)    
    
    screen.set_clip(None)

    pos = - 800

    ui = []

    ui.append(zgui.StaticText(80,   90, "Start"))
    ui.append(zgui.PopupList (145,  90, 200, ("when app starts", "on device boot"), config["daemon-start-on-boot"] , True, CFG_MAPMODE))    

    ui.append(zgui.StaticText(400,  90, "Stop"))
    ui.append(zgui.PopupList (460,  90, 200, ("when app closes", "never"), config["daemon-never-stop"] , True, CFG_MAPMODE))    
    
    ui.append(zgui.StaticText(80,  140,       "Check if location has changed"))
    ui.append(zgui.PopupList (390, 140, 270, ("every 5 minutes", "every 10 minutes", "every 15 minutes", "every 30 minutes", "every hour", "when cell tower changes" ), config["daemon-interval"] , True, CFG_MAPMODE))    

    if config["latitude"]:
        y = 190
    else:
        y = 220
    
    ui.append(zgui.Checkbox  (80,  y, "Check in Automatically to Auto-Check enabled Spots", config["daemon-autocheck"],  True,  CFG_DAEMONON))  
    ui.append(zgui.StaticText(145, y+50, "after"))    
    ui.append(zgui.PopupList (220, y+50, 125, ("1 interval", "2 intervals", "3 intervals", "4 intervals", "5 intervals" ), config["daemon-autocheck-delay"] , True, CFG_MAPMODE))    

    min_display = zgui.StaticText(380, y+50, "(about x minutes)")
    ui.append(min_display)

    if config["latitude"]:
        ui.append(zgui.Checkbox  (80,  300, "Update Google Latitude", config["daemon-latitude"],  True,  CFG_DAEMONON))  
        ui.append(zgui.PopupList (390, 310, 270, ("when position changes", "at LEAST every hour", "at LEAST every 2 hours", "at LEAST every 6 hours", "at LEAST every day"), config["daemon-latitude-freq"] , True, CFG_MAPMODE))

    def grayout():
        ui[8].set_enabled(ui[6].on)
        if config["latitude"]:
            ui[11].set_enabled(ui[10].on)
        delay = ( 5, 10, 15, 30, 60,  0)[ui[5].selected] * (ui[8].selected + 1)
        if delay:
            min_display.text = "(about %d minutes)" % delay
        else:
            min_display.text = ""
        min_display.set_enabled(True)

    grayout()
    
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x
                        
                # Any other button than disabled buttons (-1)
                # Should trigger a reload
                if event_res != -1:
                    result = True
                    grayout()
                        
                if event_res == None:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6
        if pos > -1:
            pos = 0
                
        clock.tick(30)
        pygame.display.flip()                                   

    
    config["daemon-start-on-boot"]  = ui[1].selected
    config["daemon-never-stop"]     = ui[3].selected

    config["daemon-interval"]        = ui[5].selected
    config["daemon-autocheck"]       = ui[6].on
    config["daemon-autocheck-delay"] = ui[8].selected
                
    if config["latitude"]:
        config["daemon-latitude"]        = ui[10].on
        config["daemon-latitude-freq"]   = ui[11].selected     
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        for u in ui:
            u.draw(screen, pos)        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   
        
    return result


#
# Main setup dialog
#
def config_dialog(screen, config, zmap):
    result = None
    
    background = screen.copy()    
    window = pygame.image.load("gfx/dialog.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = "Settings"
    my_rect = pygame.Rect((200, 100, 400, 100))       

    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 1, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass
            
    my_font = pygame.font.Font(None, 30)    
    
    screen.set_clip(None)

    pos = - 800

    ui = []

    ui.append(zgui.StaticText(180, 150, "Maps"))
    ui.append(zgui.PopupList (250, 150, 200, zmaps.providers, zmap.get_mode(),          True, CFG_MAPMODE))    
    
    ui.append(zgui.Checkbox  (180, 220, "Background Updates", config["daemon-enable"],  True,  CFG_DAEMONON))    
    ui.append(zgui.Button    (455, 220, "Settings...",                                  config["daemon-enable"],  CFG_DAEMON))    

    ui.append(zgui.StaticText(240, 300, "Attached Services"))
    ui.append(zgui.Button    (455, 290, "Settings...",                                  True,  CFG_SERVICES))
    
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x != None:
                        event_res = x

                if event_res == CFG_MAPMODE:
                    zmap.set_mode(ui[1].selected)
                    working = False # Should we?
                        
                if event_res == CFG_SERVICES:
                    result = config_services(screen, config)                    

                if event_res == CFG_DAEMON:
                    result  = config_daemon(screen, config)
                    working = False                    

                if event_res == CFG_DAEMONON:
                    config["daemon-enable"] = ui[2].on
                    ui[3].set_enabled(config["daemon-enable"])                    
                    result = -1
                    
                if event_res == None:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))
        
        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6
        if pos > -1:
            pos = 0
                
        clock.tick(30)
        pygame.display.flip()                                   
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        for u in ui:
            u.draw(screen, pos)        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   
        
    return result


# Allowing searches to be multithreaded
class searcher(threading.Thread):
    def __init__(self, svc, pos, friends):
        self.svc    = svc
        self.pos    = pos
        self.friends= friends
        threading.Thread.__init__(self)
        
    def run(self):
        global thread_result, thread_busy
        thread_busy = self

        # Create a dictionary for de-duping
        d = dict()
        
        # Fill in old results
        for o in thread_result:
            d[o[0]] = o
        
        try:
            if self.friends:
                result = self.svc.list_friends(self.pos)
            else:
                result = self.svc.list_spots(self.pos, 50)        
               
            # Fill in  new results
            for o in result:                
                img = load_cached_image(o[4], o[5])                                                        
                d[o[0]] = ( o[0],o[1],o[2],o[3],img )

            # Create values
            vals = d.values()
                
            # Put 'em in thread_results
            thread_result = vals
                            
        except:
            print "Unexpected error: ", sys.exc_info()[0]
            zbrowse.dialog("Error")
            
        thread_busy = None

# Allowing check-ins to be multithreaded
class checker(threading.Thread):
    def __init__(self, svc, a, b, c, d, e):
        self.svc    = svc
        self.a  = a
        self.b  = b
        self.c  = c
        self.d  = d
        self.e  = e
        
        threading.Thread.__init__(self)
        
    def run(self):
        global thread_busy
        thread_busy = self
        result = self.svc.check_in(self.a, self.b, self.c, self.d, self.e)
        thread_busy = None
        
        if result != None:
            for line in result:
                zbrowse.notification(line)
                pygame.time.wait(3000)
                    
def check_in(svc, a, b, c, d, e):
    checker(svc, a, b, c, d, e).start()                        
        
def check_in_dialog(screen, clicked_object):   
    result = None
    
    background = screen.copy()    
    window = pygame.image.load("gfx/checkin.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    my_string = clicked_object[3]     
    my_rect = pygame.Rect((230, 130, 300, 300))       

    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 0, -1, (120,120,120))    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass
    
    window.blit(clicked_object[4], (220 - clicked_object[4].get_rect().width,105))
        
    my_font = pygame.font.Font(None, 30)    
    
    screen.set_clip(None)

    pos = - 800

    ui = []
    
    ui.append(zgui.Checkbox(550, 180, "Twitter",   True, True,  TWITTERON))
    ui.append(zgui.Checkbox(550, 240, "Facebook",  True, True,  FBON))              
    ui.append(zgui.Button  (550, 310, "Check In",  True,        CHECK_IN))
    
    cursor_pos   = 0
    checkin_text = " "
    
    my_rect = pygame.Rect((160, 190, 350, 100))        
    
    window.fill((255,255,255), my_rect)
    pygame.draw.rect(window, (100,80,80), my_rect, 2)
    
    my_rect.inflate_ip(-10,-10)
    
    ignorekeys = (K_F1,K_F2,K_F3,K_F4,K_F5,K_F6,K_F7,K_F8,K_F9,K_F10,K_F11,K_F12,K_F13,K_F14,K_F15,K_NUMLOCK,K_CAPSLOCK,K_SCROLLOCK,
                  K_RSHIFT,K_LSHIFT,K_RCTRL,K_LCTRL,K_RALT,K_LALT,K_RMETA,K_LMETA,K_LSUPER,K_RSUPER,K_MODE,K_HELP,K_PRINT,K_SYSREQ,K_BREAK,K_MENU,K_POWER,
                  K_TAB,K_RETURN,K_CLEAR,K_ESCAPE,K_PAUSE )
    
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == KEYDOWN:
                if event.key == K_LEFT:
                    if cursor_pos > 0:
                        cursor_pos -= 1
                elif event.key == K_RIGHT:
                    if cursor_pos < len(checkin_text):
                        cursor_pos += 1
                elif event.key == K_BACKSPACE:
                    if cursor_pos > 0:
                        checkin_text = checkin_text[:cursor_pos-1] + checkin_text[cursor_pos:]
                        cursor_pos -= 1
                elif event.key == K_DELETE:
                    if cursor_pos < len(checkin_text):
                        checkin_text = checkin_text[:cursor_pos] + checkin_text[cursor_pos+1:]                    
                elif event.key in ignorekeys:
                    pass
                else:
                    if len(checkin_text) < 101:
                        checkin_text = checkin_text[:cursor_pos] + event.unicode + checkin_text[cursor_pos:]                                            
                        cursor_pos += 1
            if event.type == MOUSEBUTTONUP and event.button == 1:                
                event_res = None
                
                for u in ui:
                    x = u.doevent(event)
                    if x:
                        event_res = x

                if event_res == CHECK_IN:
                    result = ( checkin_text, int(ui[0].on), int(ui[1].on))
                    working = False
                        
                if event_res == None and not my_rect.collidepoint(event.pos):
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0, pos))

        
        srf = my_font.render("Chars left: %d" % ( 101 - len(checkin_text)), True, (150,120,90))
        screen.blit(srf, (160,295+pos))
        
        rendered_text = ztext.render_textrect(checkin_text, my_font, my_rect, (0,0,0), 0, cursor_pos)        
        if rendered_text:
            screen.blit(rendered_text, ( my_rect.left, my_rect.top + pos))            

        for u in ui:
            u.draw(screen, pos)
            
        pos *= 0.6
        if pos > -1:
            pos = 0
                
        clock.tick(30)
        pygame.display.flip()                                   
        
    pos = -5
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                

        for u in ui:
            u.draw(screen, pos)        

        if rendered_text:
            screen.blit(rendered_text, ( my_rect.left, my_rect.top + pos))                        
            
        pos *= 1.5            
            
        clock.tick(30)
        pygame.display.flip()                                   
        
    return result


def show_info(screen, clicked_object, my_pos, center_pos, svc, svcs):

    result = None
    
    try:
        acc = 1500 + abs(5 * my_pos[3])
    except:
        acc = 1500

    checkable = meter_distance(my_pos[0], my_pos[1],
                clicked_object[1], clicked_object[2]) < acc

    #if not checkable:
    #    zbrowse.dialog( "Site is more than %g away, can't check in" % acc)

    background = screen.copy()
    
    window = pygame.image.load("gfx/window.png").convert_alpha()
    
    working = True    

    clock = pygame.time.Clock()

    my_font = pygame.font.Font(None, 32)

    # print clicked_object
    
    my_string = clicked_object[3] 

    # print my_string
    
    my_rect = pygame.Rect((230, 115, 300, 150))    
    
    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 0, -1, (120,120,120))    
    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)    
    except:
        pass

    window.blit(clicked_object[4], (220 - clicked_object[4].get_rect().width,90))
        
    my_font = pygame.font.Font(None, 26)

    zspin.start(screen)
        
    try:
        clock.tick(10)
        desc = svc.get_description(clicked_object)    
    except:
        desc = ( "No description", None )

    zspin.stop()
                
    buttons = []
    
    buttons.append(zgui.Button(200-70, 310, "Show on Web", desc[1],   SHOW_ON_WEB))
    if svc.cfgname() == "zaploc":
        buttons.append(zgui.Button(385-70, 310, "Edit",        True,      EDIT_SPOT))
    else:
        buttons.append(zgui.Button(385-70, 310, "Make ZapLoc", True,      CONSOLIDATE))
    buttons.append(zgui.Button(570-70, 310, "Check In",    checkable, CHECK_IN))

    for b in buttons:
        b.draw(window)
        
        
    my_string = desc[0]
        
    my_rect = pygame.Rect((160, 180, 440, 110))    
    
    try:
        rendered_text = ztext.render_textrect(my_string, my_font, my_rect, (255,255,255), 0)    
    
        if rendered_text:
            window.blit(rendered_text, my_rect.topleft)            
    except:
        pass
    
    screen.set_clip(None)
            
    pos = - 800
    
    while working:
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False
            if event.type == MOUSEBUTTONUP and event.button == 1:                

                do_something = None
                
                for b in buttons:
                    res = b.doevent(event)
                    if res:
                        do_something = res
                
                if do_something == SHOW_ON_WEB: 
                    if desc[1]:
                        zbrowse.open(desc[1])
                        working = False
                elif do_something == CONSOLIDATE: 
                    result = consolidate(screen, clicked_object, my_pos, center_pos, svc, svcs, desc)
                    working = False
                elif do_something == EDIT_SPOT: 
                    result  = edit_spot(screen, clicked_object, svcs)
                    working = False
                elif do_something == CHECK_IN:
                    result = check_in_dialog(screen, clicked_object)
                    if result != None:
                        #wait  = pygame.image.load("gfx/download.png").convert()
                        #screen.set_clip(Rect(450,428,146,40))        
                        #screen.blit(wait, (450 - randint(1,100), 418))
                        #screen.set_clip(None)
                        #pygame.display.flip()

                        check_in(svc, clicked_object, my_pos, result[0], result[1], result[2])
                        result == CHECK_IN
                        working = False
                                                    
                if do_something == None:
                    working = False

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))
        
        pos *= 0.6
                
        clock.tick(30)
        pygame.display.flip()                                   
        
    pos = -5
    
    while pos > -800:
        for event in pygame.event.get():
            pass

        screen.blit(background, (0,0))
        screen.blit(window, (0,pos))                
        pos *= 1.5
                
        clock.tick(30)
        pygame.display.flip()                                   

    return result    

def setup_service(service):
    if not service.is_authorized():
        service.authorize()
    if service.is_authorized():
        return ( 
            service, 
            pygame.image.load(service.onimg ()).convert_alpha(), 
            pygame.image.load(service.offimg()).convert_alpha()
        )
    else:
        return None

def main():
    global thread_busy, thread_result

    os.chdir("/opt/zaploc")
        
    pygame.mixer.init(44100, -16, 2, 512)
    pygame.init()
    pygame.font.init()
    
    info  = pygame.display.Info()
    flags = NOFRAME | HWSURFACE | DOUBLEBUF
    
    screen_x = info.current_w
    screen_y = info.current_h

    if screen_x <= 800 or screen_y <= 480:
        pygame.mouse.set_visible(False)    
        flags |= FULLSCREEN
    
    screen = pygame.display.set_mode((800,480), flags)

    # We need key repeat for typin'
    pygame.key.set_repeat(150, 40)    
    
    logo   = pygame.image.load("gfx/titlecard.png")    
    screen.blit(logo, (0,0))
    pygame.display.flip()    
    
    zoom = 15

    clock  = pygame.time.Clock()        
    
    dot    = pygame.image.load("gfx/dot.png")  .convert_alpha()
    close  = pygame.image.load("gfx/close.png").convert_alpha()

    wait  = pygame.image.load("gfx/download.png").convert()
    setup = pygame.image.load("gfx/setup.png").convert_alpha()
    plus  = pygame.image.load("gfx/plus.png").convert_alpha()
    minus = pygame.image.load("gfx/minus.png").convert_alpha()   

    cross  = pygame.image.load("gfx/crosshair.png").convert_alpha()       
    target = pygame.image.load("gfx/target.png").convert_alpha()       
    
    gpsbut = (
        pygame.image.load("gfx/gpsoff.png").convert_alpha(),
        pygame.image.load("gfx/gpson.png").convert_alpha(),
        pygame.image.load("gfx/gpstrack.png").convert_alpha()
    )
    
    config = zconfig.get()

    zmap = zmaps.mapdisplay(screen)
    zmap.set_mode(config["mapmode"])    
    
    if not config["facebook"] and not config["gowalla"] and not config["facebook"] and not config["latitude"]:
        zbrowse.dialog("You must have at least one of the supported services configured and authorized before you can use ZapLoc")
        config_services(screen, config)

    gpsstate = config["gpsstate"]

    # Get last used position from config
    try:
        zgps.pos = ( config["lat"], config["lng"], 35, 50 )
    except:
        pass
    
    zgps.open()   
    zgps.enable(gpsstate)   
    my_pos = zgps.get_location()    
    
    # List of active services
    svcs = []

    # Load the database 
    zzaloc.get()
    
    x = setup_service(zzaloc.Service())
    if x != None:
        svcs.append(x)
    try:        
        if config["gowalla"]:
            x = setup_service(zgowalla.Service())
            if x != None:
                svcs.append(x)
    except:
        zbrowse.dialog("Error connecting to Gowalla")

    try:
        if config["foursquare"]:
            x = setup_service(zfoursquare.Service())
            if x != None:
                svcs.append(x)
    except:
        zbrowse.dialog("Error connecting to Foursquare")

    try:
        if config["facebook"]:
            x = setup_service(zfacebook.Service())
            if x != None:
                svcs.append(x)
    except:
        zbrowse.dialog("Error connecting to Facebook")

    try:
        if config["latitude"]:
            x = setup_service(zlatitude.Service())
            if x != None:
                 svcs.append(x)
    except:
        zbrowse.dialog("Error connecting to Latitude")

                
    svc = svcs[0][0]


    # Only one service (ZapLoc) - software is useless!!
    if len(svcs) == 1:
        zbrowse.dialog("You must have at least one of the supported services configured and authorized before you can use ZapLoc")

    zspin.start(screen)

    try:
        if config["daemon-enable"]:        
            os.system("/opt/zaploc/zdaemon.py -d")
        else:
            daemonize(False)
    except:
        pass
        
    zspin.stop()
    
    # DEBUG:
    # show_info(screen, ( "Foo", my_pos[0], my_pos[1], "Testy Testlocation", pygame.image.load("gfx/fbplace.png").convert_alpha()), my_pos, svc, svcs)
    # config_dialog( screen, config, zmap ) 
        
    
    switch_to_service = None
    
    working      = True
    direction    = (0,0)
    dragging     = False
    dragging_pin = None

    down_pos = (0,0)
    
    center_pos   = my_pos        

    # print svcs[0][0].list_spots(my_pos, 50)
    
    map_pos = (0,0)

    have_searched = False
    update_count  = 5

    restart = False

    adjust_location = None
    adjust_image    = None
    adjust_id       = None
    unadjust_image  = None

    glow_effect       = None
    glow_effect_count = 0
    
    ui = []
    ui.append(zgui.Button    (200, 420, "OK",       True,  ADJUST_LOCATION))
    ui.append(zgui.Button    (450, 420, "Cancel.",  True,  -1))
    
    while working:        
        for event in pygame.event.get():
            if event.type == QUIT:
                working = False

            if event.type == KEYDOWN:
                update_count = 5
                if event.unicode == '+':
                    if zoom <= zmap.maxzoom(): zoom += 1
                if event.unicode == '-':
                    if zoom >= zmap.minzoom(): zoom -= 1
                if event.unicode == 'n' or event.unicode == 'N':
                    zmap.next_mode()
                    if zoom > zmap.maxzoom(): zoom = zmap.maxzoom()
                    if zoom < zmap.minzoom(): zoom = zmap.minzoom()                    
                
            if event.type == MOUSEBUTTONDOWN and event.button == 1:
                down_pos = event.pos                    
                old_pos  = down_pos
                if event.pos[1] < 420:                    
                    if gpsstate == 0:
                        pxl = zmap.pos2pixel(my_pos)
                        if event.pos[0] > pxl[0] - 30 and event.pos[0] < pxl[0] + 30 and event.pos[1] > pxl[1] - 60 and event.pos[1] < pxl[1]:
                            clickpos = zmap.pixel2pos(event.pos)
                            dragging_pin = ( clickpos[0] - my_pos[0], clickpos[1] - my_pos[1] )
                        else:
                            dragging = True                                                
                    else:
                        dragging = True
                                        
            if event.type == MOUSEBUTTONUP and event.button == 1:
                dx = abs(down_pos[0] - event.pos[0])
                dy = abs(down_pos[1] - event.pos[1]) 
                dragging     = False                
                dragging_pin = None
                update_count = 5

                # Too short to be motion - assume a click
                if dx < 25 and dy < 25:
                      
                    # Check exit button click
                    if event.pos[0] > 740 and event.pos[1] < 50: # close
                        working = False
                    # check button-row clicks
                    elif event.pos[1] > 420:
                        if event.pos[0] > 660 and event.pos[0] < 724:
                            if zoom >= zmap.minzoom(): zoom -= 1
                        elif event.pos[0] > 724 and event.pos[0] < 800:
                            if zoom <= zmap.maxzoom(): zoom += 1
                        elif adjust_location:
                            for u in ui:
                                x = u.doevent(event)
                                if x == -1:
                                    # Snap view to old position
                                    center_pos = ( adjust_location[1], adjust_location[2] )
                                    # Disable adjustment mode
                                    adjust_location = None
                                    adjust_image    = None
                                    adjust_image2   = None
                                    adjust_id       = None
                                if x == ADJUST_LOCATION:
                                    for o in zzaloc.database:
                                        if o["id"] == adjust_location[0]:
                                            o["lat"] = center_pos[0]
                                            o["lng"] = center_pos[1]
                                    zzaloc.put()
                                    # Disable adjustment mode
                                    adjust_location = None
                                    adjust_image    = None
                                    adjust_image2   = None
                                    adjust_id       = None
                                    # Update the icons                                    
                                    switch_to_service = svcs[0][0]
                                    # Add some artsy glow
                                    glow_effect       = ( center_pos[0], center_pos[1] )
                                    glow_effect_count = 20                                
                        elif event.pos[0] > 596 and event.pos[0] < 660:
                            gpsstate += 1
                            if gpsstate > 2: 
                                gpsstate = 0
                            config["gpsstate"] = gpsstate
                            zgps.enable(gpsstate)                            
                        elif event.pos[0] > 380 and event.pos[0] < 380+64:
                            ret = config_dialog(screen, config, zmap)
                            if ret != None:
                                working = False
                                restart = True                                
                        else:
                            x_pos = 0
                            for service in svcs:
                                if event.pos[0] > x_pos and event.pos[0] < x_pos + 64:
                                    switch_to_service = service[0]
                                x_pos += 64
                                                        
                    else: # Above the bottom line and not exit - check objects
                        clicked_object = None
                        
                        if not adjust_location:
                            for spot in thread_result:
                                # spot = item[1]
                                pxl = zmap.pos2pixel((spot[1], spot[2]))
                                if event.pos[0] > pxl[0]-40 and event.pos[0] < pxl[0] + 40 and event.pos[1] > pxl[1]-80 and event.pos[1] < pxl[1]:
                                    clicked_object = spot
                                   
                        if clicked_object != None:
                            direction = ( 0, 0 )
                            
                            ret = show_info(screen, clicked_object, my_pos, center_pos, svc, svcs)
                            
                            if type(ret) == type((0,0)):
                                switch_to_service = svcs[0][0]
                                glow_effect       = ret
                                glow_effect_count = 20
                                update_count      = 20
                                    
                            if ret == ADJUST_LOCATION:
                                adjust_location   = clicked_object
                                adjust_id         = clicked_object[0]
                                adjust_image      = pygame.transform.rotozoom(clicked_object[4], 0.0, 1.2)
                                adjust_image2     = make_adjust_image(clicked_object[4])
                                                                
                                # clicked_object  = ( clicked_object[0], clicked_object[1], clicked_object[2], clicked_object[3], make_adjust_image(clicked_object[4]) )
                                center_pos      = ( clicked_object[1], clicked_object[2] )
                            if ret == CFG_DELETE_SPOT:
                                zzaloc.remove(clicked_object[0])
                                glow_effect       = ( clicked_object[1], clicked_object[2] )
                                glow_effect_count = 20
                                update_count      = 20
                                switch_to_service = svcs[0][0]
                else:
                    have_searched = False
                            
            if event.type == MOUSEMOTION and event.pos[1] < 420: 
                if dragging:
                    map_pos     = zmap.pixel2pos(old_pos)
                    new_map_pos = zmap.pixel2pos(event.pos)                
                    
                    old_pos     = event.pos
                    direction   = ( map_pos[0] - new_map_pos[0], map_pos[1] - new_map_pos[1])                                    
                elif dragging_pin != None:
                    new_pos = zmap.pixel2pos(event.pos)
                    my_pos  = ( new_pos[0] - dragging_pin[0], new_pos[1] - dragging_pin[1], my_pos[2], my_pos[3] )                    
                    update_count = 5

        if gpsstate > 0:
            pxlpos  = zmap.pos2pixel(my_pos)
            my_pos  = zgps.get_location()
            pxlpos2 = zmap.pos2pixel(my_pos)

            # Did the pin move? We need a screen update. Boohoo :)
            if abs(pxlpos[0]-pxlpos2[0]) > 2 or abs(pxlpos[1]-pxlpos2[1]) > 2:
                if pxlpos2[0] > 0 and pxlpos2[0] < 800 and pxlpos2[1] > 0 and pxlpos2[1] < 500:
                    update_count = 1
            
            if dragging: # When dragging, disable tracking
                gpsstate = 1
            if gpsstate == 2: # Tracking
                direction = ( (my_pos[0] - center_pos[0]) * 0.2, (my_pos[1] - center_pos[1]) * 0.2)
                
        screen.set_clip(Rect(0,0,800,420))

        center_pos = ( center_pos[0] + direction[0], center_pos[1] + direction[1])
            
        direction  = ( direction[0] * 0.9, direction[1] * 0.9 )

        if switch_to_service != None and not thread_busy:
            # Time to switch service!
            svc = switch_to_service            
            switch_to_service = None
            # Clear out results from old service
            thread_result    = []
            # Start a new search
            have_searched = True
            searcher(svc, center_pos, False).start()
        
        # Scrolling slows to a crawl: Halt it, and do a search
        if abs(direction[0]) < 0.001 and abs(direction[1]) < 0.001 and not thread_busy and not dragging and not have_searched:
            have_searched = True
            searcher(svc, center_pos, False).start()                                                        
                
        if zmap.set_center(center_pos, zoom) or thread_busy:
            update_count = 10

        if update_count > 0:
            update_count -= 1
            zmap.display()                        

            for spot in thread_result:                
                pxl = zmap.pos2pixel((spot[1], spot[2]))
                if spot[0] == adjust_id:
                    bottom_center_blit(screen, adjust_image2, pxl)
                else:
                    bottom_center_blit(screen, spot[4], pxl)
    
            pxl = zmap.pos2pixel(my_pos)
            bottom_center_blit(screen, dot, pxl)               

            if glow_effect:
                pxl = zmap.pos2pixel(glow_effect)
                new_spot_effect(screen, pxl, glow_effect_count)
                glow_effect_count -= 1
                if glow_effect_count <= 0:
                    glow_effect = None
                update_count = 5
            
            screen.blit(close, (745, 15))           

            if adjust_location:
                bottom_center_blit(screen, adjust_image, (400,210))
                screen.blit(target,(400-50, 220-50))           
            else:
                screen.blit(cross, (400-33, 220-35))           
            
            screen.set_clip(Rect(0,420,800,60))
            screen.fill(Color(220,200,170), Rect(0,420,800,60))

            
            screen.blit(minus, (660, 416))
            screen.blit(plus,  (724,416))
            
            if adjust_location:
                for u in ui:
                    u.draw(screen, 0)
            else:
                screen.blit(gpsbut[gpsstate], (596,416))
                screen.blit(setup, (380,416))
                # Draw active service icons
                xpos = 0            
                for service in svcs:
                    if svc == service[0]:
                        pic = service[1]
                    else:
                        pic = service[2]                
                    screen.blit(pic, (xpos, 416))
                    xpos += 64            

            if thread_busy and not adjust_location:
                screen.set_clip(Rect(450,428,146,40))        
                screen.blit(wait, (450 - randint(1,100), 418))                    
                    
            clock.tick(30)
            pygame.display.flip()               
        else:
            clock.tick(10)
            pygame.display.flip()                           

    config["mapmode"] = zmap.get_mode()

    config["lat"] = center_pos[0]
    config["lng"] = center_pos[1]
    
    if not config["daemon-never-stop"]:
        daemonize(False)    
    
    # Save the configuration    
    zconfig.put(config)
            
    print "Et tu, Brute..."
    
    pygame.display.quit()

    zgps.close()    
    
    return restart

if __name__ == '__main__': 
    while (main()):
        zbrowse.notification("Reloading configuration...")
