import pygame
import ztext
from pygame.locals import *

class UIElement:
    def __init__(self, x, y, width, height):
        self.rect = pygame.Rect(x,y,width,height)        
        
    def doevent(self, event):
        return None
    
    def draw(self, window, yoffset = 0):
        window.blit(self.image, ( self.pos[0], self.pos[1] + yoffset))

class StaticText(UIElement):
    def __init__(self, x, y, text, enabled = True, retcode = None):
        self.retcode   = retcode
        self.pos       = (x+10, y + 10)
        self.text      = text
        self.font      = pygame.font.Font(None, 28)                                        
        
        self.set_enabled(enabled)
        
        self.rect      = self.image.get_rect()
        self.rect.left = x
        self.rect.top  = y    
                        
    def set_enabled(self, enabled):
        self.enabled = enabled    

        if self.enabled:
            textcol = (255,255,255)
        else:
            textcol = (120,120,120)
            
        self.image     = self.font.render(self.text, True, textcol)

    def doevent(self, event):
        if self.rect.collidepoint(event.pos):
            return self.retcode
        return UIElement.doevent(self, event)
        
        
class PopupList(UIElement):
    def __init__(self, x, y, width, items, selected_item, enabled = True, retcode = None):
        self.window    = None
        self.retcode   = retcode
        self.pos       = (x, y)
        self.items     = items
        self.selected  = selected_item
        self.width     = width
        self.arrow     = pygame.image.load("gfx/arrow.png").convert_alpha()
        self.arrowrect = self.arrow.get_rect()
        self.font      = pygame.font.Font(None, 30)                                        
        
        self.set_enabled(enabled)
        
        self.rect      = self.image.get_rect()
        self.rect.left = x
        self.rect.top  = y    
                        
    def set_enabled(self, enabled):
        self.enabled = enabled    

        if enabled:
            textcol = (0,0,0)
            bgcol   = (255,255,255)
        else:
            textcol = (120,120,120)
            bgcol   = (180,180,180)        

        rect = pygame.Rect(0,0, self.width + self.arrowrect.width, self.arrowrect.height)
            
        self.image     = pygame.Surface((self.width + self.arrowrect.width +1 , self.arrowrect.height +1 ), 0, self.arrow)
        self.image.fill(bgcol)
        pygame.draw.rect(self.image, (100,100,100), rect, 2)

        self.image.blit(self.arrow, ( self.width, 0))
        
        my_rect = Rect(0,0,self.width,35)

        srf = self.font.render(self.items[self.selected], True, textcol)                    
        self.image.blit(srf, (10,10), pygame.Rect(0,0,self.width-10,35))
    
    def doevent(self, event):
        if self.rect.collidepoint(event.pos) and self.window != None:
            if not self.enabled:
                return -1
            
            w = self.rect.width
            h = 0
            
            # Compute size of list of texts
            for item in self.items:
                s = self.font.size(item)
                h += s[1] + 20
                if s[0]+10 > w:
                    w = s[0]+20                

            moveup = 475 - ( self.rect.top + self.rect.height + h)
            if moveup > 0:
                moveup = 0
                    
            rect = pygame.Rect((self.rect.left, self.rect.top + self.rect.height + moveup, w, h))
            pygame.draw.rect(self.window, (255,255,255), rect, 0)                        
            pygame.draw.rect(self.window, (0,0,0), rect, 3)            

            org_selected = self.selected
            
            working = True
            while working:

                rs = []
                h = 0
                i = 0
                for item in self.items:
                    if i == self.selected:
                        bgcol = (0,0,0)
                        txcol = ( 255,255, 255 )
                    else:
                        bgcol = ( 255,255,255 )
                        txcol = ( 0,0,0 )
    
                    s   = self.font.size(item)
                    r   = pygame.Rect( rect.left, rect.top + h, w, s[1]+20 )
                    rs.append(r)
                    pygame.draw.rect(self.window, bgcol, r, 0 )
                    srf = self.font.render(item, True, txcol)
                    self.window.blit(srf, (rect.left+10, rect.top + 10 + h))                
                    h += s[1] + 20
                    i += 1

                pygame.display.flip()                
                    
                for event in pygame.event.get():
                    if event.type == QUIT:
                        working = False
                    if (event.type == MOUSEBUTTONUP or
                        event.type == MOUSEBUTTONDOWN or
                        event.type == MOUSEMOTION):
                        i = 0
                        self.selected = org_selected
                        for x in rs:
                            if x.collidepoint(event.pos):
                                self.selected = i
                            i += 1

                        # Only end on mouse-up
                        if event.type == MOUSEBUTTONUP:
                            working = False
                
                    pygame.display.flip()

            # This causes an update
            self.set_enabled(self.enabled)
                                
            return self.retcode

        return UIElement.doevent(self, event)

    def draw(self, window, yoffset = 0):
        self.window = window
        window.blit(self.image, ( self.pos[0], self.pos[1] + yoffset))    
    
        
class Checkbox(UIElement):
    def __init__(self, x, y, text, on, enabled = True, retcode = True):
        self.on        = on
        self.retcode   = retcode
        self.pos       = (x, y)
        self.text      = text
        self.enabled   = enabled
        self.offimg    = pygame.image.load("gfx/togoff.png").convert_alpha()
        self.onimg     = pygame.image.load("gfx/togon.png").convert_alpha()

        if self.on:
            self.image     = self.onimg
        else:
            self.image     = self.offimg
           
        self.rect      = self.image.get_rect()
        self.rect.left = x
        self.rect.top  = y

        self.font        = pygame.font.Font(None, 28)                                
        self.text_width  = self.font.size(self.text)[0]                    
        self.rect.width += self.text_width        
        
    def set_enabled(self, enabled):
        self.enabled = enabled
                    
    def doevent(self, event):
        if self.rect.collidepoint(event.pos):
            if not self.enabled:
                return -1
            self.on = not self.on
            return self.retcode

        return UIElement.doevent(self, event)
            
    def draw(self, window, yoffset = 0):
        if self.on:
            self.image     = self.onimg
        else:
            self.image     = self.offimg

        if self.enabled:
            textcol = (255,255,255)
        else:
            textcol = (120,120,120)            
            
        window.blit(self.image, (self.rect.left, self.rect.top + yoffset))        
        my_rect            = pygame.Rect((self.rect.left + 70,self.rect.top+20,self.text_width+5,50))
        try:
            rendered_text = self.font.render(self.text, True, textcol)                    
            # rendered_text = ztext.render_textrect(self.text, self.font, my_rect, (255,255,255), 0)
            if rendered_text:
                window.blit(rendered_text, ( my_rect.left, my_rect.top + yoffset))  
        except:
            pass

class Button(UIElement):
    def __init__(self, x, y, text, enabled = True, retcode = True):
        self.retcode   = retcode
        self.pos       = (x, y)
        self.text      = text
        self.font      = pygame.font.Font(None, 28)               
        self.set_enabled(enabled)       
        self.rect      = self.image.get_rect()
        self.rect.left = x
        self.rect.top  = y
        
    def set_enabled(self, enabled):
        self.enabled = enabled

        if enabled:
            self.image     = pygame.image.load("gfx/button.png").convert_alpha()
        else:
            self.image     = pygame.image.load("gfx/buttonoff.png").convert_alpha()                    
        
        my_rect        = pygame.Rect((0,18,165,50))

        if enabled:
            textcol = (255,255,255)
        else:
            textcol = (200,200,200)
        
        try:
            rendered_text = ztext.render_textrect(self.text, self.font, my_rect, textcol, 1, -1, (127,127,127))
            if rendered_text:
                self.image.blit(rendered_text, my_rect.topleft)            
        except:
            pass            
        
    def doevent(self, event):
        if self.rect.collidepoint(event.pos):
            if self.enabled:
                return self.retcode
            else:
                return -1
        
        return UIElement.doevent(self, event)

    
class TwoStateButton(UIElement):
    def __init__(self, x, y, onimg, offimg, enabled, retcode = True):
        self.retcode   = retcode
        self.pos       = (x, y)
        self.onimg     = onimg
        self.offimg    = offimg
        self.set_enabled(enabled)       
        self.rect      = self.image.get_rect()
        self.rect.left = x
        self.rect.top  = y

    def set_enabled(self, enabled):
        self.enabled = enabled
        if enabled:
            self.image     = self.onimg
        else:
            self.image     = self.offimg
        
    def doevent(self, event):
        if self.rect.collidepoint(event.pos):
            if self.enabled:
                return self.retcode
            else:
                return -1
        
        return UIElement.doevent(self, event)
    
