#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <sys/stat.h>
#include <sys/types.h>
 
#include <libhildondesktop/libhildondesktop.h>

#include "yellownotes.h"


YNCtx *app=NULL;

void save_applet_prefs(YNCtx *app)
{
	FILE *f=NULL;

	f=fopen(app->conf_file,"w");
	if(f!=NULL)
	{
		fprintf(f,"width=%d\nheight=%d\n",app->applet_x_size,app->applet_y_size);
		fprintf(f,"bgcolor=%d,%d,%d\n",(app->color).red,(app->color).green,(app->color).blue);
		fprintf(f,"bgalpha=%d\n",(gint)((app->bg_alpha)*100));
		fprintf(f,"font=%s\n",app->fontname);
		fprintf(f,"fontcolor=%d,%d,%d\n",(app->fontcolor).red,(app->fontcolor).green,(app->fontcolor).blue);
		fclose(f);
	}
}

void adjust_settings(YNCtx *app)
{
	if(app->applet_x_size<MIN_X_SIZE)
		app->applet_x_size=MIN_X_SIZE;
	if(app->applet_x_size>MAX_X_SIZE)
		app->applet_x_size=MAX_X_SIZE;

	if(app->applet_y_size<MIN_Y_SIZE)
		app->applet_y_size=MIN_Y_SIZE;
	if(app->applet_y_size>MAX_Y_SIZE)
		app->applet_y_size=MAX_Y_SIZE;

	app->bg_alpha=1-(app->bg_alpha/100);
}

void load_applet_prefs(YNCtx *app)
{
	GIOChannel *f=NULL;
	GIOStatus st;
	GdkColormap *colormap;
	gchar *readbuf=NULL;
	gchar *fontbuffer=NULL;
	gchar *homedir=NULL;
	gchar *red=NULL;
	gchar *green=NULL;
	gchar *blue=NULL;

	homedir=(gchar *)getenv("HOME");

	app->prefs_dir=g_new0(gchar,strlen(homedir)+strlen(SETTINGS_PATH)+5);
	sprintf(app->prefs_dir,"%s/%s",homedir,SETTINGS_PATH);	

	app->conf_file=g_new0(gchar,strlen(homedir)+strlen(SETTINGS_PATH)+strlen(PREFS_FILE)+5);
	sprintf(app->conf_file,"%s/%s/%s",homedir,SETTINGS_PATH,PREFS_FILE);

	app->note_file=g_new0(gchar,strlen(homedir)+strlen(SETTINGS_PATH)+strlen(EXISTING_NOTE)+5);
	sprintf(app->note_file,"%s/%s/%s",homedir,SETTINGS_PATH,EXISTING_NOTE);
	
	app->note_number=g_new0(gint,1);
	*app->note_number=1;
	
	app->locked=g_new0(gint,1);
	*app->locked=FALSE;

	g_print("Loading applet preferences\n");
	if(chdir(app->prefs_dir)<0)
	{
		g_print("Creating preferences directory\n");
		/*create settings directory in user's home*/
		if(mkdir(app->prefs_dir,(mode_t)0755) < 0)
			g_print("Error creating settings directory\n");
	}

	g_print("Loading default values\n");
	app->applet_x_size=APPLET_X_SIZE;
	app->applet_y_size=APPLET_Y_SIZE;
	gdk_color_parse (DEFAULT_COLOR, &(app->color));
	gdk_color_parse (DEFAULT_FONT_COLOR, &(app->fontcolor));
	app->fontname=g_new0(gchar,strlen(DEFAULT_FONT)+1);
	strcpy(app->fontname,DEFAULT_FONT);
	app->bg_alpha=(gfloat)DEFAULT_BG_ALPHA/100;

	f=g_io_channel_new_file(app->conf_file,"r",NULL);

	if(f!=NULL)
	{
		while((st=g_io_channel_read_line(f,&readbuf,NULL,NULL,NULL) )!=G_IO_STATUS_EOF)
		{
			if(strstr(readbuf,"width"))
				sscanf(readbuf,"width=%d\n",&(app->applet_x_size));
			if(strstr(readbuf,"height"))
				sscanf(readbuf,"height=%d\n",&(app->applet_y_size));
			if(strstr(readbuf,"bgcolor="))
			{
				red=strchr(readbuf,'=')+1;
				green=strchr(red,',');
				*green=0;
				green+=1;
				blue=strchr(green,',');
				*blue=0;
				blue+=1;
				fontbuffer=strchr(blue,'\n');
				*fontbuffer=0;
				(app->color).red=(guint16)atoi(red);
				(app->color).green=(guint16)atoi(green);
				(app->color).blue=(guint16)atoi(blue);
				colormap=gdk_colormap_get_system();
				gdk_colormap_alloc_color(colormap,&(app->color),TRUE,TRUE);
			}
			if(strstr(readbuf,"bgalpha"))
			{
				gint alpha;
				sscanf(readbuf,"bgalpha=%d\n",&alpha);
				app->bg_alpha=(gfloat)alpha/100;
				
			}
			if(strstr(readbuf,"font="))
			{
				fontbuffer=strchr(readbuf,'\n');
				*fontbuffer=0;
				fontbuffer=strchr(readbuf,'=')+1;
				app->fontname=g_new0(gchar,strlen(fontbuffer)+1);
				strcpy(app->fontname,fontbuffer);
			}
			if(strstr(readbuf,"fontcolor"))
			{
				red=strchr(readbuf,'=')+1;
				green=strchr(red,',');
				*green=0;
				green+=1;
				blue=strchr(green,',');
				*blue=0;
				blue+=1;
				fontbuffer=strchr(blue,'\n');
				*fontbuffer=0;
				(app->fontcolor).red=(guint16)atoi(red);
				(app->fontcolor).green=(guint16)atoi(green);
				(app->fontcolor).blue=(guint16)atoi(blue);
				colormap=gdk_colormap_get_system();
				gdk_colormap_alloc_color(colormap,&(app->fontcolor),TRUE,TRUE);
			}

			g_free(readbuf);
		}
		g_io_channel_close(f);
	}

}

void load_note_text(YNCtx *app)
{
	FILE *f=NULL;
	gchar buffer[1024];
	GtkTextIter start;
	GtkTextIter end;
	gint nread=0;
	gchar *note_path;
	gchar *label;
	
	note_path=g_new0(gchar,strlen(app->note_file)+4);
	sprintf(note_path,"%s%.3i",app->note_file,*app->note_number);	
	
	gtk_text_buffer_get_bounds(GTK_TEXT_BUFFER(app->buffer),&start,&end);
	gtk_text_buffer_delete (GTK_TEXT_BUFFER(app->buffer),&start,&end);
	
	bzero(buffer,1024);
	f=fopen(note_path,"r");
	if(f!=NULL)
	{
		while(!feof(f))
		{
			nread=fread(buffer,1,1024,f);
			gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(app->buffer),&end);
			gtk_text_buffer_insert(GTK_TEXT_BUFFER(app->buffer),&end,buffer,nread);
		}
	}
	else {
		g_print("No previous note %d\n",*app->note_number);
	}	
	
	gtk_text_buffer_set_modified (GTK_TEXT_BUFFER(app->buffer),FALSE);
	
	/*show the number of the current note in the note counter label*/
	label=g_new0(gchar,3);
	sprintf(label,"%.2i",*app->note_number);
	gtk_label_set_text (GTK_LABEL(app->label),label);

}

gboolean edit_note(GObject *o,GdkEventButton *event,YNCtx *app)

{
	GtkWidget	*dialog;
	GtkWidget	*textview;
	
	GtkTextIter start;
	GtkTextIter end;
	gchar *text=NULL;

	GtkScrolledWindow *scrolledwindow;
	GtkTextBuffer	*buffer;

	gint response=0;
	gchar *title=g_new0(gchar,15);

	sprintf(title,"Edit Note %d",*app->note_number);

	dialog=gtk_dialog_new_with_buttons(title,NULL, GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);

	scrolledwindow=GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL,NULL));
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	
	gchar *parseString=g_strconcat("style \"my-style\"\n","{\n","GtkScrolledWindow::scrollbar-spacing = 0\n","}\n","class \"GtkScrolledWindow\" style \"my-style\"\n",NULL);
	gtk_rc_parse_string(parseString);
	gtk_rc_reset_styles(gtk_settings_get_for_screen(gdk_screen_get_default()));

	textview=gtk_text_view_new();
	gtk_text_view_set_editable(GTK_TEXT_VIEW(textview),TRUE);
	gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(textview),TRUE);
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview),GTK_WRAP_WORD);

	buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_get_bounds(GTK_TEXT_BUFFER(app->buffer),&start,&end);
	text=gtk_text_buffer_get_slice(GTK_TEXT_BUFFER(app->buffer),&start,&end,TRUE);
	gtk_text_buffer_set_text (GTK_TEXT_BUFFER(buffer),text,-1);	

	gtk_container_add (GTK_CONTAINER (scrolledwindow),textview);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),GTK_WIDGET(scrolledwindow));
	gtk_widget_show_all(GTK_WIDGET(scrolledwindow));

	response=gtk_dialog_run (GTK_DIALOG(dialog));
	if(response==GTK_RESPONSE_ACCEPT)
	{
		/*save and display new content*/
		save_note_text(app,buffer);
		load_note_text(app);
	}

	gtk_widget_destroy(dialog);
	return(TRUE);

}

gboolean save_note_text(YNCtx *app, GtkTextBuffer *buffer)
{
	GtkTextIter start;
	GtkTextIter end;
	gchar *buffer_text=NULL;
	gchar *note_path;
	
	/* only save, if the text buffer has been changed since the last saving*/
	if (gtk_text_buffer_get_modified (GTK_TEXT_BUFFER(buffer))) {
	
	  note_path=g_new0(gchar,strlen(app->note_file)+5);
	  sprintf(note_path,"%s%.3i",app->note_file,*app->note_number);	

	  gtk_text_buffer_get_bounds(GTK_TEXT_BUFFER(buffer),&start,&end);
	  buffer_text=gtk_text_buffer_get_slice(GTK_TEXT_BUFFER(buffer),&start,&end,TRUE);
	  
	  if(!write_buffer_to_file(buffer_text,note_path))
		  g_print("Error saving note\n");
	  else
		  g_print("Note saved\n");
	  
	}
	
	return(TRUE);
}

int write_buffer_to_file(gchar *buffer,gchar *fn)
{
	FILE *out=NULL;

	out=fopen(fn,"w");
	if(out==NULL)
		return(0);
	fwrite(buffer,strlen(buffer),1,out);
	fclose(out);

	free(buffer);
	return(1);
}

gboolean erase_note(GObject *o,GdkEventButton *event,YNCtx *app)
{
	GtkWidget	*confirmation;
	
	gint response=0;

	confirmation = hildon_note_new_confirmation(GTK_WINDOW(app->frame),"Erase note content?");
	response=gtk_dialog_run(GTK_DIALOG(confirmation));
	gtk_object_destroy(GTK_OBJECT(confirmation));
	
	if ( response == GTK_RESPONSE_OK) {
		gtk_text_buffer_set_text(GTK_TEXT_BUFFER(app->buffer),"",-1);
	}

	return(TRUE);
}

gboolean next_note(GObject *o,GdkEventButton *event,YNCtx *app)
{
	gboolean result = FALSE;
	
	result = save_note_text(app,app->buffer);
	
	if (*app->note_number < 99) {
	  *app->note_number = *app->note_number + 1;
	  load_note_text(app);
	}
	return(TRUE);
}

gboolean previous_note(GObject *o,GdkEventButton *event,YNCtx *app)
{
	gboolean result = FALSE;
	
	result = save_note_text(app,app->buffer);
	
	if (*app->note_number > 1) {
	  *app->note_number = *app->note_number - 1;
	  load_note_text(app);
	}
	return(TRUE);
}


void select_bg_color(GObject *o, YNCtx *app)
{
	GdkColor newcolor;

	hildon_color_button_get_color(HILDON_COLOR_BUTTON(o),&newcolor);
	app->color=newcolor;
}


void select_font(GObject *o, YNCtx *app)
{
	g_free(app->fontname);
	app->fontname=g_new0(gchar,strlen(gtk_font_button_get_font_name(GTK_FONT_BUTTON(o)))+1);
	strcpy(app->fontname,gtk_font_button_get_font_name(GTK_FONT_BUTTON(o)));
}

void select_font_color(GObject *o, YNCtx *app)
{
	GdkColor newcolor;

	hildon_color_button_get_color(HILDON_COLOR_BUTTON(o),&newcolor);
	app->fontcolor=newcolor;
}


void apply_preferences(YNCtx *app)
{
	PangoFontDescription *font;

	app->applet_x_size=atoi(gtk_entry_get_text(GTK_ENTRY(app->entry_width)));
	app->applet_y_size=atoi(gtk_entry_get_text(GTK_ENTRY(app->entry_height)));
	app->bg_alpha=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(app->entry_bg_alpha));
	adjust_settings(app);
	
	gtk_window_resize(GTK_WINDOW(app->parent),app->applet_x_size,app->applet_y_size);
/*	gtk_widget_modify_base(app->textview, GTK_STATE_NORMAL, &(app->color));*/
	gtk_widget_modify_text(app->textview, GTK_STATE_NORMAL, &(app->fontcolor));

	font=pango_font_description_from_string(app->fontname);
	gtk_widget_modify_font(app->textview,font);
	pango_font_description_free(font);	

	gdk_window_invalidate_rect((app->top)->window,&(app->top)->allocation,TRUE);
	save_applet_prefs(app);	

}

void set_cairo_transparency(GdkDrawable *drawable, GdkEventExpose *event, GdkColor color, gfloat alpha)
{
	cairo_t *cr;

	cr = gdk_cairo_create (drawable);
	gdk_cairo_region (cr, event->region);
	cairo_clip (cr);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_set_source_rgba (cr, color.red, color.green, color.blue, alpha);
	cairo_paint (cr);
  
	cairo_destroy (cr);
}


static gboolean widget_expose_event(GtkWidget *widget, GdkEventExpose *event, YNCtx *app)
{
	set_cairo_transparency(GDK_DRAWABLE(widget->window),event,app->color,app->bg_alpha);
	return FALSE;
}

static gboolean textarea_expose_event(GtkWidget *widget, GdkEventExpose *event, YNCtx *app)
{
	set_cairo_transparency(GDK_DRAWABLE(widget->window),event,app->color,app->bg_alpha);

	app->textwindow=gtk_text_view_get_window (GTK_TEXT_VIEW (app->textview),GTK_TEXT_WINDOW_TEXT);
	if (app->textwindow != NULL)
		set_cairo_transparency(GDK_DRAWABLE(app->textwindow),event,app->color,app->bg_alpha);

	return FALSE;
}


gboolean configure_applet_button(GObject *o,GdkEventButton *event,YNCtx *app)
{
	return (configure_applet(NULL,app));
}

gboolean configure_applet(GtkWidget *o,YNCtx *app)
{
	GtkWidget *dialog=NULL;
	GtkWidget *hbox=NULL;
	GtkWidget *label=NULL;
	GtkWidget *button=NULL;
	GtkWidget *fbutton=NULL;
	GtkWidget *fcbutton=NULL;
	gchar value[4];

	gint response=0;

	dialog=gtk_dialog_new_with_buttons("Configuration",NULL, GTK_DIALOG_MODAL, GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,GTK_STOCK_CANCEL,GTK_RESPONSE_CANCEL,NULL);

	hbox=gtk_hbox_new(TRUE,5);
		label=gtk_label_new("Width:");
		gtk_container_add (GTK_CONTAINER (hbox),label);
	
		app->entry_width=gtk_entry_new();
		sprintf(value,"%d",app->applet_x_size);
		gtk_entry_set_text(GTK_ENTRY(app->entry_width),value);
		gtk_entry_set_max_length(GTK_ENTRY(app->entry_width),3);
		gtk_entry_set_width_chars(GTK_ENTRY(app->entry_width),3);
		hildon_gtk_entry_set_input_mode (GTK_ENTRY(app->entry_width),HILDON_GTK_INPUT_MODE_NUMERIC);

		gtk_container_add (GTK_CONTAINER (hbox),app->entry_width);

		label=gtk_label_new("Height:");
		gtk_container_add (GTK_CONTAINER (hbox),label);
	
		app->entry_height=gtk_entry_new();
		sprintf(value,"%d",app->applet_y_size);
		gtk_entry_set_text(GTK_ENTRY(app->entry_height),value);
		gtk_entry_set_max_length(GTK_ENTRY(app->entry_height),3);
		gtk_entry_set_width_chars(GTK_ENTRY(app->entry_height),3);
		hildon_gtk_entry_set_input_mode (GTK_ENTRY(app->entry_height),HILDON_GTK_INPUT_MODE_NUMERIC);
		gtk_container_add (GTK_CONTAINER (hbox),app->entry_height);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),hbox);
	gtk_widget_show_all(hbox);

	hbox=gtk_hbox_new(TRUE,5);
		label=gtk_label_new("Background Color:");
		gtk_container_add (GTK_CONTAINER (hbox),label);

		button=hildon_color_button_new();
		gtk_widget_set_size_request (GTK_WIDGET(button),50,-1);
		gtk_container_add(GTK_CONTAINER(hbox),button);
		hildon_color_button_set_color(HILDON_COLOR_BUTTON(button),&(app->color));
		g_signal_connect(G_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(select_bg_color),app);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),hbox);	
	gtk_widget_show_all(hbox);
	
	hbox=gtk_hbox_new(TRUE,5);
		label=gtk_label_new("Transparency:");
		gtk_container_add (GTK_CONTAINER (hbox),label);

		app->entry_bg_alpha=hildon_number_editor_new(0, 100);
		gtk_widget_set_size_request (GTK_WIDGET(app->entry_bg_alpha),150,-1);
		hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(app->entry_bg_alpha),(gint)(100-(app->bg_alpha*100)));
		gtk_container_add(GTK_CONTAINER(hbox),app->entry_bg_alpha);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),hbox);	
	gtk_widget_show_all(hbox);

	hbox=gtk_hbox_new(TRUE,5);
		label=gtk_label_new("Text Font:");
		gtk_container_add (GTK_CONTAINER (hbox),label);

		fbutton=gtk_font_button_new_with_font(app->fontname);
		gtk_container_add(GTK_CONTAINER(hbox),fbutton);
		g_signal_connect(G_OBJECT(fbutton),"font-set",GTK_SIGNAL_FUNC(select_font),app);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),hbox);	
	gtk_widget_show_all(hbox);

	hbox=gtk_hbox_new(TRUE,5);
		label=gtk_label_new("Font Color:");
		gtk_container_add (GTK_CONTAINER (hbox),label);

		fcbutton=hildon_color_button_new();
		gtk_container_add(GTK_CONTAINER(hbox),fcbutton);
		gtk_widget_set_size_request (GTK_WIDGET(fcbutton),50,-1);
		hildon_color_button_set_color(HILDON_COLOR_BUTTON(fcbutton),&(app->fontcolor));
		g_signal_connect(G_OBJECT(fcbutton),"clicked",GTK_SIGNAL_FUNC(select_font_color),app);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),hbox);	
	gtk_widget_show_all(hbox);

	response=gtk_dialog_run (GTK_DIALOG(dialog));
	if(response==GTK_RESPONSE_ACCEPT) {
		/*aply new preferences*/
		apply_preferences(app);
	}

	gtk_widget_destroy(dialog);
	return(TRUE);
}



void build_ui (void)
{
	PangoFontDescription *font;

	app->frame= gtk_frame_new(NULL);
	gtk_container_set_border_width( GTK_CONTAINER( app->frame ), 0);

	app->vbox=gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width( GTK_CONTAINER( app->vbox ), 0);

	app->top=gtk_hbox_new(FALSE,2);

/*	app->conf_image=gtk_image_new_from_stock(GTK_STOCK_PREFERENCES,HILDON_ICON_SIZE_SMALL);
	app->conf_evbox=gtk_event_box_new ();
	gtk_event_box_set_visible_window(GTK_EVENT_BOX(app->conf_evbox),FALSE);
	gtk_container_add (GTK_CONTAINER (app->conf_evbox),app->conf_image);
	g_signal_connect(G_OBJECT(app->conf_evbox),"button_press_event",G_CALLBACK(configure_applet_button),app);
*/	
	app->next_image=gtk_image_new_from_stock(GTK_STOCK_GO_FORWARD,HILDON_ICON_SIZE_SMALL);
	app->next_evbox=gtk_event_box_new ();
	gtk_event_box_set_visible_window(GTK_EVENT_BOX(app->next_evbox),FALSE);
	gtk_container_add (GTK_CONTAINER (app->next_evbox),app->next_image);
	g_signal_connect(G_OBJECT(app->next_evbox),"button_release_event",G_CALLBACK(next_note),app);
	
	app->previous_image=gtk_image_new_from_stock(GTK_STOCK_GO_BACK,HILDON_ICON_SIZE_SMALL);
	app->previous_evbox=gtk_event_box_new ();
	gtk_event_box_set_visible_window(GTK_EVENT_BOX(app->previous_evbox),FALSE);
	gtk_container_add (GTK_CONTAINER (app->previous_evbox),app->previous_image);
	g_signal_connect(G_OBJECT(app->previous_evbox),"button_release_event",G_CALLBACK(previous_note),app);
	
	app->delete_image=gtk_image_new_from_stock(GTK_STOCK_DELETE,HILDON_ICON_SIZE_SMALL);
	app->delete_evbox=gtk_event_box_new ();
	gtk_event_box_set_visible_window(GTK_EVENT_BOX(app->delete_evbox),FALSE);
	gtk_container_add (GTK_CONTAINER (app->delete_evbox),app->delete_image);
	g_signal_connect(G_OBJECT(app->delete_evbox),"button_release_event",G_CALLBACK(erase_note),app);
	
	app->save_image=gtk_image_new_from_stock(GTK_STOCK_EDIT,HILDON_ICON_SIZE_SMALL);
	app->save_evbox=gtk_event_box_new ();
	gtk_event_box_set_visible_window(GTK_EVENT_BOX(app->save_evbox),FALSE);
	gtk_container_add (GTK_CONTAINER (app->save_evbox),app->save_image);
	g_signal_connect(G_OBJECT(app->save_evbox),"button_release_event",G_CALLBACK(edit_note),app);
	
	app->label=gtk_label_new("00");

	app->scrolledwindow=GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL,NULL));
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(app->scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	app->vscroll=gtk_scrolled_window_get_vscrollbar(app->scrolledwindow);
	gtk_widget_set_app_paintable(app->vscroll,TRUE);
	g_signal_connect(G_OBJECT(app->vscroll),"expose-event",G_CALLBACK(widget_expose_event),app);

	gtk_rc_parse_string(scroll_style);
	gtk_rc_reset_styles(gtk_settings_get_for_screen(gdk_screen_get_default()));
/*	gtk_widget_set_size_request ( GTK_WIDGET ((app->scrolledwindow)->vscrollbar), 10, -1 );*/

	app->textview=gtk_text_view_new();
	gtk_text_view_set_editable(GTK_TEXT_VIEW(app->textview),TRUE);
	gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(app->textview),TRUE);
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(app->textview),GTK_WRAP_WORD);

	font=pango_font_description_from_string(app->fontname);
	gtk_widget_modify_font(app->textview,font);
	pango_font_description_free(font);	

	app->buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(app->textview));

	gtk_widget_modify_text(app->textview, GTK_STATE_NORMAL, &(app->fontcolor));
	font=pango_font_description_from_string(app->fontname);
	gtk_widget_modify_font(app->textview,font);
	pango_font_description_free(font);	

	gtk_box_pack_start(GTK_BOX(app->vbox),app->top,FALSE,FALSE,0);
/*	gtk_box_pack_end(GTK_BOX(app->top),app->conf_evbox,FALSE,FALSE,2);*/
	gtk_box_pack_end(GTK_BOX(app->top),app->delete_evbox,FALSE,FALSE,2);
	gtk_box_pack_end(GTK_BOX(app->top),app->save_evbox,FALSE,FALSE,2);
	gtk_box_pack_end(GTK_BOX(app->top),app->next_evbox,FALSE,FALSE,2);
	gtk_box_pack_end(GTK_BOX(app->top),app->label,FALSE,FALSE,2);
	gtk_box_pack_end(GTK_BOX(app->top),app->previous_evbox,FALSE,FALSE,2);

	gtk_container_add (GTK_CONTAINER (app->vbox), GTK_WIDGET(app->scrolledwindow));
	gtk_container_add (GTK_CONTAINER (app->scrolledwindow),app->textview);

 	gtk_container_set_border_width (GTK_CONTAINER (app->textview), 0);
  	gtk_container_set_border_width (GTK_CONTAINER (app->scrolledwindow), 0);
  	gtk_container_set_border_width (GTK_CONTAINER (app->frame), 0);

	g_signal_connect(G_OBJECT(app->textview),"expose-event",G_CALLBACK(textarea_expose_event),app);
  	app->align = gtk_alignment_new (0.5,0.5,1.0,1.0);

  	gtk_container_add (GTK_CONTAINER (app->frame), app->align);
	gtk_container_add( GTK_CONTAINER( app->align ), app->vbox );

	gtk_widget_show_all (GTK_WIDGET (app->frame)); 
	
	
}

static void yellownotes_plugin_init (YellowNotesPlugin *desktop_plugin)
{

	app = g_new0 (YNCtx, 1);

	load_applet_prefs(app);
	build_ui ();
	load_note_text(app);
	
	gtk_container_set_border_width( GTK_CONTAINER( desktop_plugin ), 0);
	gtk_container_add (GTK_CONTAINER (desktop_plugin), app->frame);
	
	/* Specify that a settings button should be shown in layout mode, 
	* and handle a request to configure the settings: */
	hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM(desktop_plugin),TRUE);
	g_signal_connect (desktop_plugin, "show-settings",G_CALLBACK (configure_applet), app); 
	gtk_window_resize( GTK_WINDOW (desktop_plugin), app->applet_x_size,app->applet_y_size);

	app->parent=desktop_plugin;
} 

static void yellownotes_plugin_class_init (YellowNotesPluginClass *class) {
        GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

        widget_class->realize = yellownotes_plugin_realize;
	widget_class->expose_event = yellownotes_plugin_expose_event;
} 

static void yellownotes_plugin_class_finalize (YellowNotesPluginClass *class) {} 

static void yellownotes_plugin_realize (GtkWidget *widget)
{
        GdkScreen *screen;

        screen = gtk_widget_get_screen (widget);
        gtk_widget_set_colormap (widget,gdk_screen_get_rgba_colormap (screen));

        gtk_widget_set_app_paintable (widget,TRUE);

        GTK_WIDGET_CLASS (yellownotes_plugin_parent_class)->realize (widget);

}

static gboolean yellownotes_plugin_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
	cairo_t *cr;

	cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
	gdk_cairo_region (cr, event->region);
	cairo_clip (cr);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_set_source_rgba (cr, (app->color).red, (app->color).green, (app->color).blue, app->bg_alpha);
	cairo_paint (cr);
  
	cairo_destroy (cr);

	return GTK_WIDGET_CLASS (yellownotes_plugin_parent_class)->expose_event (widget,event);

}
