#include <glib.h>
#include <glib/gstdio.h>
#include <stdio.h>
#include <hildon/hildon.h>
#include <despotify.h>
#include <libplayback/playback.h>
#include <glib-object.h>
#include <conic.h>
#include "string.h"

#include "playlist.h"

#include "audio.h"
#include "glue.h"


#define GLUE_STR_ERROR "<error>"

#define YASPOT_CONFIG_DIR "/home/user/.yaspot"
#define YASPOT_CONFIG_PLAYLIST YASPOT_CONFIG_DIR "/playlist"

typedef struct state_callback_t {
    yaspot_glue_state_changed_t state_cb;
    gpointer state_cb_data;
} state_callback_t;

struct yaspot_glue_t {
    struct despotify_session *session;
    yaspot_audio_t *audio;

    gchar *username;
    gchar *password;
    gboolean high_bitrate;

    /* dbus & libplayback */
    DBusConnection *dbus;
    pb_playback_t *pb;

    /* list of state callbacks */
    GList *state_cb;

    yaspot_glue_timer_update_t timer_cb;
    gpointer timer_cb_data;

    /* global reference counter, mostly for debugging */
    gint all_refs;

    /* control thread and async queue */
    GThread *control_thread;
    GAsyncQueue *control_queue;

    /* internal thread */
    GThread *thread;
    GMutex *mutex;
    GCond *wait;
    GCond *pause_wait;

    /* data behind data_lock */

    yaspot_glue_response_t *current; /* current play queue */
    struct track *current_track; /* currently playing track */
    gint current_track_num; /* nth track in playlist, current */
    gboolean playing; /* internal play states */
    gboolean paused;

    /* data behind mutex */
    enum {
        GLUE_PAUSE,
        GLUE_PLAY,
        GLUE_EXIT
    } play_state;

    ConIcConnection *conic;
    gboolean conic_connected;
    yaspot_state_t client_state;

    /* hackish ..*/
    gboolean playback_starting;

    GList *playlist_history;
};

enum control_command_t {
    CONTROL_RECONNECT = 1,
    CONTROL_RECONNECT_CONTINUE,
    CONTROL_DISCONNECT,
    CONTROL_STORED_LISTS,
    CONTROL_GET_LIST,
    CONTROL_GET_ALBUM,
    CONTROL_PLAY,
    CONTROL_PAUSE,
    CONTROL_NEXT,
    CONTROL_PREV,
    CONTROL_QUIT,
};
typedef enum control_command_t control_command_t;

typedef struct control_command_msg {
    control_command_t command;
    gint argument;
    gchar *argument_string;
    yaspot_glue_response_cb response_cb;
    gpointer data;
} control_command_msg;

/* parameters:
 * glue
 * control command
 * integer argument
 * string argument
 * response callback
 * userdata
 */
static void new_command(yaspot_glue_t *t, control_command_t c, gint a, gchar *a_string, yaspot_glue_response_cb cb, gpointer d) {
    control_command_msg *msg;

    msg = g_new0(control_command_msg, 1);
    msg->command = c;
    msg->argument = a;
    if (a_string)
        msg->argument_string = g_strdup(a_string);
    msg->response_cb = cb;
    msg->data = d;

    g_async_queue_push(t->control_queue, (gpointer)msg);
}

static void del_command(control_command_msg *msg) {
    g_free(msg);
}

static GStaticRWLock data_lock = G_STATIC_RW_LOCK_INIT;

struct yaspot_glue_response_internal_t {
    yaspot_glue_t *glue;

    struct search_result *search;
    struct playlist *pl;
    struct artist_browse *artist;
    struct album_browse *album;

    struct track *tracks;

    /* this is only for the special case of stored
     * playlist (might be other uses as well).
     * when this is set, when creating child object, ref parent
     * so that parent is not removed before child is. */
    struct yaspot_glue_response_t *parent;

    /* start with 1, when 0 free object */
    gint ref;
};


/* static declarations */
static struct track* get_nth_track(struct track *tracks, const gint num);
static void session_cb(struct despotify_session *session, gint signal, void *data, void *userdata);
static yaspot_glue_response_t* res_type(yaspot_glue_t *t, gint type);
static yaspot_playlist_entry_t* playlist_entry_new(const gchar *name, const gchar *author, const gchar *id);
static gint playlist_to_disk(yaspot_glue_t *t);
static gint playlist_from_disk(yaspot_glue_t *t);


static void thread_play(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    if (t->play_state != GLUE_PLAY) {
        t->play_state = GLUE_PLAY;
        t->playing = TRUE;
        t->paused = FALSE;
        g_cond_signal(t->wait);
    }
    g_mutex_unlock(t->mutex);
}

static void thread_pause(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    if (t->play_state != GLUE_PAUSE) {
        t->play_state = GLUE_PAUSE;
        t->paused = TRUE;
        g_print("thread: set pause, block while setting..\n");
        g_cond_wait(t->pause_wait, t->mutex);
        g_print("thread: set pause done.\n");
    }
    g_mutex_unlock(t->mutex);
}

static void thread_exit(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    t->play_state = GLUE_EXIT;
    g_cond_signal(t->wait);
    g_mutex_unlock(t->mutex);

    g_thread_join(t->thread);
}

static void set_current(yaspot_glue_t *t, yaspot_glue_response_t *r) {
    g_static_rw_lock_writer_lock(&data_lock);
    g_assert(t);
    t->current = r;
    g_static_rw_lock_writer_unlock(&data_lock);
}

static void set_current_track(yaspot_glue_t *t, struct track *new_tr) {
    struct track *tr;
    gint i;
    g_assert(t);
    g_static_rw_lock_writer_lock(&data_lock);

    if (!new_tr) {
        t->current_track = NULL;
        t->current_track_num = -1;
    } else {
        g_assert(t->current);

        for (i = 1, tr = t->current->priv->tracks; tr; i++, tr = tr->next) {
            if (memcmp(new_tr->track_id, tr->track_id, 33) == 0) {
                t->current_track = tr;
                t->current_track_num = i;
                break;
            }
        }
    }
    g_static_rw_lock_writer_unlock(&data_lock);
}

static void* thread_loop(void *arg) {
    yaspot_glue_t *t = (yaspot_glue_t*)arg;
    struct pcm_data pcm;
    gint loop = 1;

    while (loop) {
        //g_print("switch (%d): ", t->play_state);
        switch (t->play_state) {
            case GLUE_PAUSE: {
                g_mutex_lock(t->mutex);
                g_print("thread: PAUSE, sleeping\n");
                g_cond_signal(t->pause_wait);
                g_cond_wait(t->wait, t->mutex);
                g_print("thread: PAUSE, woke up\n");
                g_mutex_unlock(t->mutex);
                break;
            }

            case GLUE_PLAY: {
                if (!t->conic_connected) {
                    g_print("thread_loop: conic disconnected while playing, try to pause\n");
                    g_mutex_lock(t->mutex);
                    t->play_state = GLUE_PAUSE;
                    t->playing = FALSE;
                    t->paused = TRUE;
                    g_mutex_unlock(t->mutex);
                    continue;
                }

                gint rc = despotify_get_pcm(t->session, &pcm);
                if (rc == 0) {
                    yaspot_audio_write(t->audio, pcm.buf, pcm.len);
                    //g_print("audio_write %d\n", pcm.len);
                }
                else
                    g_print("thread: despotify_get_pcm() failed\n");
                break;
            }

            case GLUE_EXIT: {
                g_print("thread: EXIT\n");
                loop = 0;
                break;
            }

        }
    }

    return NULL;
}

static void emit_state_changed(yaspot_glue_t *t, yaspot_playback_state_t extra_flags) {
    state_callback_t *s;
    GList *i;
    yaspot_playback_state_t pb_state;
    g_assert(t);

    pb_state = YASPOT_NO_STATE;
    if (t->playing)
        pb_state |= YASPOT_PLAYING;
    if (t->paused)
        pb_state |= YASPOT_PAUSED;
    pb_state |= extra_flags;

    for (i = g_list_first(t->state_cb); i; i = g_list_next(i)) {
        s = (state_callback_t*)i->data;
        g_print("emit state %p data %p\n", s->state_cb, s->state_cb_data);
        s->state_cb(t->client_state, pb_state, s->state_cb_data);
    }
}

static void emit_timer_update(yaspot_glue_t *t, gint now, gint length) {
    g_assert(t);

    if (t->timer_cb) {
        t->timer_cb(t, now, length, t->timer_cb_data);
    }
}

static void control_reconnect(yaspot_glue_t *t) {

    if (t->session) {
        despotify_exit(t->session);
        t->session = NULL;
    }
    t->session = despotify_init_client(session_cb, t, t->high_bitrate);

    if (t->session && despotify_authenticate(t->session, t->username, t->password)) {
        g_print("authentication successfull\n");
        t->client_state = YASPOT_CONNECTED;
    } else {
        g_print("authentication failed: %s\n", despotify_get_error(t->session));
        t->client_state = YASPOT_AUTH_FAILED;
    }

    emit_state_changed(t, 0);
}

static void control_play(yaspot_glue_t *t, yaspot_glue_response_t *r, gint track_no) {
    struct track *tr;
    yaspot_glue_response_t *to_unref = NULL;
    gint ret = -1;

    g_static_rw_lock_writer_lock(&data_lock);
    if (r->priv->glue->current) {
        r->priv->glue->playing = FALSE;

        to_unref = r->priv->glue->current;
        r->priv->glue->current = NULL;
        thread_pause(r->priv->glue);
    }
    g_static_rw_lock_writer_unlock(&data_lock);

    tr = get_nth_track(r->priv->tracks, track_no-1);

    if (!tr) {
        set_current(r->priv->glue, NULL);
        set_current_track(r->priv->glue, NULL);
    }
    else if (despotify_play(r->priv->glue->session, tr, TRUE)) {
        set_current(r->priv->glue, r);
        yaspot_glue_response_ref(r->priv->glue->current);
        g_print("playing %s\n", tr->title);

        /* FIXME why does policy cork us if we request PLAY :( */
        //pb_playback_req_state(r->glue->pb, PB_STATE_PLAY, playback_req_cb, r->glue);

        if (!t->playing)
            thread_play(r->priv->glue);

        ret = 0;
    } else {
        g_print("yaspot_glue_play failed\n");
    }

    if (to_unref)
        yaspot_glue_response_unref(to_unref);
}

static void control_stored_lists(yaspot_glue_t *t, control_command_msg *msg) {
    struct playlist *pl;
    struct playlist *i;
    yaspot_playlist_entry_t *e;
    gint success = 0;

    pl = despotify_get_stored_playlists(t->session);

    if (pl) {
        for (i = pl; i; i = i->next) {
            e = playlist_entry_new(i->name, i->author, i->playlist_id);
            t->playlist_history = g_list_append(t->playlist_history, e);
        }

        despotify_free_playlist(pl);
        success = 1;
    }

    if (msg->response_cb)
        msg->response_cb(t, GINT_TO_POINTER(success), msg->data);
}

static void control_get_list(yaspot_glue_t *t, control_command_msg *msg) {
    yaspot_glue_response_t *response = NULL;
    struct playlist *pl = NULL;

    pl = despotify_get_playlist(t->session, msg->argument_string);

    if (pl) {
        response = res_type(t, GLUE_LIST);
        response->priv->pl = pl;
        response->priv->tracks = pl->tracks;
    }

    msg->response_cb(t, response, msg->data);
}

static void control_get_album(yaspot_glue_t *t, control_command_msg *msg) {
    struct album_browse *album;
    yaspot_glue_response_t *response = NULL;

    album = despotify_get_album(t->session, msg->argument_string);

    if (album) {
        response = res_type(t, GLUE_ALBUM);
        response->priv->album = album;
        response->priv->tracks = album->tracks;
        g_print("glue_album FOUND\n");
    }

    msg->response_cb(t, response, msg->data);
}

static void* control_thread_loop(gpointer userdata) {
    yaspot_glue_t *t = (yaspot_glue_t*)userdata;
    gint loop = 1;
    control_command_msg *msg;

    g_async_queue_ref(t->control_queue);

    while (loop) {

        /* when control thread is started, immediately pause and wait
         * for control commands.
         */
        msg = (control_command_msg*)g_async_queue_pop(t->control_queue);

        g_print("control_thread: got command %u and argument %u\n", msg->command, msg->argument);

        switch (msg->command) {
            case CONTROL_RECONNECT:
                /* first signal we're connecting */
                t->client_state = YASPOT_CONNECTING;
                emit_state_changed(t, 0);

                if (!t->conic_connected) {
                    con_ic_connection_connect(t->conic, CON_IC_CONNECT_FLAG_NONE);
                    break;
                }
                /* else -> let execution flow to reconnect continue */
                g_print("control_thread: continue to %u\n", CONTROL_RECONNECT_CONTINUE);

            case CONTROL_RECONNECT_CONTINUE:
                control_reconnect(t);
                break;

            case CONTROL_PLAY:
                t->playback_starting = TRUE;
            case CONTROL_NEXT:
            case CONTROL_PREV:
                control_play(t, msg->data, msg->argument);
                break;

            case CONTROL_STORED_LISTS:
                control_stored_lists(t, msg);
                playlist_to_disk(t);
                break;

            case CONTROL_GET_LIST:
                control_get_list(t, msg);
                break;

            case CONTROL_GET_ALBUM:
                control_get_album(t, msg);
                break;

            case CONTROL_DISCONNECT:
            case CONTROL_PAUSE:
                g_print("not implemented\n");
                break;

            case CONTROL_QUIT: {
                loop = 0;
                break;
            }
        }

        if (msg) {
            del_command(msg);
            msg = NULL;
        }

    }

    g_async_queue_unref(t->control_queue);

    g_print("control_thread: exiting\n");

    return NULL;
}

static void session_cb(struct despotify_session *session, gint signal, void *data, void *userdata) {
    yaspot_glue_t *t = (yaspot_glue_t*)userdata;
    static gint last = 0;

    switch (signal) {
        case DESPOTIFY_NEW_TRACK: {
            struct track *tr = (struct track*)data;
            g_print("new track name: %s bitrate: %d\n", tr->title, tr->file_bitrate);
            set_current_track(t, tr);
            thread_play(t);
            if (t->playback_starting) {
                emit_state_changed(t, YASPOT_NEW_TRACK | YASPOT_START_PLAYBACK);
                t->playback_starting = FALSE;
            } else {
                emit_state_changed(t, YASPOT_NEW_TRACK);
            }
            last = 0;
            break;
        }
        case DESPOTIFY_TIME_TELL: {
            double elapsed_time = *(double*)data;
            if ((gint)elapsed_time > last) {
                last = (gint)elapsed_time;
                emit_timer_update(t, (gint)elapsed_time, t->current_track->length/1000);
            }
            break;
        }

        case DESPOTIFY_END_OF_PLAYLIST: {
            g_print("end_of_playlist\n");
            set_current_track(t, NULL);
            emit_state_changed(t, 0);
            break;
        }

        default: {
            g_print("default - should never be here\n");
            break;
        }
    }
}

int yaspot_glue_response_unref(yaspot_glue_response_t *r) {
    if (!r)
        return -1;

    r->priv->ref--;
    r->priv->glue->all_refs--;

    g_print("unref %p (%d)\n", r, r->priv->ref);

    if (r->priv->ref > 0)
        return r->priv->ref;

    if (r->priv->parent)
        yaspot_glue_response_unref(r->priv->parent);

    switch (r->type) {
        case GLUE_SEARCH:
            despotify_free_search(r->priv->search);
            break;
        case GLUE_PLAYLISTS:
            despotify_free_playlist(r->priv->pl);
            break;
        case GLUE_LIST:
            if (!r->priv->parent)
                despotify_free_playlist(r->priv->pl);
            /* if parent is set, we don't free anything, since our list
             * is part of previous playlist search, which will be freed later on.
             */
            break;
        case GLUE_ARTIST:
            if (!r->priv->parent)
                despotify_free_artist_browse(r->priv->artist);
            break;
        case GLUE_ALBUM:
            if (!r->priv->parent)
                despotify_free_album_browse(r->priv->album);
            break;
        default:
            g_print("glue_response_free undef situation\n");
            break;
    }

    g_free(r->priv);
    g_free(r);

    return 0;
}

int yaspot_glue_response_ref(yaspot_glue_response_t *r) {
    if (!r)
        return -1;

    r->priv->ref++;
    r->priv->glue->all_refs++;

    g_print("ref %p (%d)\n", r, r->priv->ref);

    return r->priv->ref;
}

void yaspot_glue_add_state_cb(yaspot_glue_t *t, yaspot_glue_state_changed_t callback, gpointer userdata) {
    g_assert(t);
    g_assert(callback);

    state_callback_t *s = g_new0(state_callback_t, 1);
    s->state_cb = callback;
    s->state_cb_data = userdata;
    g_print("add state cb %p data %p\n", callback, userdata);

    t->state_cb = g_list_append(t->state_cb, s);
}

void yaspot_glue_del_state_cb(yaspot_glue_t *t, yaspot_glue_state_changed_t callback, gpointer userdata) {
    GList *i;
    state_callback_t *s;

    g_assert(t);
    g_assert(callback);

    i = g_list_first(t->state_cb);
    do {
        s = (state_callback_t*)i->data;

        if (s && s->state_cb == callback && s->state_cb_data == userdata) {
            g_print("del state cb %p data %p\n", callback, userdata);
            t->state_cb = g_list_remove_link(t->state_cb, i);
            g_free(s);
            break;
        }
    } while ((i = g_list_next(i)));
}

void yaspot_glue_set_timer_cb(yaspot_glue_t *t, yaspot_glue_timer_update_t callback, gpointer userdata) {
    g_assert(t);

    if (!callback) {
        t->timer_cb = NULL;
        t->timer_cb_data = NULL;
        return;
    }

    t->timer_cb_data = userdata;
    t->timer_cb = callback;
}

static void playback_cmd_cb(pb_playback_t *pb, enum pb_state_e state, pb_req_t *cmd, void *data) {
    switch (state) {
        case PB_STATE_STOP:
            g_print("libplayback: received STOP\n");
            break;
        case PB_STATE_PLAY:
            g_print("libplayback: received PLAY\n");
            break;
        default:
            g_print("libplayback: received unknown command\n");
            break;
    }

    pb_playback_req_completed(pb, cmd);
}

static void playback_req_cb(pb_playback_t *pb,
                            enum pb_state_e granted_state,
                            const gchar *reason,
                            pb_req_t *req,
                            void *data) {
    g_print("request state cb\n");

    switch (granted_state) {
        case PB_STATE_STOP:
            g_print("libplayback: granted STOP\n");
            break;
        case PB_STATE_PLAY:
            g_print("libplayback: granted PLAY\n");
            break;
        default:
            g_print("libplayback: granted unknown command (%s)\n", reason);
            break;
    }

    pb_playback_req_completed(pb, req);
}

static void playback_hint_cb(G_GNUC_UNUSED pb_playback_t *pb, const gint *allowed_state, void *data) {
    if (!allowed_state[PB_STATE_PLAY]) {
        g_print("libplayback: sounds disabled by policy, pause playback\n");
    } else {
        g_print("libplayback: sounds enabled by policy, resume playback\n");
    }
}

static void init_pb(yaspot_glue_t *t) {
    g_assert(t);
    g_assert(t->dbus);

    if ((t->pb = pb_playback_new_2(t->dbus, PB_CLASS_MEDIA, PB_FLAG_AUDIO, PB_STATE_STOP, playback_cmd_cb, t)) == NULL) {
        g_print("Failed to initialize libplayback ginterface\n");
        goto error;
    }

    pb_playback_set_state_hint(t->pb, playback_hint_cb, t);

    return;

error:
    if (t->dbus) {
        dbus_connection_unref(t->dbus);
        t->dbus = NULL;
    }
    return;
}

void conic_event_cb(ConIcConnection *connection, ConIcConnectionEvent *event, gpointer userdata) {
    yaspot_glue_t *t = (yaspot_glue_t*)userdata;
    ConIcConnectionStatus status;

    g_print("conic_event_cb: ");
    status = con_ic_connection_event_get_status(event);

    switch (status) {
        case CON_IC_STATUS_CONNECTED:
            g_print("connected\n");
            t->conic_connected = TRUE;
            new_command(t, CONTROL_RECONNECT_CONTINUE, 0, NULL, NULL, NULL);
            break;

        case CON_IC_STATUS_DISCONNECTED:
            g_print("disconnected\n");
            t->conic_connected = FALSE;
            t->client_state = YASPOT_DISCONNECTED;
            emit_state_changed(t, 0);
            break;

        case CON_IC_STATUS_DISCONNECTING:
            g_print("disconnecting\n");
            t->conic_connected = FALSE;
            t->client_state = YASPOT_DISCONNECTED;
            emit_state_changed(t, 0);
            break;

        case CON_IC_STATUS_NETWORK_UP:
            g_print("network up\n");
            t->conic_connected = FALSE;
            break;
    }
}

yaspot_glue_t* yaspot_glue_init(DBusConnection *dbus) {
    yaspot_glue_t *t;

    t = g_new0(yaspot_glue_t, 1);
    t->dbus = dbus;

    if (despotify_init())
        g_print("init despotify\n");
    else {
        g_print("failed to init despotify\n");
        goto fail;
    }

    t->conic = con_ic_connection_new();

    g_signal_connect(G_OBJECT(t->conic), "connection-event",
                     G_CALLBACK(conic_event_cb), t);

    g_print("init audio()\n");
    t->audio = yaspot_audio_init();

    t->mutex = g_mutex_new();
    t->wait = g_cond_new();
    t->pause_wait = g_cond_new();
    t->play_state = GLUE_PAUSE;
    t->thread = g_thread_create(thread_loop, t, TRUE, NULL);

    /* control thread */
    t->control_queue = g_async_queue_new();
    t->control_thread = g_thread_create(control_thread_loop, t, TRUE, NULL);

    init_pb(t);

    playlist_from_disk(t);

    return t;

fail:
    g_free(t);
    return NULL;
}

static void state_cb_cleanup(gpointer data, gpointer userdata) {
    state_callback_t *s = (state_callback_t*)data;
    g_free(s);
}

int yaspot_glue_cleanup(yaspot_glue_t *t) {

    if (t->pb) {
        pb_playback_destroy(t->pb);
        t->pb = NULL;
    }

    if (t->dbus) {
        dbus_connection_unref(t->dbus);
        t->dbus = NULL;
    }

    yaspot_glue_stop(t);

    if (t->audio)
        yaspot_audio_cleanup(t->audio);

    thread_exit(t);

    if (t->username)
        g_free(t->username);

    if (t->password)
        g_free(t->password);

    if (t->session)
        despotify_exit(t->session);

    new_command(t, CONTROL_QUIT, 0, NULL, NULL, NULL);
    g_thread_join(t->control_thread);
    g_async_queue_unref(t->control_queue);

    if (t->state_cb) {
        g_list_foreach(t->state_cb, state_cb_cleanup, NULL);
        g_list_free(t->state_cb);
        t->state_cb = NULL;
    }

    if (t->wait) {
        g_cond_free(t->wait);
        t->wait = NULL;
    }
    if (t->pause_wait) {
        g_cond_free(t->pause_wait);
        t->pause_wait = NULL;
    }

    if (t->mutex) {
        g_mutex_free(t->mutex);
        t->mutex = NULL;
    }

    con_ic_connection_disconnect(t->conic);
    g_object_unref(t->conic);

    despotify_cleanup();
    g_print("all refs: %d\n", t->all_refs);

    g_free(t);

    return 0;
}

void yaspot_glue_set_credentials(yaspot_glue_t *t, const gchar *username, const gchar *password) {
    g_assert(t);
    g_assert(username);
    g_assert(password);

    if (t->username)
        g_free(t->username);
    if (t->password)
        g_free(t->password);

    t->username = g_strdup(username);
    t->password = g_strdup(password);

}

void yaspot_glue_set_high_bitrate(yaspot_glue_t *t, gboolean high_bitrate) {
    g_assert(t);

    t->high_bitrate = high_bitrate;
}

gint yaspot_glue_connect(yaspot_glue_t *t) {
    g_assert(t);

    if (t->client_state == YASPOT_CONNECTING)
        return -1;

    new_command(t, CONTROL_RECONNECT, 0, NULL, NULL, NULL);

    return 0;
}

void yaspot_glue_print_info(yaspot_glue_t *t) {
    struct user_info *u = t->session->user_info;

    g_print("User      : %s\n", u->username);
    g_print("Country   : %s\n", u->country);
    g_print("Type      : %s\n", u->type);
    g_print("Expiry    : %s\n", ctime(&u->expiry));
    g_print("Host      : %s:%d\n", u->server_host, u->server_port);
    g_print("Last ping : %s\n", ctime(&u->last_ping));

}

/* get nth track in track list, start from 0 */
static struct track* get_nth_track(struct track *tracks, const gint num) {
    struct track *tr;
    gint i;

    for (i = 0, tr = tracks; tr; tr = tr->next, i++) {
        if (i == num)
            break;
    }

    if (i == num)
        return tr;
    else
        return NULL;
}

static yaspot_glue_response_t* res_type(yaspot_glue_t *t, gint type) {
    yaspot_glue_response_t *r;
    r = g_new0(yaspot_glue_response_t, 1);
    r->type = type;
    r->priv = g_new0(yaspot_glue_response_internal_t, 1);
    r->priv->glue = t;
    yaspot_glue_response_ref(r);
    return r;
}

yaspot_glue_response_t* yaspot_glue_search(yaspot_glue_t *t, const gchar *search, gint maxresults) {
    yaspot_glue_response_t *r;
    struct search_result *res;
    struct playlist *pl;
    gint num_tracks;

    res = despotify_search(t->session, search, maxresults);

    if (res) {
        pl = res->playlist;
        num_tracks = pl->num_tracks;
        g_print("found %d artists %d albums %d tracks\n", res->total_artists, res->total_albums, res->total_tracks);
        g_print("playlist has %d entries\n", pl->num_tracks);

        r = res_type(t, GLUE_SEARCH);

        r->priv->search = res;
        r->priv->pl = pl;
        r->priv->tracks = pl->tracks;

        return r;
    } else {
        g_print("failed to search\n");
        return NULL;
    }
}

yaspot_glue_response_t* yaspot_glue_artist(yaspot_glue_response_t *r, gint num) {
    struct track *tr;
    struct artist_browse *artist;
    yaspot_glue_response_t *response = NULL;

    if (!r)
        return NULL;

    tr = get_nth_track(r->priv->tracks, num-1);
    if (tr) {
        artist = despotify_get_artist(r->priv->glue->session, tr->artist->id);

        if (artist) {
            response = res_type(r->priv->glue, GLUE_ARTIST);
            response->priv->artist = artist;
            response->priv->tracks = NULL; /* FIXME */
            g_print("glue_artist FOUND\n");
        }
    }

    return response;
}

yaspot_glue_response_t* yaspot_glue_artist_album(yaspot_glue_response_t *r, gint num) {
    struct album_browse *a;
    yaspot_glue_response_t *response = NULL;
    gint i;

    if (!r || r->type != GLUE_ARTIST)
        return NULL;

    for (i = 0, a = r->priv->artist->albums; a; i++, a = a->next) {
        if (i == num) {
            response = res_type(r->priv->glue, GLUE_ALBUM);
            response->priv->album = a;
            response->priv->tracks = a->tracks;
            response->priv->parent = r;
            yaspot_glue_response_ref(response->priv->parent);
            g_print("glue: created album %d: %s / %s / %d \n", num, a->name, a->id, a->year);
        }
    }

    return response;
}

gchar* yaspot_glue_artist_description(yaspot_glue_response_t *r, gint max_chars) {
    gchar *response = NULL;
    gint len;

    if (!r || r->type != GLUE_ARTIST)
        return NULL;

    if (r->priv->artist->text) {
        len = strlen(r->priv->artist->text);
        if (max_chars <= 0)
            max_chars = len;
        response = g_strndup(r->priv->artist->text, max_chars);
        if (response && len > max_chars && max_chars > 3)
            response[max_chars-1] = response[max_chars-2] = response[max_chars-3] = '.';
    }

    return response;
}

gboolean yaspot_glue_album(yaspot_glue_response_t *r, yaspot_glue_response_cb cb, gint num, gpointer userdata) {
    struct track *tr;
    gboolean response = FALSE;

    g_assert(r);
    g_assert(cb);

    tr = get_nth_track(r->priv->tracks, num-1);
    if (tr) {
        yaspot_glue_album_by_id(r->priv->glue, cb, tr->album_id, userdata);
        response = TRUE;
    }

    return response;
}

gboolean yaspot_glue_album_by_id(yaspot_glue_t *t, yaspot_glue_response_cb cb, const gchar *id, gpointer userdata) {
    g_assert(t);
    g_assert(cb);
    g_assert(id);

    new_command(t, CONTROL_GET_ALBUM, 0, id, cb, userdata);

    return TRUE;
}

gboolean yaspot_glue_album_is_single(yaspot_glue_response_t *r, gint single_limit) {
    if (!r)
        return FALSE;

    if (!yaspot_glue_album_is_compilation(r, single_limit) && r->priv->album->num_tracks <= single_limit)
        return TRUE;
    else
        return FALSE;
}

gboolean yaspot_glue_album_is_compilation(yaspot_glue_response_t *r, gint artist_limit) {
    struct artist *artist = NULL;
    struct artist *prev_artist = NULL;
    struct track *tr;
    gint different_artists = 0;

    if (!r)
        return FALSE;

    for (tr = r->priv->album->tracks; tr; tr = tr->next) {
        artist = tr->artist;

        if (prev_artist) {
            if (g_strcmp0(prev_artist->id, artist->id))
                different_artists++;
        }

        prev_artist = artist;
    }

    if (different_artists >= artist_limit)
        return TRUE;
    else
        return FALSE;
}

gboolean yaspot_glue_album_from_artist(yaspot_glue_response_t *album, yaspot_glue_response_t *artist) {
    struct track *tr;
    gint num = 0;
    gint others = 0;
    g_assert(album);
    g_assert(artist);

    for (tr = album->priv->album->tracks; tr; tr = tr->next) {
        num++;
        if (g_strcmp0(tr->artist->id, artist->priv->artist->id))
            others++;
    }

    /* if ratio of others in album is smaller or equal than 2/1,
     * this is artist's album */
    if (!others)
        return TRUE;

    if (num/others <= 2)
        return TRUE;
    else
        return FALSE;
}

int yaspot_glue_album_year(yaspot_glue_response_t *r) {
    if (!r ||r->type != GLUE_ALBUM)
        return -1;

    return r->priv->album->year;
}

int yaspot_glue_artist_num_albums(yaspot_glue_response_t *r) {
    if (!r || r->type != GLUE_ARTIST)
        return -1;

    return r->priv->artist->num_albums;
}

gint yaspot_glue_res_n_tracks(yaspot_glue_response_t *r) {
    struct track *tr;
    gint result = -1;
    g_assert(r);

    if (r->priv->tracks) {
        for (result = 0, tr = r->priv->tracks; tr; result++, tr = tr->next);
    }

    return result;
}

static void playlist_to_model(struct playlist *pl, GtkListStore *model) {
    struct track *tr;
    gint i;

    gtk_list_store_clear(model);
    for (i = 0, tr = pl->tracks; tr; tr = tr->next, i++) {
        gtk_list_store_insert_with_values(model, NULL, i,
                PLAYLIST_COL_TRACK, i+1,
                PLAYLIST_COL_TITLE, tr->title,
                PLAYLIST_COL_ARTIST, tr->artist->name,
                PLAYLIST_COL_ALBUM, tr->album,
                -1);
    }
}

static void album_to_model(struct track *tracks, GtkListStore *model) {
    struct track *tr;
    gint i;

    gtk_list_store_clear(model);
    for (i = 0, tr = tracks; tr; tr = tr->next, i++) {
        gint min = tr->length/1000 / 60;
        gint sec = tr->length/1000 % 60;
        gchar *length = g_strdup_printf("%d:%02d", min, sec);

        gtk_list_store_insert_with_values(model, NULL, i,
                ALBUM_COL_TRACK, i+1,
                ALBUM_COL_TITLE, tr->title,
                ALBUM_COL_LENGTH, length,
                -1);
        g_free(length);
    }
}

void yaspot_glue_response_to_model(yaspot_glue_response_t *r, GtkListStore *model) {
    if (!r) {
        g_print("response_to_model NULL r\n");
        return;
    }

    switch (r->type) {
        case GLUE_SEARCH:
            playlist_to_model(r->priv->pl, model);
            break;
        case GLUE_PLAYLISTS:
            g_print("yaspot_glue_response_to_model - GLUE_PLAYLISTS: deprecated!\n");
            break;
        case GLUE_LIST:
            playlist_to_model(r->priv->pl, model);
            break;
        case GLUE_ALBUM:
            album_to_model(r->priv->album->tracks, model);
            break;
        case GLUE_ARTIST:
            /* TODO */
            break;
        default:
            g_print("response_to_model undefined case\n");
            break;
    }
}

void yaspot_glue_response_to_queue_model(yaspot_glue_response_t *r, GtkListStore *model) {
    struct track *tr, *tracks;
    gint i;

    g_assert(r);
    g_assert(model);
    g_assert(r->type != GLUE_PLAYLISTS);
    g_assert(r->type != GLUE_ARTIST);

    if (r->type == GLUE_ALBUM)
        tracks = r->priv->album->tracks;
    else
        tracks = r->priv->pl->tracks;

    gtk_list_store_clear(model);
    for (i = 0, tr = tracks; tr; i++, tr = tr->next) {
        gint min = tr->length/1000 / 60;
        gint sec = tr->length/1000 % 60;
        gchar *length = g_strdup_printf("%d:%02d", min, sec);
        gchar *artist = g_strdup_printf("(%s)", tr->artist->name);

        gtk_list_store_insert_with_values(model, NULL, i,
                QUEUE_COL_TRACK, i+1,
                QUEUE_COL_TITLE, tr->title,
                QUEUE_COL_ARTIST, artist,
                QUEUE_COL_LENGTH, length,
                -1);
        g_free(length);
        g_free(artist);
    }
}

void yaspot_glue_stored_lists(yaspot_glue_t *t, yaspot_glue_response_cb cb, gpointer userdata) {
    g_assert(t);
    g_assert(cb);

    new_command(t, CONTROL_STORED_LISTS, 0, NULL, cb, userdata);

    return;
}

/* get nth track in playlist list, start from 0 */
static struct playlist* get_nth_playlist(struct playlist *playlists, gint num) {
    struct playlist *pl;
    gint i;

    for (pl = playlists, i = 0; pl; pl = pl->next, i++) {
        if (i == num)
            break;
    }

    if (i == num)
        return pl;
    else
        return NULL;
}

yaspot_glue_response_t* yaspot_glue_stored_playlist(yaspot_glue_response_t *r, gint num) {
    yaspot_glue_response_t *response = NULL;
    struct playlist *pl = NULL;
    //struct playlist *new_pl;

    if (r && r->type == GLUE_PLAYLISTS && r->priv->pl && num > 0)
        pl = get_nth_playlist(r->priv->pl, num-1);

    if (pl/* && (new_pl = despotify_get_playlist(r->glue->session, pl->playlist_id)) */ ) {

        response = res_type(r->priv->glue, GLUE_LIST);
        response->priv->parent = r;
        response->priv->pl = pl;
        response->priv->tracks = pl->tracks;
        yaspot_glue_response_ref(r); /* ref parent, so that it is not removed if this is used */
    }

    return response;
}

gboolean yaspot_glue_stored_playlist_by_id(yaspot_glue_t *t, yaspot_glue_response_cb cb, const gchar *id, gpointer userdata) {
    g_assert(t);
    g_assert(cb);
    g_assert(id);

    new_command(t, CONTROL_GET_LIST, 0, id, cb, userdata);

    return TRUE;
}

void* yaspot_glue_get_image(yaspot_glue_response_t *r, gint *len) {
    if (!r)
        return NULL;

    void *response = NULL;

    switch (r->type) {
        case GLUE_ALBUM:
            if (r->priv->album->cover_id[0])
                response = despotify_get_image(r->priv->glue->session, r->priv->album->cover_id, len);
            break;
        case GLUE_ARTIST:
            if (r->priv->artist && r->priv->artist->portrait_id[0])
                response = despotify_get_image(r->priv->glue->session, r->priv->artist->portrait_id, len);
            break;
        default:
            /* shouldn't be here */
            break;
    }

    return response;
}

gchar* yaspot_glue_response_get_name(yaspot_glue_response_t *r) {
    if (!r)
        return NULL;

    gchar *response = NULL;

    switch (r->type) {
        case GLUE_LIST:
            response = g_strdup_printf("%s (%s)", r->priv->pl->name, r->priv->pl->author);
            break;
        case GLUE_ALBUM:
            if (r->priv->album->year > 0)
                response = g_strdup_printf("%s (%d)", r->priv->album->name, r->priv->album->year);
            else
                response = g_strdup(r->priv->album->name);
            break;
        case GLUE_ARTIST:
            response = g_strdup(r->priv->artist->name);
            break;
        default:
            g_print("undef response_get_name\n");
            break;
    }

    return response;
}

int yaspot_glue_play(yaspot_glue_response_t *r, guint num) {
    g_assert(r);
    g_assert(num > 0);

    new_command(r->priv->glue, CONTROL_PLAY, num, NULL, NULL, r);

    return 0;
}

int yaspot_glue_set_pause(yaspot_glue_t *t, gboolean pause) {
    if (!t->playing) {
        g_print("set_pause, nothing playing atm, cannot set pause\n");
        return -1;
    }

    if (pause) {
        if (!t->paused) {
            thread_pause(t);
            //pb_playback_req_state(t->pb, PB_STATE_STOP, playback_req_cb, t);
            g_print("pausing playback\n");
        }
    } else {
        if (t->paused) {
            thread_play(t);
            //pb_playback_req_state(t->pb, PB_STATE_PLAY, playback_req_cb, t);
            g_print("resuming playback\n");
        }
    }

    emit_state_changed(t, 0);

    return 0;
}

int yaspot_glue_stop(yaspot_glue_t* t) {
    if (t->playing) {
        thread_pause(t);
        despotify_stop(t->session);
    }
    if (t->current) {
        yaspot_glue_response_unref(t->current);
        t->current = NULL;
    }
    t->playing = FALSE;
    g_print("stopping playback\n");

    emit_state_changed(t, 0);

    return 0;
}

int yaspot_glue_res_stop(yaspot_glue_response_t* t) {
    return yaspot_glue_stop(t->priv->glue);
}

int yaspot_glue_next(yaspot_glue_t *t) {
    struct track *tr;
    gint ret = 0;
    g_assert(t);
    g_static_rw_lock_reader_lock(&data_lock);

    if (!t->playing || !t->current) {
        g_print("next: no current playlist\n");
        goto end;
        ret = -1;
    }

    /* nth_track function starts counting from 0 */
    tr = get_nth_track(t->current->priv->tracks, t->current_track_num);
    if (!tr) {
        g_print("next: unknown next track\n");
        goto end;
        ret = -1;
    }

    new_command(t, CONTROL_NEXT, t->current_track_num+1, NULL, NULL, t->current);

end:
    g_static_rw_lock_reader_unlock(&data_lock);
    return ret;
}

int yaspot_glue_prev(yaspot_glue_t *t) {
    gint ret = 0;
    g_assert(t);
    g_static_rw_lock_reader_lock(&data_lock);

    if (!t->playing || !t->current) {
        g_print("prev: no current playlist\n");
        ret = -1;
        goto end;
    }
    if (t->current_track_num <= 1) {
        g_print("prev: already first track\n");
        ret = -1;
        goto end;
    }

    new_command(t, CONTROL_PREV, t->current_track_num-1, NULL, NULL, t->current);

end:
    g_static_rw_lock_reader_unlock(&data_lock);
    return ret;
}

yaspot_glue_response_t* yaspot_glue_current_queue(yaspot_glue_t *t) {
    yaspot_glue_response_ref(t->current);
    return t->current;
}

yaspot_glue_response_t* yaspot_glue_res_current_queue(yaspot_glue_response_t *r) {
    return yaspot_glue_current_queue(r->priv->glue);
}

gboolean yaspot_glue_playing(yaspot_glue_t *t) {
    g_assert(t);

    return (t->playing && !t->paused);
}

gchar* yaspot_glue_current_track(yaspot_glue_t *t) {
    gchar *response = NULL;
    g_assert(t);

    g_static_rw_lock_reader_lock(&data_lock);

    if (t->current_track) {
        if (t->current_track->artist && t->current_track->title)
            response = g_strdup_printf("%s: %s", t->current_track->artist->name, t->current_track->title);
        if (!response)
            response = g_strdup_printf("<undef>");
    }
    g_static_rw_lock_reader_unlock(&data_lock);

    return response;
}

gint yaspot_glue_current_track_num(yaspot_glue_t *t) {
    gint response = -1;
    struct track *tr;
    gint i;

    g_assert(t);

    g_static_rw_lock_reader_lock(&data_lock);

    if (t->current) {
        i = 1;

        for (tr = t->current->priv->tracks; tr != t->current_track; tr = tr->next)
            i++;

        response = i;
    }
    g_static_rw_lock_reader_unlock(&data_lock);

    return response;
}

const char* yaspot_glue_response_id(yaspot_glue_response_t *r) {
    g_assert(r);

    switch (r->type) {
        case GLUE_SEARCH:
            return NULL;

        case GLUE_ALBUM:
            return (const char*)r->priv->album->id;

        case GLUE_LIST:
            return (const char*)r->priv->pl->playlist_id;

        case GLUE_PLAYLISTS:
            return NULL;

        case GLUE_ARTIST:
            g_print("GLUE_ARTIST not implemented\n");
            return NULL;
    }

    return NULL;
}

static gint playlist_to_disk(yaspot_glue_t *t) {
    GKeyFile *keyfile = NULL;
    GList *i;
    FILE *file = NULL;
    gchar *bigdump = NULL;
    gsize length;
    size_t written;
    const yaspot_playlist_entry_t *e;
    gint ret = 0;

    if (g_list_length(t->playlist_history) == 0) {
        g_print("playlist_to_disk: won't even try to save 0 lists\n");
        goto end;
    }

    if (!g_file_test(YASPOT_CONFIG_DIR, G_FILE_TEST_IS_DIR)) {
        if (g_mkdir(YASPOT_CONFIG_DIR, 0700)) {
            g_print("playlist_to_disk: failed to create config dir %s\n", YASPOT_CONFIG_DIR);
            ret = -1;
            goto end;
        }
    }

    file = fopen(YASPOT_CONFIG_PLAYLIST, "w");

    if (!file) {
        g_print("playlist_to_disk: failed to open %s for writing\n", YASPOT_CONFIG_PLAYLIST);
        ret = -1;
        goto end;
    }

    keyfile = g_key_file_new();

    for (i = t->playlist_history; i; i = i->next) {
        e = (yaspot_playlist_entry_t*)i->data;
        g_key_file_set_string(keyfile, e->id, "name", e->name);
        g_key_file_set_string(keyfile, e->id, "author", e->author);
    }

    bigdump = g_key_file_to_data(keyfile, &length, NULL);

    if (!bigdump || !length) {
        g_print("playlist_to_disk: dump failed\n");
        ret = -1;
        goto end;
    }

    written = fwrite(bigdump, sizeof(char), length, file);

    if (written != length) {
        g_print("playlist_to_disk: error, length %d written %d\n", length, written);
        ret = -1;
        goto end;
    }

end:
    if (file)
        fclose(file);

    if (keyfile)
        g_key_file_free(keyfile);

    if (bigdump)
        g_free(bigdump);

    return ret;
}

static gint playlist_from_disk(yaspot_glue_t *t) {
    GKeyFile *keyfile;
    yaspot_playlist_entry_t *e;
    gchar **ids;
    gchar *item_name;
    gchar *item_author;
    gsize length;
    gint i;

    keyfile = g_key_file_new();

    if (!g_key_file_load_from_file(keyfile, YASPOT_CONFIG_PLAYLIST, G_KEY_FILE_NONE, NULL)) {
        g_print("playlist_from_disk: could not open %s for reading\n", YASPOT_CONFIG_PLAYLIST);
        g_key_file_free(keyfile);
        return -1;
    }

    ids = g_key_file_get_groups(keyfile, &length);

    if (!ids) {
        g_key_file_free(keyfile);
        return -1;
    }

    for (i = 0; i < length; i++) {
        if (!(item_name = g_key_file_get_string(keyfile, ids[i], "name", NULL)))
            continue;

        if (!(item_author = g_key_file_get_string(keyfile, ids[i], "author", NULL))) {
            g_free(item_name);
            continue;
        }

        e = playlist_entry_new(item_name, item_author, ids[i]);
        t->playlist_history = g_list_append(t->playlist_history, e);

        g_free(item_name);
        g_free(item_author);
    }

    g_strfreev(ids);

    return 0;
}

static void playlist_entry_free(yaspot_playlist_entry_t *e) {
    g_free(e->name);
    g_free(e->author);
    g_free(e->id);
    g_free(e);
}

static yaspot_playlist_entry_t* playlist_entry_new(const gchar *name, const gchar *author, const gchar *id) {
    yaspot_playlist_entry_t *e;

    e = g_new0(yaspot_playlist_entry_t, 1);
    e->name = g_strdup(name);
    e->author = g_strdup(author);
    e->id = g_strdup(id);

    return e;
}

static void playlist_free_cb(gpointer data, gpointer userdata) {
    yaspot_playlist_entry_t *e = (yaspot_playlist_entry_t*)data;

    g_assert(e);
    playlist_entry_free(e);
}

void yaspot_playlist_reload(yaspot_glue_t *t, yaspot_glue_response_cb cb, gpointer userdata) {
    g_assert(t);

    g_list_foreach(t->playlist_history, playlist_free_cb, NULL);
    g_list_free(t->playlist_history);
    t->playlist_history = NULL;

    new_command(t, CONTROL_STORED_LISTS, 0, NULL, cb, userdata);
}

gint yaspot_playlist_length(yaspot_glue_t *t) {
    return g_list_length(t->playlist_history);
}

const yaspot_playlist_entry_t* yaspot_playlist_iterate(yaspot_glue_t *t, void **state) {
    GList *position;
    g_assert(t);

    if (*state) {
        position = *state;
        position = position->next;
        if (position) {
            *state = (void*)position;
            return position->data;
        } else {
            *state = NULL;
            return NULL;
        }
    } else {
        *state = (void*)t->playlist_history;
        if (t->playlist_history)
            return t->playlist_history->data;
        else
            return NULL;
    }
}

// vim: expandtab shiftwidth=4

