#include <glib.h>
#include <hildon/hildon.h>
#include <despotify.h>
#include <libplayback/playback.h>
#include "string.h"

#include "playlist.h"

#include "audio.h"
#include "glue.h"


#define GLUE_STR_ERROR "<error>"

typedef struct state_callback_t {
    yaspot_glue_state_changed_t state_cb;
    gpointer state_cb_data;
} state_callback_t;

struct yaspot_glue_t {
    struct despotify_session *session;
    yaspot_audio_t *audio;

    gchar *username;
    gchar *password;

    /* dbus & libplayback */
    DBusConnection *dbus;
    pb_playback_t *pb;

    /* list of state callbacks */
    GList *state_cb;

    yaspot_glue_timer_update_t timer_cb;
    gpointer timer_cb_data;

    /* global reference counter, mostly for debugging */
    gint all_refs;

    /* internal thread */
    GThread *thread;
    GMutex *mutex;
    GCond *wait;
    GCond *pause_wait;

    /* data behind data_lock */

    yaspot_glue_response_t *current; /* current play queue */
    struct track *current_track; /* currently playing track */
    gint current_track_num; /* nth track in playlist, current */
    gboolean playing; /* internal play states */
    gboolean paused;

    /* data behind mutex */
    enum {
        GLUE_PAUSE,
        GLUE_PLAY,
        GLUE_EXIT
    } play_state;

};

static GStaticRWLock data_lock = G_STATIC_RW_LOCK_INIT;

struct yaspot_glue_response_t {
    yaspot_glue_t *glue;

    enum {
        GLUE_SEARCH,
        GLUE_LIST,
        GLUE_PLAYLISTS,
        GLUE_ARTIST,
        GLUE_ALBUM
    } type;

    struct search_result *search;
    struct playlist *pl;
    struct artist_browse *artist;
    struct album_browse *album;

    struct track *tracks;


    /* this is only for the special case of stored
     * playlist (might be other uses as well).
     * when this is set, when creating child object, ref parent
     * so that parent is not removed before child is. */
    struct yaspot_glue_response_t *parent;

    /* start with 1, when 0 free object */
    gint ref;
};

static void thread_play(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    if (t->play_state != GLUE_PLAY) {
        t->play_state = GLUE_PLAY;
        t->paused = FALSE;
        g_cond_signal(t->wait);
    }
    g_mutex_unlock(t->mutex);
}

static void thread_pause(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    if (t->play_state != GLUE_PAUSE) {
        t->play_state = GLUE_PAUSE;
        t->paused = TRUE;
        g_print("thread: set pause, block while setting..\n");
        g_cond_wait(t->pause_wait, t->mutex);
        g_print("thread: set pause done.\n");
    }
    g_mutex_unlock(t->mutex);
}

static void thread_exit(yaspot_glue_t *t) {
    g_mutex_lock(t->mutex);
    t->play_state = GLUE_EXIT;
    g_cond_signal(t->wait);
    g_mutex_unlock(t->mutex);

    g_thread_join(t->thread);
}

static void set_current(yaspot_glue_t *t, yaspot_glue_response_t *r) {
    g_static_rw_lock_writer_lock(&data_lock);
    g_assert(t);
    t->current = r;
    g_static_rw_lock_writer_unlock(&data_lock);
}

static void set_current_track(yaspot_glue_t *t, struct track *new_tr) {
    struct track *tr;
    gint i;
    g_assert(t);
    g_static_rw_lock_writer_lock(&data_lock);

    if (!new_tr) {
        t->current_track = NULL;
        t->current_track_num = -1;
    } else {
        g_assert(t->current);

        for (i = 1, tr = t->current->tracks; tr; i++, tr = tr->next) {
            if (memcmp(new_tr->track_id, tr->track_id, 33) == 0) {
                t->current_track = tr;
                t->current_track_num = i;
                break;
            }
        }
    }
    g_static_rw_lock_writer_unlock(&data_lock);
}

static void* thread_loop(void *arg) {
    yaspot_glue_t *t = (yaspot_glue_t*)arg;
    struct pcm_data pcm;
    gint loop = 1;

    while (loop) {
        //g_print("switch (%d): ", t->play_state);
        switch (t->play_state) {
            case GLUE_PAUSE: {
                g_mutex_lock(t->mutex);
                g_print("thread: PAUSE, sleeping\n");
                g_cond_signal(t->pause_wait);
                g_cond_wait(t->wait, t->mutex);
                g_print("thread: PAUSE, woke up\n");
                g_mutex_unlock(t->mutex);
                break;
            }

            case GLUE_PLAY: {
                gint rc = despotify_get_pcm(t->session, &pcm);
                if (rc == 0) {
                    yaspot_audio_write(t->audio, pcm.buf, pcm.len);
                    //g_print("audio_write %d\n", pcm.len);
                }
                else
                    g_print("thread: despotify_get_pcm() failed\n");
                break;
            }

            case GLUE_EXIT: {
                g_print("thread: EXIT\n");
                loop = 0;
                break;
            }

        }
    }

    return NULL;
}

static void emit_state_cb(gpointer data, gpointer userdata) {
    state_callback_t *s = (state_callback_t*)data;
    yaspot_glue_t *t = (yaspot_glue_t*)userdata;

    g_assert(t);
    g_assert(s);

    g_print("emit state %p data %p\n", s->state_cb, s->state_cb_data);

    if (s->state_cb)
        s->state_cb(t->playing, t->paused, s->state_cb_data);
}

static void emit_state_changed(yaspot_glue_t *t) {
    g_assert(t);

    g_list_foreach(t->state_cb, emit_state_cb, t);
}

static void emit_timer_update(yaspot_glue_t *t, gint now, gint length) {
    g_assert(t);

    if (t->timer_cb) {
        t->timer_cb(t, now, length, t->timer_cb_data);
    }
}

static void session_cb(struct despotify_session *session, gint signal, void *data, void *userdata) {
    yaspot_glue_t *t = (yaspot_glue_t*)userdata;
    static gint last = 0;

    switch (signal) {
        case DESPOTIFY_NEW_TRACK: {
            struct track *tr = (struct track*)data;
            g_print("new track name: %s bitrate: %d\n", tr->title, tr->file_bitrate);
            set_current_track(t, tr);
            thread_play(t);
            emit_state_changed(t);
            last = 0;
            break;
        }
        case DESPOTIFY_TIME_TELL: {
            double elapsed_time = *(double*)data;
            if ((gint)elapsed_time > last) {
                last = (gint)elapsed_time;
                emit_timer_update(t, (gint)elapsed_time, t->current_track->length/1000);
            }
            break;
        }

        case DESPOTIFY_END_OF_PLAYLIST: {
            g_print("end_of_playlist\n");
            set_current_track(t, NULL);
            emit_state_changed(t);
            break;
        }

        default: {
            g_print("default - should never be here\n");
            break;
        }
    }
}

int yaspot_glue_response_unref(yaspot_glue_response_t *r) {
    if (!r)
        return -1;

    r->ref--;
    r->glue->all_refs--;

    g_print("unref %p (%d)\n", r, r->ref);

    if (r->ref > 0)
        return r->ref;

    if (r->parent)
        yaspot_glue_response_unref(r->parent);

    switch (r->type) {
        case GLUE_SEARCH:
            despotify_free_search(r->search);
            break;
        case GLUE_PLAYLISTS:
            despotify_free_playlist(r->pl);
            break;
        case GLUE_LIST:
            if (!r->parent)
                despotify_free_playlist(r->pl);
            /* if parent is set, we don't free anything, since our list
             * is part of previous playlist search, which will be freed later on.
             */
            break;
        case GLUE_ARTIST:
            if (!r->parent)
                despotify_free_artist_browse(r->artist);
            break;
        case GLUE_ALBUM:
            if (!r->parent)
                despotify_free_album_browse(r->album);
            break;
        default:
            g_print("glue_response_free undef situation\n");
            break;
    }

    g_free(r);

    return 0;
}

int yaspot_glue_response_ref(yaspot_glue_response_t *r) {
    if (!r)
        return -1;

    r->ref++;
    r->glue->all_refs++;

    g_print("ref %p (%d)\n", r, r->ref);

    return r->ref;
}

void yaspot_glue_add_state_cb(yaspot_glue_t *t, yaspot_glue_state_changed_t callback, gpointer userdata) {
    g_assert(t);
    g_assert(callback);

    state_callback_t *s = g_new0(state_callback_t, 1);
    s->state_cb = callback;
    s->state_cb_data = userdata;
    g_print("add state cb %p data %p\n", callback, userdata);

    t->state_cb = g_list_append(t->state_cb, s);
}

void yaspot_glue_del_state_cb(yaspot_glue_t *t, yaspot_glue_state_changed_t callback, gpointer userdata) {
    GList *i;
    state_callback_t *s;

    g_assert(t);
    g_assert(callback);

    i = g_list_first(t->state_cb);
    do {
        s = (state_callback_t*)i->data;

        if (s && s->state_cb == callback && s->state_cb_data == userdata) {
            g_print("del state cb %p data %p\n", callback, userdata);
            t->state_cb = g_list_remove_link(t->state_cb, i);
            g_free(s);
            break;
        }
    } while ((i = g_list_next(i)));
}

void yaspot_glue_set_timer_cb(yaspot_glue_t *t, yaspot_glue_timer_update_t callback, gpointer userdata) {
    g_assert(t);

    if (!callback) {
        t->timer_cb = NULL;
        t->timer_cb_data = NULL;
        return;
    }

    t->timer_cb_data = userdata;
    t->timer_cb = callback;
}

static void playback_cmd_cb(pb_playback_t *pb, enum pb_state_e state, pb_req_t *cmd, void *data) {
    switch (state) {
        case PB_STATE_STOP:
            g_print("libplayback: received STOP\n");
            break;
        case PB_STATE_PLAY:
            g_print("libplayback: received PLAY\n");
            break;
        default:
            g_print("libplayback: received unknown command\n");
            break;
    }

    pb_playback_req_completed(pb, cmd);
}

static void playback_req_cb(pb_playback_t *pb,
                            enum pb_state_e granted_state,
                            const gchar *reason,
                            pb_req_t *req,
                            void *data) {
    g_print("request state cb\n");

    switch (granted_state) {
        case PB_STATE_STOP:
            g_print("libplayback: granted STOP\n");
            break;
        case PB_STATE_PLAY:
            g_print("libplayback: granted PLAY\n");
            break;
        default:
            g_print("libplayback: granted unknown command (%s)\n", reason);
            break;
    }

    pb_playback_req_completed(pb, req);
}

static void playback_hint_cb(G_GNUC_UNUSED pb_playback_t *pb, const gint *allowed_state, void *data) {
    if (!allowed_state[PB_STATE_PLAY]) {
        g_print("libplayback: sounds disabled by policy, pause playback\n");
    } else {
        g_print("libplayback: sounds enabled by policy, resume playback\n");
    }
}

static void init_pb(yaspot_glue_t *t) {
    g_assert(t);
    g_assert(t->dbus);

    if ((t->pb = pb_playback_new_2(t->dbus, PB_CLASS_MEDIA, PB_FLAG_AUDIO, PB_STATE_STOP, playback_cmd_cb, t)) == NULL) {
        g_print("Failed to initialize libplayback ginterface\n");
        goto error;
    }

    pb_playback_set_state_hint(t->pb, playback_hint_cb, t);

    return;

error:
    if (t->dbus) {
        dbus_connection_unref(t->dbus);
        t->dbus = NULL;
    }
    return;
}

yaspot_glue_t* yaspot_glue_init(DBusConnection *dbus) {
    yaspot_glue_t *t;

    t = g_new0(yaspot_glue_t, 1);
    t->dbus = dbus;

    if (despotify_init())
        g_print("init despotify\n");
    else {
        g_print("failed to init despotify\n");
        goto fail;
    }

    g_print("init audio()\n");
    t->audio = yaspot_audio_init();

    t->mutex = g_mutex_new();
    t->wait = g_cond_new();
    t->pause_wait = g_cond_new();
    t->play_state = GLUE_PAUSE;
    t->thread = g_thread_create(thread_loop, t, TRUE, NULL);

    init_pb(t);

    return t;

fail:
    g_free(t);
    return NULL;
}

static void state_cb_cleanup(gpointer data, gpointer userdata) {
    state_callback_t *s = (state_callback_t*)data;
    g_free(s);
}

int yaspot_glue_cleanup(yaspot_glue_t *t) {

    if (t->pb) {
        pb_playback_destroy(t->pb);
        t->pb = NULL;
    }

    if (t->dbus) {
        dbus_connection_unref(t->dbus);
        t->dbus = NULL;
    }

    yaspot_glue_stop(t);

    if (t->audio)
        yaspot_audio_cleanup(t->audio);

    thread_exit(t);

    if (t->username)
        g_free(t->username);

    if (t->password)
        g_free(t->password);

    if (t->session)
        despotify_exit(t->session);

    if (t->state_cb) {
        g_list_foreach(t->state_cb, state_cb_cleanup, NULL);
        g_list_free(t->state_cb);
        t->state_cb = NULL;
    }

    if (t->wait) {
        g_cond_free(t->wait);
        t->wait = NULL;
    }
    if (t->pause_wait) {
        g_cond_free(t->pause_wait);
        t->pause_wait = NULL;
    }

    if (t->mutex) {
        g_mutex_free(t->mutex);
        t->mutex = NULL;
    }

    despotify_cleanup();
    g_print("all refs: %d\n", t->all_refs);

    g_free(t);

    return 0;
}

int yaspot_glue_auth(yaspot_glue_t *t, const gchar *username, const gchar *password, gint high_bitrate) {
    if (t->username)
        g_free(t->username);

    if (t->password)
        g_free(t->password);

    t->username = g_strdup(username);
    t->password = g_strdup(password);

    if (!t->session) {
        t->session = despotify_init_client(session_cb, t, !!high_bitrate);
    }

    if (!t->session) {
        g_error("failed to init session\n");
        return 1;
    }

    if (!despotify_authenticate(t->session, t->username, t->password)) {
        g_print("authentication failed: %s\n", despotify_get_error(t->session));
        return 1;
    }

    return 0;
}

void yaspot_glue_print_info(yaspot_glue_t *t) {
    struct user_info *u = t->session->user_info;

    g_print("User      : %s\n", u->username);
    g_print("Country   : %s\n", u->country);
    g_print("Type      : %s\n", u->type);
    g_print("Expiry    : %s\n", ctime(&u->expiry));
    g_print("Host      : %s:%d\n", u->server_host, u->server_port);
    g_print("Last ping : %s\n", ctime(&u->last_ping));

}

/* get nth track in track list, start from 0 */
static struct track* get_nth_track(struct track *tracks, const gint num) {
    struct track *tr;
    gint i;

    for (i = 0, tr = tracks; tr; tr = tr->next, i++) {
        if (i == num)
            break;
    }

    if (i == num)
        return tr;
    else
        return NULL;
}

static yaspot_glue_response_t* res_type(yaspot_glue_t *t, gint type) {
    yaspot_glue_response_t *r;
    r = g_new0(yaspot_glue_response_t, 1);
    r->type = type;
    r->glue = t;
    yaspot_glue_response_ref(r);
    return r;
}

yaspot_glue_response_t* yaspot_glue_search(yaspot_glue_t *t, const gchar *search, gint maxresults) {
    yaspot_glue_response_t *r;
    struct search_result *res;
    struct playlist *pl;
    gint num_tracks;

    res = despotify_search(t->session, search, maxresults);

    if (res) {
        pl = res->playlist;
        num_tracks = pl->num_tracks;
        g_print("found %d artists %d albums %d tracks\n", res->total_artists, res->total_albums, res->total_tracks);
        g_print("playlist has %d entries\n", pl->num_tracks);

        r = res_type(t, GLUE_SEARCH);

        r->search = res;
        r->pl = pl;
        r->tracks = pl->tracks;

        return r;
    } else {
        g_print("failed to search\n");
        return NULL;
    }
}

yaspot_glue_response_t* yaspot_glue_artist(yaspot_glue_response_t *r, gint num) {
    struct track *tr;
    struct artist_browse *artist;
    yaspot_glue_response_t *response = NULL;

    if (!r)
        return NULL;

    tr = get_nth_track(r->tracks, num-1);
    if (tr) {
        artist = despotify_get_artist(r->glue->session, tr->artist->id);

        if (artist) {
            response = res_type(r->glue, GLUE_ARTIST);
            response->artist = artist;
            response->tracks = NULL; /* FIXME */
            g_print("glue_artist FOUND\n");
        }
    }

    return response;
}

yaspot_glue_response_t* yaspot_glue_artist_album(yaspot_glue_response_t *r, gint num) {
    struct album_browse *a;
    yaspot_glue_response_t *response = NULL;
    gint i;

    if (!r || r->type != GLUE_ARTIST)
        return NULL;

    for (i = 0, a = r->artist->albums; a; i++, a = a->next) {
        if (i == num) {
            response = res_type(r->glue, GLUE_ALBUM);
            response->album = a;
            response->tracks = a->tracks;
            response->parent = r;
            yaspot_glue_response_ref(response->parent);
            g_print("glue: created album %d: %s / %s / %d \n", num, a->name, a->id, a->year);
        }
    }

    return response;
}

gchar* yaspot_glue_artist_description(yaspot_glue_response_t *r, gint max_chars) {
    gchar *response = NULL;
    gint len;

    if (!r || r->type != GLUE_ARTIST)
        return NULL;

    if (r->artist->text) {
        len = strlen(r->artist->text);
        if (max_chars <= 0)
            max_chars = len;
        response = g_strndup(r->artist->text, max_chars);
        if (response && len > max_chars && max_chars > 3)
            response[max_chars-1] = response[max_chars-2] = response[max_chars-3] = '.';
    }

    return response;
}

yaspot_glue_response_t* yaspot_glue_album(yaspot_glue_response_t *r, gint num) {
    struct track *tr;
    struct album_browse *album;
    yaspot_glue_response_t *response = NULL;

    if (!r)
        return NULL;

    tr = get_nth_track(r->tracks, num-1);
    if (tr) {
        album = despotify_get_album(r->glue->session, tr->album_id);

        if (album) {
            response = res_type(r->glue, GLUE_ALBUM);
            response->album = album;
            response->tracks = album->tracks;
            g_print("glue_album FOUND\n");
        }
    }

    return response;
}

gboolean yaspot_glue_album_is_single(yaspot_glue_response_t *r, gint single_limit) {
    if (!r)
        return FALSE;

    if (!yaspot_glue_album_is_compilation(r, single_limit) && r->album->num_tracks <= single_limit)
        return TRUE;
    else
        return FALSE;
}

gboolean yaspot_glue_album_is_compilation(yaspot_glue_response_t *r, gint artist_limit) {
    struct artist *artist = NULL;
    struct artist *prev_artist = NULL;
    struct track *tr;
    gint different_artists = 0;

    if (!r)
        return FALSE;

    for (tr = r->album->tracks; tr; tr = tr->next) {
        artist = tr->artist;

        if (prev_artist) {
            if (g_strcmp0(prev_artist->id, artist->id))
                different_artists++;
        }

        prev_artist = artist;
    }

    if (different_artists >= artist_limit)
        return TRUE;
    else
        return FALSE;
}

gboolean yaspot_glue_album_from_artist(yaspot_glue_response_t *album, yaspot_glue_response_t *artist) {
    struct track *tr;
    gint num = 0;
    gint others = 0;
    g_assert(album);
    g_assert(artist);

    for (tr = album->album->tracks; tr; tr = tr->next) {
        num++;
        if (g_strcmp0(tr->artist->id, artist->artist->id))
            others++;
    }

    /* if ratio of others in album is smaller or equal than 2/1,
     * this is artist's album */
    if (!others)
        return TRUE;

    if (num/others <= 2)
        return TRUE;
    else
        return FALSE;
}

int yaspot_glue_album_year(yaspot_glue_response_t *r) {
    if (!r ||r->type != GLUE_ALBUM)
        return -1;

    return r->album->year;
}

int yaspot_glue_artist_num_albums(yaspot_glue_response_t *r) {
    if (!r || r->type != GLUE_ARTIST)
        return -1;

    return r->artist->num_albums;
}

gint yaspot_glue_res_n_tracks(yaspot_glue_response_t *r) {
    struct track *tr;
    gint result = -1;
    g_assert(r);

    if (r->tracks) {
        for (result = 0, tr = r->tracks; tr; result++, tr = tr->next);
    }

    return result;
}

static void playlist_to_model(struct playlist *pl, GtkListStore *model) {
    struct track *tr;
    gint i;

    gtk_list_store_clear(model);
    for (i = 0, tr = pl->tracks; tr; tr = tr->next, i++) {
        gtk_list_store_insert_with_values(model, NULL, i,
                PLAYLIST_COL_TRACK, i+1,
                PLAYLIST_COL_TITLE, tr->title,
                PLAYLIST_COL_ARTIST, tr->artist->name,
                PLAYLIST_COL_ALBUM, tr->album,
                -1);
    }
}

static void stored_lists_to_model(struct playlist *pl, GtkListStore *model) {
    gint i;

    gtk_list_store_clear(model);

    for (i = 0; pl; pl = pl->next, i++) {
        gtk_list_store_insert_with_values(model, NULL, i,
                PLAYLISTS_COL_NUM, i+1,
                PLAYLISTS_COL_NAME, pl->name,
                PLAYLISTS_COL_AUTHOR, pl->author,
                -1);
    }

    g_print("%d stored playlists\n", i+1);
}

static void album_to_model(struct track *tracks, GtkListStore *model) {
    struct track *tr;
    gint i;

    gtk_list_store_clear(model);
    for (i = 0, tr = tracks; tr; tr = tr->next, i++) {
        gint min = tr->length/1000 / 60;
        gint sec = tr->length/1000 % 60;
        gchar *length = g_strdup_printf("%d:%02d", min, sec);

        gtk_list_store_insert_with_values(model, NULL, i,
                ALBUM_COL_TRACK, i+1,
                ALBUM_COL_TITLE, tr->title,
                ALBUM_COL_LENGTH, length,
                -1);
        g_free(length);
    }
}

void yaspot_glue_response_to_model(yaspot_glue_response_t *r, GtkListStore *model) {
    if (!r) {
        g_print("response_to_model NULL r\n");
        return;
    }

    switch (r->type) {
        case GLUE_SEARCH:
            playlist_to_model(r->pl, model);
            break;
        case GLUE_PLAYLISTS:
            stored_lists_to_model(r->pl, model);
            break;
        case GLUE_LIST:
            playlist_to_model(r->pl, model);
            break;
        case GLUE_ALBUM:
            album_to_model(r->album->tracks, model);
            break;
        case GLUE_ARTIST:
            /* TODO */
            break;
        default:
            g_print("response_to_model undefined case\n");
            break;
    }
}

void yaspot_glue_response_to_queue_model(yaspot_glue_response_t *r, GtkListStore *model) {
    struct track *tr, *tracks;
    gint i;

    g_assert(r);
    g_assert(model);
    g_assert(r->type != GLUE_PLAYLISTS);
    g_assert(r->type != GLUE_ARTIST);

    if (r->type == GLUE_ALBUM)
        tracks = r->album->tracks;
    else
        tracks = r->pl->tracks;

    gtk_list_store_clear(model);
    for (i = 0, tr = tracks; tr; i++, tr = tr->next) {
        gint min = tr->length/1000 / 60;
        gint sec = tr->length/1000 % 60;
        gchar *length = g_strdup_printf("%d:%02d", min, sec);
        gchar *artist = g_strdup_printf("(%s)", tr->artist->name);

        gtk_list_store_insert_with_values(model, NULL, i,
                QUEUE_COL_TRACK, i+1,
                QUEUE_COL_TITLE, tr->title,
                QUEUE_COL_ARTIST, artist,
                QUEUE_COL_LENGTH, length,
                -1);
        g_free(length);
        g_free(artist);
    }
}

yaspot_glue_response_t* yaspot_glue_stored_lists(yaspot_glue_t *t) {
    struct playlist *pl;
    yaspot_glue_response_t *response = NULL;

    pl = despotify_get_stored_playlists(t->session);

    if (pl) {
        response = res_type(t, GLUE_PLAYLISTS);
        response->pl = pl;
    }

    return response;
}

/* get nth track in playlist list, start from 0 */
static struct playlist* get_nth_playlist(struct playlist *playlists, gint num) {
    struct playlist *pl;
    gint i;

    for (pl = playlists, i = 0; pl; pl = pl->next, i++) {
        if (i == num)
            break;
    }

    if (i == num)
        return pl;
    else
        return NULL;
}

yaspot_glue_response_t* yaspot_glue_stored_playlist(yaspot_glue_response_t *r, gint num) {
    yaspot_glue_response_t *response = NULL;
    struct playlist *pl = NULL;
    //struct playlist *new_pl;

    if (r && r->type == GLUE_PLAYLISTS && r->pl && num > 0)
        pl = get_nth_playlist(r->pl, num-1);

    if (pl/* && (new_pl = despotify_get_playlist(r->glue->session, pl->playlist_id)) */ ) {

        response = res_type(r->glue, GLUE_LIST);
        response->parent = r;
        response->pl = pl;
        response->tracks = pl->tracks;
        yaspot_glue_response_ref(r); /* ref parent, so that it is not removed if this is used */
    }

    return response;
}

void* yaspot_glue_get_image(yaspot_glue_response_t *r, gint *len) {
    if (!r)
        return NULL;

    void *response = NULL;

    switch (r->type) {
        case GLUE_ALBUM:
            if (r->album->cover_id[0])
                response = despotify_get_image(r->glue->session, r->album->cover_id, len);
            break;
        case GLUE_ARTIST:
            if (r->artist && r->artist->portrait_id[0])
                response = despotify_get_image(r->glue->session, r->artist->portrait_id, len);
            break;
        default:
            /* shouldn't be here */
            break;
    }

    return response;
}

gchar* yaspot_glue_response_get_name(yaspot_glue_response_t *r) {
    if (!r)
        return NULL;

    gchar *response = NULL;

    switch (r->type) {
        case GLUE_LIST:
            response = g_strdup_printf("%s (%s)", r->pl->name, r->pl->author);
            break;
        case GLUE_ALBUM:
            if (r->album->year > 0)
                response = g_strdup_printf("%s (%d)", r->album->name, r->album->year);
            else
                response = g_strdup(r->album->name);
            break;
        case GLUE_ARTIST:
            response = g_strdup(r->artist->name);
            break;
        default:
            g_print("undef response_get_name\n");
            break;
    }

    return response;
}

int yaspot_glue_play(yaspot_glue_response_t *r, gint num) {
    struct track *tr;
    yaspot_glue_response_t *to_unref = NULL;
    gint ret = -1;

    g_static_rw_lock_writer_lock(&data_lock);
    if (r->glue->current) {
        r->glue->playing = FALSE;

        to_unref = r->glue->current;
        r->glue->current = NULL;
        thread_pause(r->glue);
    }
    g_static_rw_lock_writer_unlock(&data_lock);

    tr = get_nth_track(r->tracks, num-1);

    if (!tr) {
        set_current(r->glue, NULL);
        set_current_track(r->glue, NULL);
        goto end;
    }

    if (despotify_play(r->glue->session, tr, TRUE)) {
        set_current(r->glue, r);
        yaspot_glue_response_ref(r->glue->current);
        g_print("playing %s\n", tr->title);
        r->glue->playing = TRUE;

        /* FIXME why does policy cork us if we request PLAY :( */
        //pb_playback_req_state(r->glue->pb, PB_STATE_PLAY, playback_req_cb, r->glue);

        thread_play(r->glue);

        ret = 0;
    } else {
        g_print("yaspot_glue_play failed\n");
    }

end:
    if (to_unref)
        yaspot_glue_response_unref(to_unref);

    return ret;
}

int yaspot_glue_set_pause(yaspot_glue_t *t, gboolean pause) {
    if (!t->playing) {
        g_print("set_pause, nothing playing atm, cannot set pause\n");
        return -1;
    }

    if (pause) {
        if (!t->paused) {
            thread_pause(t);
            //pb_playback_req_state(t->pb, PB_STATE_STOP, playback_req_cb, t);
            g_print("pausing playback\n");
        }
    } else {
        if (t->paused) {
            thread_play(t);
            //pb_playback_req_state(t->pb, PB_STATE_PLAY, playback_req_cb, t);
            g_print("resuming playback\n");
        }
    }

    emit_state_changed(t);

    return 0;
}

int yaspot_glue_stop(yaspot_glue_t* t) {
    if (t->playing) {
        thread_pause(t);
        despotify_stop(t->session);
    }
    if (t->current) {
        yaspot_glue_response_unref(t->current);
        t->current = NULL;
    }
    t->playing = FALSE;
    g_print("stopping playback\n");

    emit_state_changed(t);

    return 0;
}

int yaspot_glue_res_stop(yaspot_glue_response_t* t) {
    return yaspot_glue_stop(t->glue);
}

int yaspot_glue_next(yaspot_glue_t *t) {
    struct track *tr;
    g_assert(t);

    if (!t->playing || !t->current) {
        g_print("next: no current playlist\n");
        return -1;
    }

    /* nth_track function starts counting from 0 */
    tr = get_nth_track(t->current->tracks, t->current_track_num);
    if (!tr) {
        g_print("next: unknown next track\n");
        return -1;
    }

    return yaspot_glue_play(t->current, t->current_track_num+1);
}

int yaspot_glue_prev(yaspot_glue_t *t) {
    g_assert(t);

    if (!t->playing || !t->current) {
        g_print("prev: no current playlist\n");
        return -1;
    }
    if (t->current_track_num <= 1) {
        g_print("prev: already first track\n");
        return -1;
    }

    return yaspot_glue_play(t->current, t->current_track_num-1);
}

yaspot_glue_response_t* yaspot_glue_current_queue(yaspot_glue_t *t) {
    yaspot_glue_response_ref(t->current);
    return t->current;
}

yaspot_glue_response_t* yaspot_glue_res_current_queue(yaspot_glue_response_t *r) {
    return yaspot_glue_current_queue(r->glue);
}

gboolean yaspot_glue_playing(yaspot_glue_t *t) {
    g_assert(t);

    return (t->playing && !t->paused);
}

gchar* yaspot_glue_current_track(yaspot_glue_t *t) {
    gchar *response = NULL;
    g_assert(t);

    g_static_rw_lock_reader_lock(&data_lock);

    if (t->current_track) {
        if (t->current_track->artist && t->current_track->title)
            response = g_strdup_printf("%s: %s", t->current_track->artist->name, t->current_track->title);
        if (!response)
            response = g_strdup_printf("<undef>");
    }
    g_static_rw_lock_reader_unlock(&data_lock);

    return response;
}

gint yaspot_glue_current_track_num(yaspot_glue_t *t) {
    gint response = -1;
    struct track *tr;
    gint i;

    g_assert(t);

    g_static_rw_lock_reader_lock(&data_lock);

    if (t->current) {
        i = 1;

        for (tr = t->current->tracks; tr != t->current_track; tr = tr->next)
            i++;

        response = i;
    }
    g_static_rw_lock_reader_unlock(&data_lock);

    return response;
}

// vim: expandtab shiftwidth=4

