#include <hildon/hildon.h>

#include <locale.h>
#include <config.h>
#include <glib/gi18n.h>

#include "userdata.h"
#include "views.h"
#include "view-main.h"
#include "settings.h"
#include "app-menu.h"

#include "glue.h"

static void show_search(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;
    const gchar *search;

    if (gtk_entry_get_text_length(GTK_ENTRY(u->main.entry_search))) {
        search = gtk_entry_get_text(GTK_ENTRY(u->main.entry_search));
        g_print("search with %s\n", search);
        yaspot_show_search(u, search);
    }
}

static void show_playlists(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_stored_playlists(u);
}

static void show_queue(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_queue(u);
}

GtkWidget* yaspot_view_main_new(userdata_t *u) {
    GtkWidget *win;
    GtkWidget *button_search;
    GtkWidget *button_playlists;
    GtkWidget *button_queue;
    GtkWidget *entry_search;
    GtkWidget *vbox;
    GtkWidget *search_box;

    win = hildon_stackable_window_new();
    gtk_window_set_title(GTK_WINDOW(win), "yaspot");

    button_search = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                _("Search"),
                                                NULL);

    button_playlists = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                   HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                   _("Stored Playlists"),
                                                   NULL);

    button_queue = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                               HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                               _("Playing Queue"),
                                               NULL);

    entry_search = hildon_entry_new(HILDON_SIZE_AUTO);
    u->main.entry_search = entry_search;

    vbox = gtk_vbox_new(FALSE, 0);
    search_box = gtk_hbox_new(FALSE, 0);


    gtk_box_pack_start_defaults(GTK_BOX(search_box), button_search);
    gtk_box_pack_start_defaults(GTK_BOX(search_box), entry_search);

    gtk_box_pack_start(GTK_BOX(vbox),
                       search_box,
                       FALSE,
                       FALSE,
                       10);
    gtk_box_pack_start(GTK_BOX(vbox),
                       button_playlists,
                       FALSE,
                       FALSE,
                       10);
    gtk_box_pack_start(GTK_BOX(vbox),
                       button_queue,
                       FALSE,
                       FALSE,
                       10);

    gtk_container_add(GTK_CONTAINER(win), vbox);

    /* search button & entry */
    g_signal_connect(button_search, "clicked", G_CALLBACK(show_search), u);
    g_signal_connect(entry_search, "activate", G_CALLBACK(show_search), u);

    /* playlists */
    g_signal_connect(button_playlists, "clicked", G_CALLBACK(show_playlists), u);

    /* playing queue */
    g_signal_connect(button_queue, "clicked", G_CALLBACK(show_queue), u);

    /* x */
    g_signal_connect(G_OBJECT(win), "delete_event",
                     G_CALLBACK(gtk_main_quit), NULL);

    return win;
}

enum {
    BUTTON_LOGIN = 9999,
    BUTTON_OFFLINE
};

static void display_login(userdata_t *u);

static void try_login(userdata_t *u, GtkWidget *dialog, const gchar *username, const gchar *password, gint high_bitrate) {
    if (!yaspot_glue_auth(u->session, username, password, high_bitrate)) {
        yaspot_glue_print_info(u->session);
        u->connection.connected = TRUE;
        gtk_widget_destroy(dialog);
    } else {
        g_print("failed to log in\n");
        gtk_widget_destroy(dialog);
        display_login(u);
    }
}

static void display_login(userdata_t *u) {
    GtkWidget *dialog;
    GtkWidget *caption_user;
    GtkWidget *caption_pass;
    GtkWidget *entry_user;
    GtkWidget *entry_pass;
    GtkSizeGroup *group;
    GtkWidget *content_area;
    GtkWidget *save_credentials;
    GtkWidget *high_bitrate;
    gchar *username = NULL;
    gchar *password = NULL;
    gint result;
    gint quit;
    gint save_cred;
    gint high_br;

    quit = 0;

    dialog = gtk_dialog_new();
    gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(u->main.window));
    gtk_window_set_title(GTK_WINDOW(dialog), _("Login"));

    entry_user = hildon_entry_new(HILDON_SIZE_AUTO);
    entry_pass = hildon_entry_new(HILDON_SIZE_AUTO);
    g_object_set (entry_user, "hildon-input-mode", HILDON_GTK_INPUT_MODE_FULL, NULL);
    g_object_set (entry_pass, "hildon-input-mode", HILDON_GTK_INPUT_MODE_FULL, NULL);
    gtk_entry_set_visibility(GTK_ENTRY(entry_pass), FALSE);

    group = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);

    caption_user = hildon_caption_new(group, _("Username"), entry_user, NULL, HILDON_CAPTION_MANDATORY);
    caption_pass = hildon_caption_new(group, _("Password"), entry_pass, NULL, HILDON_CAPTION_MANDATORY);
    g_object_unref(group);

    save_credentials = hildon_check_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label(GTK_BUTTON(save_credentials), _("Save credentials"));
    high_bitrate = hildon_check_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label(GTK_BUTTON(high_bitrate), _("High bitrate"));

    /* settings stuff */
    if (!yaspot_settings_save_credentials(&save_cred)) {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(save_credentials), save_cred);

        if (save_cred) {
            username = yaspot_settings_username();
            password = yaspot_settings_password();

            if (username)
                hildon_entry_set_text(HILDON_ENTRY(entry_user), username);
            if (password)
                hildon_entry_set_text(HILDON_ENTRY(entry_pass), password);

            g_free(username);
            g_free(password);
        }
    } else {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(save_credentials), 0);
    }

    if (!yaspot_settings_high_bitrate(&high_br)) {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(high_bitrate), high_br);
    }

    gtk_dialog_add_button(GTK_DIALOG(dialog), _("Login"), BUTTON_LOGIN);
    /* OFFLINE disabled for now. */
    /* gtk_dialog_add_button(dialog, "Offline", BUTTON_OFFLINE); */

    content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));

    gtk_box_pack_start(GTK_BOX(content_area),
                       caption_user,
                       FALSE,
                       FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(content_area),
                       caption_pass,
                       FALSE,
                       FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(content_area),
                       save_credentials,
                       FALSE,
                       FALSE,
                       4);
    gtk_box_pack_start(GTK_BOX(content_area),
                       high_bitrate,
                       FALSE,
                       FALSE,
                       2);

    gtk_widget_show_all(GTK_WIDGET(dialog));
    result = gtk_dialog_run(GTK_DIALOG(dialog));

    if (hildon_check_button_get_active(HILDON_CHECK_BUTTON(save_credentials))) {
        yaspot_settings_set_save_credentials(1);
        yaspot_settings_set_username(hildon_entry_get_text(HILDON_ENTRY(entry_user)));
        yaspot_settings_set_password(hildon_entry_get_text(HILDON_ENTRY(entry_pass)));
    } else {
        yaspot_settings_set_save_credentials(0);
        yaspot_settings_set_username("foo");
        yaspot_settings_set_password("bar");
    }

    high_br = hildon_check_button_get_active(HILDON_CHECK_BUTTON(high_bitrate));
    yaspot_settings_set_high_bitrate(high_br);

    switch (result) {
        case BUTTON_LOGIN:
            g_print("try login\n");
            try_login(u,
                      dialog,
                      hildon_entry_get_text(HILDON_ENTRY(entry_user)),
                      hildon_entry_get_text(HILDON_ENTRY(entry_pass)),
                      hildon_check_button_get_active(HILDON_CHECK_BUTTON(high_bitrate)));
            break;
        case BUTTON_OFFLINE:
            g_print("offline not supported\n");
            gtk_widget_destroy(dialog);
            quit = 1;
            break;
        default:
            g_print("cancelled -> quit\n");
            gtk_widget_destroy(dialog);
            quit = 1;
            break;
    }

    if (quit) {
        gtk_widget_destroy(u->main.window);
    }
}

void yaspot_show_main(userdata_t *u) {
    if (!u->main.window) {
        u->main.window = yaspot_view_main_new(u);
    }

    yaspot_app_menu_attach(u, HILDON_WINDOW(u->main.window));

    gtk_widget_show_all(u->main.window);

    if (!u->connection.connected) {
        display_login(u);
    }
}

// vim: expandtab

