#include <hildon/hildon.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <locale.h>
#include <config.h>
#include <glib/gi18n.h>

#include "userdata.h"
#include "playlist.h"
#include "glue.h"
#include "app-menu.h"
#include "views.h"

typedef struct progress_t {
    //GtkWidget *window;
    GtkWidget *dialog;
    GtkWidget *progress;

    gdouble fract;
    gboolean stop;
    gboolean updated;
} progress_t;

/* view */
typedef struct yaspot_album_t {
    yaspot_glue_response_t *r;
    GtkWidget *vbox_all;
    GtkWidget *label;
    GtkWidget *hbox;
    GtkWidget *bin;
    GtkTreeView *tree;
    GtkWidget *image;
} yaspot_album_t;

typedef struct yaspot_view_t {
    GtkWidget *window;
    HildonPannableArea *area;
    GtkTreeView *treeview;
    GtkWidget *image;

    /* for play queue timer */
    GtkTreeSelection *selection;
    gint current_playback_row;
    GtkProgressBar *timer;
    GThread *ui_thread;
    GAsyncQueue *timer_queue;
    HildonButton *button_pp;
    gboolean old_paused;

    GList *artist_view;
    progress_t progress;

    userdata_t *u;
    yaspot_glue_response_t *r;
} yaspot_view_t;

static gint artist_view_block = FALSE;

/* *** common *** */

static const gint32 STOP_TIMER_LOOP = -1;
static const gint32 UPDATE_SELECTION = -2;
static const gint32 UPDATE_PLAYPAUSE = -3;

static void state_changed_cb(gboolean playing, gboolean paused, gpointer data);

static yaspot_view_t* view_new(userdata_t *u) {
    yaspot_view_t *v;
    g_assert(u);
    v = g_new0(yaspot_view_t, 1);
    v->u = u;
    g_print("new view %p\n", v);
    return v;
}

static void view_free(yaspot_view_t *v) {
    if (v->r)
        yaspot_glue_response_unref(v->r);
    g_free(v);
}

static void destroy(GtkWidget *widget, GdkEvent *event, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    g_print("destroy view %p\n", v);
    gtk_widget_destroy(widget);
    view_free(v);
}

static void free_artist(gpointer data, gpointer userdata) {
    yaspot_album_t *a = (yaspot_album_t*)data;
    yaspot_glue_response_unref(a->r);
    g_free(a);
}

static void artist_destroy(GtkWidget *widget, GdkEvent *event, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    g_print("destroy artist view %p\n", v);
    g_list_foreach(v->artist_view, free_artist, NULL);
    g_list_free(v->artist_view);
    g_print("and now foo\n");
    destroy(widget, event, data);
}

static void queue_destroy(GtkWidget *widget, GdkEvent *event, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;

    if (v->timer) {
        yaspot_glue_set_timer_cb(v->u->session, NULL, NULL);
        g_async_queue_push(v->timer_queue, GINT_TO_POINTER(STOP_TIMER_LOOP));
        g_print("queue_destroy: wait for thread to terminate\n");
        g_thread_join(v->ui_thread);

        g_async_queue_unref(v->timer_queue);
    }

    yaspot_glue_del_state_cb(v->u->session, state_changed_cb, v);

    destroy(widget, event, data);
}

static yaspot_view_t* new_basic_view(yaspot_view_t *v) {
    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );
    v->treeview = yaspot_playlist_view_new();
    //hildon_pannable_area_add_with_viewport(v->area, GTK_WIDGET(v->treeview));
    gtk_container_add(GTK_CONTAINER(v->area), GTK_WIDGET(v->treeview));

    gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(v->area));

    yaspot_app_menu_attach(v->u, HILDON_WINDOW(v->window));

    return v;
}

static void row_activated(GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    GtkTreeIter iter;
    GtkTreeModel *model;
    gint value = -1;

    if (artist_view_block) {
        g_print("views: artist view block set, returning..\n");
        return;
    }

    model = gtk_tree_view_get_model(treeview);
    if (!gtk_tree_model_get_iter(model, &iter, path)) {
        g_print("failed to get iterator\n");
        return;
    }

    gtk_tree_model_get(model, &iter, PLAYLIST_COL_TRACK, &value, -1);

    if (value > -1) {
        g_print("playlist: row_activated, row %d column %s\n", value, gtk_tree_view_column_get_title(column));

        if (column == gtk_tree_view_get_column(treeview, PLAYLIST_COL_TITLE))
            yaspot_glue_play(v->r, value);
        else if (column == gtk_tree_view_get_column(treeview, PLAYLIST_COL_ALBUM))
            yaspot_show_album(v->u, v->r, value);
        else if (column == gtk_tree_view_get_column(treeview, PLAYLIST_COL_ARTIST))
            yaspot_show_artist(v->u, v->r, value, v->window);
    }
}

static void artist_view_row_activated(GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data) {
    yaspot_album_t *a = (yaspot_album_t*)data;
    GtkTreeIter iter;
    GtkTreeModel *model;
    gint value = -1;

    model = gtk_tree_view_get_model(treeview);

    if (!gtk_tree_model_get_iter(model, &iter, path)) {
        g_print("failed to get iterator\n");
        return;
    }

    gtk_tree_model_get(model, &iter, ALBUM_COL_TRACK, &value, -1);

    if (value > -1) {
        g_print("playlist: artist_view_row_activated, row %d\n", value);

        yaspot_glue_play(a->r, value);
    }
}

/* *** playlists *** */

static yaspot_view_t* new_playlist_view(yaspot_view_t *v) {
    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );
    v->treeview = yaspot_playlists_view_new();
    gtk_container_add(GTK_CONTAINER(v->area), GTK_WIDGET(v->treeview));

    gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(v->area));

    yaspot_app_menu_attach(v->u, HILDON_WINDOW(v->window));

    return v;
}

static void lists_row_activated(GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    GtkTreeIter iter;
    GtkTreeModel *model;
    gint value;

    model = gtk_tree_view_get_model(treeview);
    if (!gtk_tree_model_get_iter(model, &iter, path)) {
        g_print("failed to get iterator\n");
        return;
    }

    gtk_tree_model_get(model, &iter, PLAYLIST_COL_TRACK, &value, -1);

    if (value > -1) {
        g_print("load playlist %d\n", value);

        yaspot_show_stored_list(v->u, v->r, value);
    }
}

void yaspot_show_stored_list(userdata_t *u, yaspot_glue_response_t *list, gint num) {

    gchar *playlist_name;
    GtkListStore *model;
    yaspot_glue_response_t *r;
    yaspot_view_t *v;

    if (!(r = yaspot_glue_stored_playlist(list, num))) {
        g_print("couldn't get stored playlist %d\n", num);
        return;
    }

    v = new_basic_view(view_new(u));
    v->r = r;

    model = GTK_LIST_STORE(gtk_tree_view_get_model(v->treeview));
    yaspot_glue_response_to_model(r, model);

    playlist_name = yaspot_glue_response_get_name(r);

    if (playlist_name) {
        gtk_window_set_title(GTK_WINDOW(v->window), playlist_name);
        g_free(playlist_name);
    } else {
        gtk_window_set_title(GTK_WINDOW(v->window), _("Playlist"));
    }

    g_signal_connect(G_OBJECT(v->treeview), "row-activated", G_CALLBACK(row_activated), v);

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(destroy), v);

    gtk_widget_show_all(v->window);
}

void yaspot_show_stored_playlists(userdata_t *u) {
    GtkListStore *model;
    yaspot_view_t *v;
    yaspot_glue_response_t *r;

    g_print("get playlists\n");
    if (!(r = yaspot_glue_stored_lists(u->session))) {
        g_print("couldn't get stored playlists\n");
        return;
    }
    g_print("get playlists done\n");

    v = new_playlist_view(view_new(u));
    v->r = r;

    model = GTK_LIST_STORE(gtk_tree_view_get_model(v->treeview));
    yaspot_glue_response_to_model(r, model);

    g_signal_connect(G_OBJECT(v->treeview), "row-activated", G_CALLBACK(lists_row_activated), v);

    gtk_window_set_title(GTK_WINDOW(v->window), _("Stored Playlists"));

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(destroy), v);

    gtk_widget_show_all(v->window);
}

void yaspot_show_search(userdata_t *u, const gchar *search) {
    yaspot_view_t *v;
    yaspot_glue_response_t *r;

    if (!search)
        return;

    if (!(r = yaspot_glue_search(u->session, search, 100))) {
        g_print("show search failed\n");
        return;
    }


    v = new_basic_view(view_new(u));
    v->r = r;

    gchar *label = g_strdup_printf(_("Search - %s"), search);
    gtk_window_set_title(GTK_WINDOW(v->window), label);
    g_free(label);

    yaspot_glue_response_to_model(r, GTK_LIST_STORE(gtk_tree_view_get_model(v->treeview)));

    g_signal_connect(G_OBJECT(v->treeview), "row-activated", G_CALLBACK(row_activated), v);

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(destroy), v);

    gtk_widget_show_all(v->window);
}

static gint image_to(yaspot_glue_response_t *r, GtkWidget *image) {
    GdkPixbufLoader *loader;
    loader = gdk_pixbuf_loader_new();
    GError *error = NULL;
    gint len = 0;
    void *img = NULL;
    gint ret = -1;

    img = yaspot_glue_get_image(r, &len);

    if (img && len > 0) {
        gdk_pixbuf_loader_write(loader, img, len, &error);
        if (!error) {
            GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
            gtk_image_set_from_pixbuf(GTK_IMAGE(image), pixbuf);
            ret = 0;
        } else {
            g_print("error loading image: %s\n", error->message);
        }
        g_free(img);
    }

    return ret;
}

static yaspot_view_t* new_artist_view(yaspot_view_t *v) {
    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );
//    v->treeview = yaspot_playlist_view_new();
//    hildon_pannable_area_add_with_viewport(v->area, GTK_WIDGET(v->treeview));

    gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(v->area));

    return v;
}

static gboolean proge_cb(gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    progress_t *p = &v->progress;

    if (p->updated) {
        g_print("SETTING PROGE\n");
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(p->progress), p->fract);
        p->updated = FALSE;
    }

    if (p->stop) {
        /* now we destroy our progress bar and display what our thread has come up with */

        g_signal_connect(G_OBJECT(v->window), "delete_event",
                         G_CALLBACK(artist_destroy), v);

        gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(v->area));

        yaspot_app_menu_attach(v->u, HILDON_WINDOW(v->window));

        gtk_widget_show_all(v->window);

        gtk_widget_destroy(p->dialog);
        p->dialog = p->progress = NULL;

        return FALSE;
    }

    return TRUE;
}

static void dummy(void) {
}

static void proge(yaspot_view_t *v, GtkWidget *window) {
    progress_t *p = &v->progress;
    GtkWidget *content_area;

    p->stop = FALSE;
    p->updated = TRUE;
    p->fract = 0.0;
    p->progress = gtk_progress_bar_new();
    //gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progress), "Loading");
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(p->progress), (gdouble)0);
    p->dialog = gtk_dialog_new();

    gtk_window_set_transient_for(GTK_WINDOW(p->dialog), GTK_WINDOW(window));
    //gtk_window_set_transient_for(GTK_WINDOW(p->dialog), NULL);
    gtk_window_set_title(GTK_WINDOW(p->dialog), _("Loading Artist"));

    content_area = gtk_dialog_get_content_area (GTK_DIALOG (p->dialog));

    gtk_box_pack_start(GTK_BOX(content_area),
                       p->progress,
                       FALSE,
                       FALSE,
                       10);

    g_signal_connect(G_OBJECT(p->dialog), "delete_event",
                     dummy, NULL);

    gtk_widget_show_all(GTK_WIDGET(p->progress));
    gtk_widget_show_all(GTK_WIDGET(p->dialog));
    g_timeout_add_seconds(1, proge_cb, v);
}

static void proge_stop(progress_t *p) {
    artist_view_block = FALSE;

    p->stop = TRUE;
}

static void proge_update(progress_t *p, gdouble fract) {
    p->fract = fract;
    p->updated = TRUE;
}

static void artist_show_biography(GtkWidget *widget, gpointer userdata) {
    yaspot_view_t *v = (yaspot_view_t*)userdata;
    g_assert(v);

    yaspot_show_biography(v->u, v->r);
}

gpointer artist_thread(gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    yaspot_glue_response_t *new_r;
    GtkWidget *vbox;
    GtkWidget *vbox_info;
    GtkWidget *hbox_info;
    GtkWidget *info_image;
    GtkWidget *info_text;
    gboolean biography_found = FALSE;
    GtkWidget *info_biography_button;
    GtkWidget *vbox_albums = NULL;
    GtkWidget *vbox_singles = NULL;
    GtkWidget *vbox_compilations = NULL;
    GtkWidget *vbox_label;
    gchar *text;
    gint num_albums = 0;
    gint i;

    /* FIXME MAGIC NUMBERS */
    const gint MAX_SINGLES = 8;
    const gint MAX_COMPILATIONS = 4;
    gint compilations = 0;
    gint singles = 0;

    enum {
        album_single = 0,
        album_compilation,
        album_regular
    } album_type;

    num_albums = yaspot_glue_artist_num_albums(v->r);

    vbox = gtk_vbox_new(FALSE, 30);

    vbox_info = gtk_vbox_new(FALSE, 30);

    /* INFO */
    hbox_info = gtk_hbox_new(FALSE, 30);
    info_image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU);
    text = yaspot_glue_artist_description(v->r, 300);
    if (text) {
        info_text = gtk_label_new(text);
        biography_found = TRUE;
        g_free(text);
    } else
        info_text = gtk_label_new(_("No Biography."));
    gtk_label_set_line_wrap(GTK_LABEL(info_text), TRUE);
    gtk_widget_set_size_request(info_image, 280, 280);
    gtk_widget_set_size_request(info_text, 450, -1); /* FIXME MAGIC NUMBER */
    image_to(v->r, info_image);

    gtk_box_pack_start(GTK_BOX(hbox_info), info_image, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox_info), info_text, FALSE, FALSE, 0);

    /* add to big box */
    gtk_box_pack_start(GTK_BOX(vbox), hbox_info, FALSE, FALSE, 0);
    if (biography_found) {
        info_biography_button = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                            HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                            _("View Biography"),
                                                            NULL);
        g_signal_connect(info_biography_button, "clicked", G_CALLBACK(artist_show_biography), v);
        gtk_box_pack_start(GTK_BOX(vbox), info_biography_button, FALSE, FALSE, 0);
    }

    /* ALBUMS */

    for (i = 0; i < num_albums; i++) {
        gchar *label;
        album_type = album_regular;

        new_r = yaspot_glue_artist_album(v->r, i);
        g_print("album %d\n", i);

        if (!yaspot_glue_album_from_artist(new_r, v->r)) {
            g_print("is not from our artist\n");
            yaspot_glue_response_unref(new_r);
            continue;
        } else if (yaspot_glue_album_is_single(new_r, 4)) {
            g_print("is single\n");
            if (singles >= MAX_SINGLES) {
                yaspot_glue_response_unref(new_r);
                g_print("skipping single, too many :O\n");
                continue;
            }
            singles++;
            album_type = album_single;
        } else if (yaspot_glue_album_is_compilation(new_r, 4)) {
            g_print("is compilation\n");
            if (compilations >= MAX_COMPILATIONS) {
                yaspot_glue_response_unref(new_r);
                g_print("skipping compilation, too many :O\n");
                continue;
            }
            compilations++;
            album_type = album_compilation;
        }

        yaspot_album_t *a = g_new0(yaspot_album_t, 1);
        v->artist_view = g_list_append(v->artist_view, a);

        a->r = new_r;
        a->vbox_all = gtk_vbox_new(FALSE, 10);
        a->hbox = gtk_hbox_new(FALSE, 10);
        a->image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU);
        gtk_widget_set_size_request(a->image, 280, 280); /* FIXME MAGIC NUMBER */
        a->tree = yaspot_playlist_album_new();

        /* I guess this is quite stupid way to do this, but which other container
         * enables HILDON_UI_MODE_NORMAL with treeview??? */
        a->bin = hildon_pannable_area_new_full(HILDON_PANNABLE_AREA_MODE_AUTO, TRUE, 0, 0, 0, 2);
        gtk_widget_set_size_request(a->bin, -1, yaspot_glue_res_n_tracks(a->r)*70); /* FIXME MAGIC NUMBER */
        gtk_container_add(GTK_CONTAINER(a->bin), GTK_WIDGET(a->tree));

        label = yaspot_glue_response_get_name(a->r);
        a->label = gtk_label_new(label);
        gtk_widget_set_size_request(a->label, 450, -1); /* FIXME MAGIC NUMBER */
        g_free(label);

        yaspot_glue_response_to_model(a->r, GTK_LIST_STORE(gtk_tree_view_get_model(a->tree)));
        image_to(a->r, a->image);

        gtk_box_pack_start(GTK_BOX(a->hbox), a->image, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(a->hbox), GTK_WIDGET(a->bin), TRUE, TRUE, 0);

        gtk_box_pack_start(GTK_BOX(a->vbox_all), a->label, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(a->vbox_all), a->hbox, FALSE, FALSE, 0);

        switch (album_type) {
            case album_regular:
                if (!vbox_albums)
                    vbox_albums = gtk_vbox_new(FALSE, 30);
                gtk_box_pack_start(GTK_BOX(vbox_albums), a->vbox_all, FALSE, FALSE, 0);
                break;
            case album_single:
                if (!vbox_singles)
                    vbox_singles = gtk_vbox_new(FALSE, 30);
                gtk_box_pack_start(GTK_BOX(vbox_singles), a->vbox_all, FALSE, FALSE, 0);
                break;
            case album_compilation:
                if (!vbox_compilations)
                    vbox_compilations = gtk_vbox_new(FALSE, 30);
                gtk_box_pack_start(GTK_BOX(vbox_compilations), a->vbox_all, FALSE, FALSE, 0);
                break;
        }

        g_signal_connect(G_OBJECT(a->tree), "row-activated", G_CALLBACK(artist_view_row_activated), a);

        proge_update(&v->progress, (gdouble)(i+1)/(gdouble)(num_albums));
        g_print("fraction set to %f\n", (gdouble)(i+1)/(gdouble)(num_albums));
    }

    gtk_box_pack_start(GTK_BOX(vbox), vbox_info, FALSE, FALSE, 0);
    if (vbox_albums) {
        vbox_label = gtk_label_new(_("Albums"));
        gtk_box_pack_start(GTK_BOX(vbox), vbox_label, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(vbox), vbox_albums, FALSE, FALSE, 0);
    }
    if (vbox_singles) {
        vbox_label = gtk_label_new(_("Singles"));
        gtk_box_pack_start(GTK_BOX(vbox), vbox_label, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(vbox), vbox_singles, FALSE, FALSE, 0);
    }
    if (vbox_compilations) {
        vbox_label = gtk_label_new(_("Appears on"));
        gtk_box_pack_start(GTK_BOX(vbox), vbox_label, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(vbox), vbox_compilations, FALSE, FALSE, 0);
    }


    hildon_pannable_area_add_with_viewport(v->area, GTK_WIDGET(vbox));

    proge_stop(&v->progress);

    return NULL;
}

void yaspot_show_artist(userdata_t *u, yaspot_glue_response_t *list, gint num, GtkWidget *window) {
    yaspot_view_t *v;
    yaspot_glue_response_t *r;
    GThread *thread;

    artist_view_block = TRUE;

    if (!(r = yaspot_glue_artist(list, num))) {
        g_warning("show artist failed\n");
        return;
    }

    v = new_artist_view(view_new(u));
    v->r = r;

    //struct proge_t pro;

    //proge(&pro);

    gchar *label = yaspot_glue_response_get_name(r);
    if (label) {
        gtk_window_set_title(GTK_WINDOW(v->window), label);
        g_free(label);
    }

    proge(v, window);

    thread = g_thread_create(artist_thread, v, TRUE, NULL);
}

static yaspot_view_t* new_album_view(yaspot_view_t *v) {

    GtkWidget *hbox;

    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );
    v->treeview = yaspot_playlist_album_new();
    v->image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU);
    hbox = gtk_hbox_new(FALSE, 10);

    gtk_container_add(GTK_CONTAINER(v->area), GTK_WIDGET(v->treeview));

    gtk_box_pack_start(GTK_BOX(hbox), v->image, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(v->area), TRUE, TRUE, 0);

    gtk_container_add(GTK_CONTAINER(v->window), hbox);

    yaspot_app_menu_attach(v->u, HILDON_WINDOW(v->window));

    return v;
}

void yaspot_show_album(userdata_t *u, yaspot_glue_response_t *list, gint num) {
    yaspot_view_t *v;
    yaspot_glue_response_t *r;

    if (!(r = yaspot_glue_album(list, num))) {
        g_warning("show album failed\n");
        return;
    }

    v = new_album_view(view_new(u));
    v->r = r;

    gchar *label = yaspot_glue_response_get_name(r);
    if (label) {
        gtk_window_set_title(GTK_WINDOW(v->window), label);
        g_free(label);
    }

    yaspot_glue_response_to_model(r, GTK_LIST_STORE(gtk_tree_view_get_model(v->treeview)));

    image_to(r, v->image);

    g_signal_connect(G_OBJECT(v->treeview), "row-activated", G_CALLBACK(row_activated), v);

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(destroy), v);

    gtk_widget_show_all(v->window);
}

static void update_timer(yaspot_glue_t *t, gint now, gint length, gpointer userdata) {
    yaspot_view_t *v = (yaspot_view_t*)userdata;
    guint32 msg;
    g_assert(v);

    /* only update timer while display is on */
    if (!v->u->display_on)
        return;

    msg = (guint32)now + ((guint32)length << 16);

    g_async_queue_push(v->timer_queue, GINT_TO_POINTER((guint32)msg));
}

static void update_selection(yaspot_view_t *v, gboolean scroll) {
    GtkTreeModel *model;
    GtkTreeIter iter;
    gint new_row;
    gint scroll_pos;

    g_assert(v);
    g_assert(v->selection);
    g_assert(v->treeview);

    new_row = yaspot_glue_current_track_num(v->u->session);

    g_print("update_selection: old row %d new row %d\n", v->current_playback_row, new_row);

    if (v->current_playback_row == new_row)
        return;

    v->current_playback_row = new_row;

    model = gtk_tree_view_get_model(v->treeview);
    gtk_tree_model_iter_nth_child(model, &iter, NULL, v->current_playback_row - 1);

    gtk_tree_selection_select_iter(v->selection, &iter);

    /* FIXME MAGIC NUMBER */
    scroll_pos = v->current_playback_row * 70 + 35;
    if (scroll)
        hildon_pannable_area_scroll_to(v->area, -1, scroll_pos);
    else
        hildon_pannable_area_jump_to(v->area, -1, scroll_pos);
}

/* use two unsigned 16bit values in 32bit variable, pass as pointer.
 * value 1 stops thread */
static void* ui_thread_loop(gpointer userdata) {
    yaspot_view_t *v = (yaspot_view_t*)userdata;
    guint32 msg;
    gint now;
    gint length;
    gdouble fraction;
    gint loop = 1;
    gchar text[32];
    gboolean paused;
    g_assert(v);

    g_async_queue_ref(v->timer_queue);

    while (loop) {
        msg = (gint32)GPOINTER_TO_INT(g_async_queue_pop(v->timer_queue));

        if (msg == STOP_TIMER_LOOP) {
            loop = 0;
        } else if (msg == UPDATE_SELECTION) {

            update_selection(v, TRUE);

        } else if (msg == UPDATE_PLAYPAUSE) {

            paused = !yaspot_glue_playing(v->u->session);
            if (paused != v->old_paused) {
                if (paused)
                    hildon_button_set_image(v->button_pp, gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON));
                else
                    hildon_button_set_image(v->button_pp, gtk_image_new_from_stock(GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_BUTTON));
                v->old_paused = paused;
            }

        } else {

            now = (gint16)msg;
            length = (gint32)msg >> 16;

            if (length/60 > 10)
                g_snprintf(text, 32, "%02d:%02d [%02d:%02d]", now/60, now%60, length/60, length%60);
            else
                g_snprintf(text, 32, "%d:%02d [%d:%02d]", now/60, now%60, length/60, length%60);

            fraction = (gdouble)now / (gdouble)length;

            gtk_progress_bar_set_fraction(v->timer, fraction);
            gtk_progress_bar_set_text(v->timer, text);
        }
    }

    g_async_queue_unref(v->timer_queue);

    g_print("ui_thread_loop: done.\n");

    return NULL;
}

static void queue_selection_cb(GtkTreeSelection *selection, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;
    yaspot_glue_response_t *r;
    GtkTreeIter iter;
    GtkTreeModel *model;
    gint value;

    g_assert(selection);
    g_assert(v);

    model = gtk_tree_view_get_model(v->treeview);
    if (!gtk_tree_selection_get_selected(selection, NULL, &iter)) {
        g_print("queue_selection_cb: failed to get iter\n");
    }

    gtk_tree_model_get(model, &iter, PLAYLIST_COL_TRACK, &value, -1);

    if (value == v->current_playback_row) {
        return;
    }

    if (value > -1) {
        if ((r = yaspot_glue_current_queue(v->u->session))) {
            g_print("play song %d\n", value);
            yaspot_glue_play(r, value);
            yaspot_glue_response_unref(r);
            update_selection(v, TRUE);
        }
    }
}

static void state_changed_cb(gboolean playing, gboolean paused, gpointer data) {
    yaspot_view_t *v = (yaspot_view_t*)data;

    g_assert(v);

    g_print("queue: state changed\n");

    if (!playing)
        return;

    g_async_queue_push(v->timer_queue, GINT_TO_POINTER(UPDATE_PLAYPAUSE));

    if (!paused)
        g_async_queue_push(v->timer_queue, GINT_TO_POINTER(UPDATE_SELECTION));
}

void yaspot_show_queue(userdata_t *u) {
    yaspot_view_t *v;
    yaspot_glue_response_t *r;
    GtkListStore *model;
    GtkWidget *vbox;
    GtkWidget *status_hbox;
    HildonButton *button_prev;
    HildonButton *button_next;
    GtkWidget *image_prev;
    GtkWidget *image_next;
    GtkWidget *image_pp;

    v = view_new(u);
    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );
    v->treeview = yaspot_playlist_queue_new();
    gtk_container_add(GTK_CONTAINER(v->area), GTK_WIDGET(v->treeview));

    vbox = gtk_vbox_new(FALSE, 10);
    status_hbox = gtk_hbox_new(FALSE, 10);
    v->timer = GTK_PROGRESS_BAR( gtk_progress_bar_new() );

    button_prev = hildon_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                    HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    v->button_pp= hildon_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                    HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    button_next = hildon_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                    HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_widget_set_size_request(GTK_WIDGET(button_prev), 100, -1);
    gtk_widget_set_size_request(GTK_WIDGET(v->button_pp), 100, -1);
    gtk_widget_set_size_request(GTK_WIDGET(button_next), 100, -1);

    gtk_box_pack_start(GTK_BOX(status_hbox), GTK_WIDGET(button_prev), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(status_hbox), GTK_WIDGET(v->button_pp), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(status_hbox), GTK_WIDGET(button_next), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(status_hbox), GTK_WIDGET(v->timer), TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(v->area), TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), status_hbox, FALSE, FALSE, 0);

    gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(vbox));

    image_prev = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_BUTTON);
    image_next = gtk_image_new_from_stock(GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_BUTTON);
    if (yaspot_glue_playing(u->session))
        image_pp = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_BUTTON);
    else
        image_pp = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON);

    v->old_paused = !yaspot_glue_playing(u->session);

    hildon_button_set_image(button_prev, image_prev);
    hildon_button_set_image(button_next, image_next);
    hildon_button_set_image(v->button_pp, image_pp);

    /* FIXME are these images freed with window etc? */

    v->u = u;

    gtk_window_set_title(GTK_WINDOW(v->window), _("Playing Queue"));

    model = GTK_LIST_STORE(gtk_tree_view_get_model(v->treeview));

    if ((r = yaspot_glue_current_queue(u->session))) {
        yaspot_glue_response_to_queue_model(r, model);
        v->r = r;

        v->selection = gtk_tree_view_get_selection(v->treeview);
        gtk_tree_selection_set_mode(v->selection, GTK_SELECTION_BROWSE);

    } else {
        gtk_list_store_clear(model);
    }

    g_signal_connect(button_prev, "clicked", G_CALLBACK(msp_prev_cb), u);
    g_signal_connect(v->button_pp, "clicked", G_CALLBACK(msp_playpause_cb), u);
    g_signal_connect(button_next, "clicked", G_CALLBACK(msp_next_cb), u);

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(queue_destroy), v);

    gtk_widget_show_all(v->window);
    gtk_widget_show_all(GTK_WIDGET(v->timer));
    if (!v->timer_queue)
        v->timer_queue = g_async_queue_new();
    if (!v->ui_thread)
        v->ui_thread = g_thread_create(ui_thread_loop, v, TRUE, NULL);
    yaspot_glue_set_timer_cb(u->session, update_timer, v);

    if (v->r) {
        update_selection(v, FALSE);
        g_signal_connect(G_OBJECT(v->selection), "changed", G_CALLBACK(queue_selection_cb), v);
        yaspot_glue_add_state_cb(u->session, state_changed_cb, v);
    }
}

void yaspot_show_biography(userdata_t *u, yaspot_glue_response_t *r) {
    yaspot_view_t *v;
    GtkWidget *info_text;
    gchar *text = NULL;
    gchar *label = NULL;

    g_assert(u);
    g_assert(r);

    v = view_new(u);
    v->r = r;
    yaspot_glue_response_ref(v->r);

    v->window = hildon_stackable_window_new();
    v->area = HILDON_PANNABLE_AREA( hildon_pannable_area_new() );

    label = g_strdup_printf(_("%s's Biography"), yaspot_glue_response_get_name(r));
    gtk_window_set_title(GTK_WINDOW(v->window), label);

    text = yaspot_glue_artist_description(v->r, -1);
    if (text)
        info_text = gtk_label_new(text);
    else
        info_text = gtk_label_new("<no description>");
    gtk_label_set_line_wrap(GTK_LABEL(info_text), TRUE);
    gtk_widget_set_size_request(info_text, -1, -1); /* FIXME MAGIC NUMBER */

    hildon_pannable_area_add_with_viewport(v->area, GTK_WIDGET(info_text));
    gtk_container_add(GTK_CONTAINER(v->window), GTK_WIDGET(v->area));

    yaspot_app_menu_attach(v->u, HILDON_WINDOW(v->window));

    g_signal_connect(G_OBJECT(v->window), "delete_event",
                     G_CALLBACK(destroy), v);

    if (text)
        g_free(text);
    if (label)
        g_free(label);

    gtk_widget_show_all(v->window);
}

// vim: expandtab

