import QtQuick 1.0
import "videoinfoscripts.js" as Scripts
import "settings.js" as Settings
import "menu.js" as Menu

Rectangle {
    id: window

    property variant video
    property string playerUrl
    property string title
    property string author
    property string date
    property string thumbnail
    property string duration
    property string views : "0"
    property variant tags : []

    signal authorClicked(string feed, string username)
//    signal playVideo(variant video)
    signal goToVideo(variant video)
    signal search(string query, string order)

    function setVideo(videoObject) {

        toggleBusy(true);

        video = videoObject;
        title = video.title;
        duration = video.duration;

        var request = new XMLHttpRequest();
        request.onreadystatechange = function() {
            if (request.readyState == XMLHttpRequest.DONE) {
                var page = request.responseText.replace(/ /g,'');
                var tagsPart = page.split("content=\"")[2].split('"')[0];
                tags = tagsPart.split(",")
                var urlPart = page.split("'file':'")[1];
                playerUrl = "http://xhamster.com/flv2/" + urlPart.slice(0, urlPart.indexOf("\'"));
                var pos1 = urlPart.indexOf("'image':'") + 9;
                var pos2 = urlPart.indexOf(".jpg") + 4;
                thumbnail = urlPart.slice(pos1, pos2);
                var authorPart = urlPart.split("/user/")[1];
                author = authorPart.slice(0, authorPart.indexOf('"'));
                pos1 = urlPart.indexOf("Addedon:") + 23;
                date = urlPart.substr(pos1, 10);
                var viewsPart = urlPart.split("Views:</B></td>")[1];
                views = viewsPart.slice(6, viewsPart.indexOf("</"));

                toggleBusy(false);
            }
        }
        request.open("GET", video.link);
        request.send();
    }

    function toggleMenu() {
        Menu.toggleMenu();
    }

    color: "black"
    onStateChanged: Scripts.rotateDialogs()

    Loader {
        id: dialog

        anchors { top: window.bottom; topMargin: 10; horizontalCenter: window.horizontalCenter }
        z: 1
        onLoaded: {
            dialog.item.state = window.state;
            dialog.state = "show";
        }

        Timer {
            id: timer

            interval: 600
            onTriggered: dialog.source = ""
        }

        states: [
            State {
                name: "show"
                AnchorChanges { target: dialog; anchors { top: undefined; verticalCenter: window.verticalCenter } }
            }
        ]

        transitions: [
            Transition {
                AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
            }
        ]
    }

    Rectangle {
        id: dimmer

        anchors.fill: window
        color: window.color

        Loader {
            id: menuLoader

            z: 20
            width: dimmer.width / 2
            anchors { left: dimmer.left; bottom: dimmer.bottom; bottomMargin: 60 }

            onLoaded: menuLoader.item.addMenuItems([ { name: "Copy to clipboard" } ])

            Timer {
                id: menuTimer

                interval: 600
                onTriggered: menuLoader.source = ""
            }

            Connections {
                target: menuLoader.item

                onItemClicked: {
                    toggleMenu();
                    if (index == 1) {
                        Controller.copyToClipboard(playerUrl);
                    }
                }
            }
        }

        Rectangle {
            id: frame

            width: 250
            height: 190
            anchors { left: dimmer.left; leftMargin: 10; top: dimmer.top; topMargin: 60 }
            color: "black"
            border.width: 3
            border.color: frameMouseArea.pressed ? activeColorLow : "white"

            Image {
                id: thumb

                anchors { fill: frame; margins: 3 }
                source: thumbnail
                sourceSize.width: thumb.width
                sourceSize.height: thumb.height
                smooth: true

                Rectangle {
                    id: durationLabel

                    width: 100
                    height: 44
                    anchors { bottom: thumb.bottom; right: thumb.right }
                    color: "black"
                    opacity: 0.5

                }

                Text {
                    id: durationText

                    anchors.fill: durationLabel
                    text: duration
                    color: "white"
                    font.pixelSize: standardFontSize
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            MouseArea {
                id: frameMouseArea

                anchors.fill: frame
                onClicked: Controller.playVideo(playerUrl) /*playVideo([{ title: title,
                                      thumbnail: video.thumbnail,
                                      duration: duration,
                                      playerUrl: playerUrl }])*/
            }

            PushButton {
                id: videoButton

                width: 250
                anchors { left: frame.left; top: frame.bottom; topMargin: 10 }
                icon: "ui-images/videodownloadicon.png"
                iconWidth: 65
                iconHeight: 65
                onButtonClicked: addDownload({ title: title,
                                    thumbnail: video.thumbnail,
                                    playerUrl: playerUrl })
            }
        }

        Item {
            id: tabItem

            anchors { fill: dimmer; leftMargin: 270; rightMargin: 10; topMargin: 60 }

            Row {
                id: tabRow

                Rectangle {
                    id: infoTab

                    width: tabItem.width / 2
                    height: 40
                    color: "black"

                    BorderImage {
                        anchors.fill: infoTab
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 0
                    }

                    Text {
                        anchors.fill: infoTab
                        font.pixelSize: standardFontSize
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 0 ? "white" : "grey"
                        text: qsTr("Info")
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: infoTab.bottom; left: infoTab.left; right: infoTab.right }
                        color: activeColor
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 0)
                    }

//                    MouseArea {
//                        id: infoMouseArea

//                        anchors.fill: infoTab
//                        onClicked: tabView.currentIndex = 0
//                    }
                }

                Rectangle {
                    id: relatedTab

                    width: tabItem.width / 2
                    height: 40
                    color: "black"

                    BorderImage {
                        anchors.fill: parent
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 1
                    }

                    Text {
                        anchors.fill: relatedTab
                        font.pixelSize: standardFontSize
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 1 ? "white" : "grey"
                        text: qsTr("Related")
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: relatedTab.bottom; left: relatedTab.left; right: relatedTab.right }
                        color: activeColor
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 1)
                    }

//                    MouseArea {
//                        id: relatedMouseArea

//                        anchors.fill: relatedTab
//                        onClicked: tabView.currentIndex = 1
//                    }
                }
            }

            ListView {
                id: tabView

                anchors { left: tabItem.left; right: tabItem.right; top: tabRow.bottom; bottom: tabItem.bottom }
                orientation: ListView.Horizontal
                highlightMoveDuration: 200
                highlightRangeMode: ListView.StrictlyEnforceRange
                snapMode: ListView.SnapOneItem
                flickDeceleration: 500
                boundsBehavior: Flickable.StopAtBounds
                model: tabModel
                clip: true

                onCurrentIndexChanged: {
                    if ((tabView.currentIndex == 1) && (!relatedView.loaded) && (relatedView.count == 0)) {
                        relatedView.loadVideos();
                    }
                }
            }
        }

        VisualItemModel {
            id: tabModel

            Flickable {
                id: scrollArea

                width: 520
                height: 320
                clip: true
                contentWidth: childrenRect.width
                contentHeight: textColumn.height
                flickableDirection: Flickable.VerticalFlick
                boundsBehavior: Flickable.DragOverBounds
                interactive: visibleArea.heightRatio < 1
                visible: tabView.currentIndex == 0

                Column {
                    id: textColumn

                    spacing: 10
                    width: 520
                    height: childrenRect.height

                    Text {
                        id: titleText

                        width: textColumn.width
                        text: title
                        color: "white"
                        font.pixelSize: standardFontSize
                        wrapMode: TextEdit.WordWrap
                    }

                    Text {
                        id: authorText

                        width: textColumn.width
                        color: "white"
                        font.pixelSize: smallFontSize
                        textFormat: Text.StyledText
                        wrapMode: TextEdit.WordWrap
                        text: authorMouseArea.pressed ? qsTr("By ") + "<font color='"
                                                        + activeColor + "'>" + author
                                                        + "</font>" + qsTr(" on ") + date
                                                      : qsTr("By ") + "<font color='"
                                                        + activeColorLow + "'>" + author
                                                        + "</font>" + qsTr(" on ") + date

                        MouseArea {
                            id: authorMouseArea

                            anchors.fill: authorText
                            onClicked: {
                                var feed = "http://xhamster.com/user/video/" + author + "/new-1.html";
                                authorClicked(feed, author);
                            }
                        }
                    }

                    Text {
                        font.pixelSize: smallFontSize
                        color: "white"
                        text: qsTr("Views ") + "<font color=grey>" + views + "</font>"
                    }

                    Text {
                        font.pixelSize: smallFontSize
                        color: "white"
                        text: qsTr("Tags")
                    }

                    Flow {
                        spacing: 10
                        width: parent.width

                        Repeater {
                            model: tags

                            Text {
                                font.pixelSize: smallFontSize
                                color: children[0].pressed ? activeColor : activeColorLow
                                text: modelData

                                MouseArea {
                                    anchors.fill: parent
                                    onClicked: search(parent.text, Settings.getSetting("searchOrder").toLowerCase())
                                }

                                Text {
                                    anchors.left: parent.right
                                    font.pixelSize: smallFontSize
                                    color: "grey"
                                    text: ","
                                }
                            }
                        }
                    }

                }
            }

            ListView {
                id: relatedView

                property bool loaded : false // True if related videos have been loaded
                property bool loading : false // True if related videos are loading.

                function loadVideos() {
                    relatedView.loading = true;

                    var doc = new XMLHttpRequest();
                    doc.onreadystatechange = function() {
                        if (doc.readyState == XMLHttpRequest.DONE) {
                            var results = doc.responseText.split("Related Videos")[1].split('<a href="');
                            var pattern = /movies/;
                            for (var i = 2; i < results.length; i++) {
                                if (pattern.test(results[i])) {
                                    var r = results[i];
                                    relatedModel.append({ link: "http://xhamster.com" + r.split('"')[0],
                                                       title: r.slice(r.indexOf("<u>") + 3, r.indexOf("</u>")),
                                                       thumbnail: r.slice(r.indexOf("src=") + 5, r.indexOf("' width")),
                                                       duration: getDuration(r.slice(r.indexOf("Runtime") + 9, r.indexOf("<BR>"))) });
                                }
                            }
                            relatedView.loading = false;
                            relatedView.loaded = true;
                        }
                    }
                    doc.open("GET", video.link);
                    doc.send();
                }

                function getDuration(durationString) {
                    /* Converts string to MM:SS format. */
                    var a = durationString.split("m")
                    var minutes = a[0];
                    var seconds = a[1].slice(0, -1);
                    if (seconds < 10) {
                        seconds = "0" + seconds;
                    }
                    var duration = minutes + ":" + seconds;
                    return duration;
                }

                width: 520
                height: 320
                boundsBehavior: Flickable.DragOverBounds
                highlightMoveDuration: 500
                preferredHighlightBegin: 0
                preferredHighlightEnd: 100
                highlightRangeMode: ListView.StrictlyEnforceRange
                cacheBuffer: 2500
                interactive: visibleArea.heightRatio < 1
                clip: true
                visible: tabView.currentIndex == 1

                model: ListModel {
                    id: relatedModel
                }

                delegate: XXXListDelegate {
                    id: delegate

                    Connections {
                        onDelegateClicked: goToVideo(relatedModel.get(index))
                    }
                }

                ScrollBar {}

                BusyDialog {
                    anchors.centerIn: relatedView
                    opacity: (tabView.currentIndex == 1 && relatedView.loading)
                }
            }
        }

        MouseArea {
            id: mouseArea

            anchors.fill: dimmer
            enabled: false
            onClicked: Scripts.closeDialogs()
        }

        states: State {
            name: "dim"
            PropertyChanges { target: dimmer; opacity: 0.3 }
        }

        transitions: Transition {
            PropertyAnimation { properties: "opacity"; duration: 500 }
        }
    }

    states: State {
        name: "portrait"
        when: window.height > window.width
        PropertyChanges { target: tabItem; anchors { leftMargin: 10; rightMargin: 10; topMargin: 260 } }
        PropertyChanges { target: scrollArea; width: 460; height: 440 }
        PropertyChanges { target: textColumn; width: 460 }
        PropertyChanges { target: relatedView; width: 460; height: 440 }
        AnchorChanges { target: videoButton; anchors { left: frame.right; top: frame.top } }
        PropertyChanges { target: videoButton; anchors.leftMargin: 10; anchors.topMargin: 0; width: 200 }
    }
}

