#include "wpnew.h"
#include "ui_wpnew.h"
#include "wallpaeper.h"
#include "ui_wallpaeper.h"

#include <QFileDialog>
#include <QMaemo5InformationBox>
#include <QTextStream>

wpNew::wpNew(QWidget* parent) :
    QMainWindow(parent),
    ui(new Ui::wpNew)
{
    ui->setupUi(this);

    // Setup a few QLists with 4 items each
    for(i=0; i<4; i++) {
        // Objects
        copyToBtn.append(new QPushButton);
        copyToBtnExists.append(new bool);
        copyImageBtn.append(new QPushButton);
        imgPath.append(new QString);
        imgPrevName.append(new QLabel);

        // UI elements
        vertLayout.append(new QVBoxLayout);
        imgBtnPrev.append(new QPushButton);
    }

    setupFIMenuBar();

    vertLayout[0] = ui->verticalLayout_1;
    vertLayout[1] = ui->verticalLayout_2;
    vertLayout[2] = ui->verticalLayout_3;
    vertLayout[3] = ui->verticalLayout_4;
    for(i=0; i<4; i++)
        vertLayout[i]->setAlignment(Qt::AlignTop);

    imgBtnPrev[0] = ui->imgBtnPrev1;
    imgBtnPrev[1] = ui->imgBtnPrev2;
    imgBtnPrev[2] = ui->imgBtnPrev3;
    imgBtnPrev[3] = ui->imgBtnPrev4;

    imgPrevName[0] = ui->imgPrevName1;
    imgPrevName[1] = ui->imgPrevName2;
    imgPrevName[2] = ui->imgPrevName3;
    imgPrevName[3] = ui->imgPrevName4;
    for(i=0; i<4; i++)
        imgPrevName[i]->setStyleSheet("font-size: 14pt");

    ui->visibChkbox->setChecked(false);

    copyImageExists = false;
    for(i=0; i<4; i++)
        *copyToBtnExists[i] = false;

    fileWithPath = new QString;
    owDialogExists = false;
    svDialogIsMade = false;
    saveFileName = "";

    curCopyToBtn = 8;
    // This default dir changes after selecting an image
    lastUsedDir = "/home/user/MyDocs/.images/";

    QRegExp nameRegExp("^[a-zA-Z0-9_-\\s]*$");
    // ^                From the beginning of the string
    // [a-zA-Z0-9_-\s]  Only alphanumeric, _ and - and whitespace
    // *                At least 0 characters
    // $                Until the end of the string
    ui->nameEdit->setValidator(new QRegExpValidator(nameRegExp, ui->nameEdit));

    // Show default button at startup
    for(i=0; i<4; i++) {
        imgBtnPrev[i]->setText(tr("Image ") + QString::number(i+1));
        imgBtnPrev[i]->setStyleSheet("color: white; border-image: url(:/images/select.png);");
    }

    connect(imgBtnPrev[0], SIGNAL(clicked()),
            this, SLOT(clickedImgBtnPrev0()));
    connect(imgBtnPrev[1], SIGNAL(clicked()),
            this, SLOT(clickedImgBtnPrev1()));
    connect(imgBtnPrev[2], SIGNAL(clicked()),
            this, SLOT(clickedImgBtnPrev2()));
    connect(imgBtnPrev[3], SIGNAL(clicked()),
            this, SLOT(clickedImgBtnPrev3()));

    connect(copyToBtn[0], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn0()));
    connect(copyToBtn[1], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn1()));
    connect(copyToBtn[2], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn2()));
    connect(copyToBtn[3], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn3()));

    connect(ui->nameEdit, SIGNAL(textChanged(QString)),
            this, SLOT(updateSaveBtn()));

    connect(ui->saveBtn, SIGNAL(clicked()),
            this, SLOT(createSvDialog()));
}





wpNew::~wpNew() {
    delete ui;
}





//********* START wpNew public functions *******************
void wpNew::setupFIMenuBar() {
    fiMenuActionReset = new QAction(tr("Reset screen"), this);

    fiMenu = menuBar()->addMenu("");
    fiMenu->addAction(fiMenuActionReset);

    connect(fiMenuActionReset, SIGNAL(triggered()),
            this, SLOT(resetScreen()));
}





void wpNew::openImage(int imgIndex) {
    *fileWithPath = QFileDialog::getOpenFileName(this,
                                                 "Open image",
                                                 lastUsedDir.absolutePath(),
                                                 "Images (*.jpeg *.jpg *.gif *.png *.bmp *.tiff)");

    if(!fileWithPath->isEmpty()) { // If the filedialog is NOT cancelled (path NOT empty)
        // Save last used directory for next button press
        lastUsedDir.setPath(QFileInfo(*fileWithPath).absolutePath());

        *imgPath[imgIndex] = *fileWithPath;
        updateImgBtnPrev(imgIndex);
        imgPrevName[imgIndex]->setText(QFileInfo(*fileWithPath).fileName());
        setupCopyToBtn(imgIndex);
        updateSaveBtn();
    }
}





void wpNew::setupCopyToBtn(int imgBtnPrevNum) {
    // Setup the current copyToBtn
    // If hidden after resetScreen(), show it again
    if(*copyToBtnExists[imgBtnPrevNum] == true && copyToBtn[imgBtnPrevNum]->isVisible() == false)
        copyToBtn[imgBtnPrevNum]->show();
    else if(*copyToBtnExists[imgBtnPrevNum] == false) {
        copyToBtn[imgBtnPrevNum]->setParent(this);
        copyToBtn[imgBtnPrevNum]->setMaximumSize(190, 50);
        copyToBtn[imgBtnPrevNum]->setText(tr("Copy to..."));
        copyToBtn[imgBtnPrevNum]->setVisible(true);
        vertLayout[imgBtnPrevNum]->addWidget(copyToBtn[imgBtnPrevNum], 0, Qt::AlignHCenter);
    }
    *copyToBtnExists[imgBtnPrevNum] = true;
}





void wpNew::createCopyImage(int copyToBtnNum) {
    // Don't create a new copyImage (dialog) if it exists already
    // Just enable buttons again
    if(copyImageExists == true) {
        for(i=0; i<4; i++) {
            copyImageBtn[i]->setEnabled(true);
            copyImageBtn[i]->setChecked(false);
        }
    }
    else {
        copyImage = new copyToDialog(this);

        copyImageBtn[0] = copyImage->ui->copyToDialogBtn1;
        copyImageBtn[1] = copyImage->ui->copyToDialogBtn2;
        copyImageBtn[2] = copyImage->ui->copyToDialogBtn3;
        copyImageBtn[3] = copyImage->ui->copyToDialogBtn4;

        // OK button closes dialog and copies images
        connect(copyImage->ui->copyToDialogBtnBox, SIGNAL(accepted()),
                wpNew::copyImage, SLOT(close()));
        connect(copyImage->ui->copyToDialogBtnBox, SIGNAL(accepted()),
                this, SLOT(copyToOthers()));
        // Cancel button only closes the dialog
        connect(copyImage->ui->copyToDialogBtnBox, SIGNAL(rejected()),
                wpNew::copyImage, SLOT(close()));

        copyImage->adjustSize();
        copyImageExists = true;
    }

    // Disable the button that is being copied from
    for(i=0; i<4; i++) {
        if(copyToBtnNum == i) { // If user wants to copy image #, disable button #
            curCopyToBtn = i;
            copyImageBtn[i]->setEnabled(false);
            copyImageBtn[i]->setChecked(false);
        }
        else if(copyToBtnNum < 0 || copyToBtnNum > 3) { // If current button value is not valid
            // Checks 4 times, so disables all 4 buttons
            copyImageBtn[i]->setEnabled(false);
            copyImageBtn[i]->setChecked(false);
        }
    }

    copyImage->show();
}





void wpNew::updateImgBtnPrev(int pathNum) {
    // Show default button when image doesn't exist
    if(QFile(*imgPath[pathNum]).exists() == false) {
        imgBtnPrev[pathNum]->setText(tr("Image ") + QString::number(pathNum+1));
        imgBtnPrev[pathNum]->setStyleSheet("color: white; border-image: url(:/images/select.png);");
        *imgPath[pathNum] = "";
    }
    else {
        // Update imgBtnPrev[#] according to *imgPath[pathNum]
        imgBtnPrev[pathNum]->setStyleSheet("border-image: url(" + *imgPath[pathNum] + ")");
        imgBtnPrev[pathNum]->setText("");
    }
}





void wpNew::createOwDialog() {
    // Don't create the dialog if it exists already
    if(owDialogExists == false) {
        owDialog = new overwriteDialog(this);

        connect(owDialog->ui->owBtnBox, SIGNAL(accepted()),
                this, SLOT(confirmOverwrite()));
        connect(owDialog->ui->owBtnBox, SIGNAL(rejected()),
                this->owDialog, SLOT(close()));
    }

    // Update owQuestion (label)
    owDialog->ui->owQuestion->setText(saveFileName + tr(" already exists, overwrite?"));
    owDialog->ui->owQuestion->adjustSize();
    owDialog->adjustSize();
    owDialog->show();
}





void wpNew::checkDesktopFileSaved() {
    if(saveFile.exists() == true)
        QMaemo5InformationBox::information(this,
                                           "<br><b>"
                                           + tr("Image set saved!")
                                           + "</b><br><br>"
                                           + saveFile.fileName()
                                           + "<br>",
                                           QMaemo5InformationBox::NoTimeout);
    if(saveFile.exists() == false)
        QMaemo5InformationBox::information(this,
                                           "<br><b>"
                                           + tr("Image set NOT saved...")
                                           + "</b><br><br>"
                                           + tr("Please try again")
                                           + "<br>",
                                           QMaemo5InformationBox::NoTimeout);
}
//********* END wpNew public functions *******************





//********* START wpNew public slots *******************
void wpNew::clickedImgBtnPrev0() { openImage(0); }
void wpNew::clickedImgBtnPrev1() { openImage(1); }
void wpNew::clickedImgBtnPrev2() { openImage(2); }
void wpNew::clickedImgBtnPrev3() { openImage(3); }





void wpNew::updateSaveBtn() {
    if(ui->nameEdit->text().isEmpty()
        || *imgPath[0] == ""
        || *imgPath[1] == ""
        || *imgPath[2] == ""
        || *imgPath[3] == "") // If one or more objects are empty
    {
        // saveBtn disabled, set default button text
        ui->saveBtn->setEnabled(false);
        ui->saveBtn->setText(tr("Please set name and images first"));
    } else { // If everything is filled in
        // Enable and update saveBtn
        ui->saveBtn->setEnabled(true);
        ui->saveBtn->setText(tr("Save ") + ".desktop" + tr(" file as..."));
    }
}





void wpNew::createSvDialog() {
    if(svDialogIsMade == false) {
        svDialog = new saveDialog(this);

        connect(svDialog->ui->saveBtnBox, SIGNAL(accepted()),
                this, SLOT(setSaveFileName()));
        connect(svDialog->ui->saveBtnBox, SIGNAL(rejected()),
                svDialog, SLOT(close()));

        svDialog->ui->saveLabel->adjustSize();
        svDialogIsMade = true;
    }

    if(saveFileName == "") // If saveFileName is empty (default), set it on name given in nameEdit (lowercase)
        saveFileName = ui->nameEdit->text().toLower();
    else { // If saveFileName is already set:
        if(saveFileName.startsWith("."))
            saveFileName.remove(0, 1); // Remove preceding "."
        if(saveFileName.endsWith(".desktop"))
            saveFileName.chop(8); // Remove appended ".desktop"
    }

    // Put saveFileName in saveFileNameEdit (svDialog)
    svDialog->ui->saveFileNameEdit->setText(saveFileName);

    svDialog->adjustSize();
    svDialog->show();
}





void wpNew::setSaveFileName() {
    svDialog->close();

    // Retrieve name from svDialog (lowercase)
    saveFileName = svDialog->ui->saveFileNameEdit->text().toLower();

    // Check if "Invisible file" is checked, change filename accordingly
    if(ui->visibChkbox->isChecked())
        saveFileName = "." + saveFileName;
    // Add extenstion to saveFileName
    saveFileName += ".desktop";

    // Always save to /home/user/MyDocs/.images/
    saveFile.setFileName("/home/user/MyDocs/.images/" + saveFileName);

    if(saveFile.exists()) {
        // File already exists, overwrite?
        createOwDialog();
    } else {
        // File doesn't exist, save the .desktop file
        saveDesktopFile();
        // After saving, check if the file is saved
        checkDesktopFileSaved();
    }
}





void wpNew::saveDesktopFile() {
    saveFile.open(QFile::WriteOnly | QFile::Truncate);
        QTextStream fileContent(&saveFile);
        fileContent << "[Desktop Entry]\n"
                << "Type=Background Image\nName=" << ui->nameEdit->text() << "\n"
                << "Hidden=true\n\n"
                << "X-File1=" << *imgPath[0] << "\n"
                << "X-File2=" << *imgPath[1] << "\n"
                << "X-File3=" << *imgPath[2] << "\n"
                << "X-File4=" << *imgPath[3] << "\n\n"
                << "X-Order=01";
    saveFile.close();
}





void wpNew::clickedCopyToBtn0() { createCopyImage(0); }
void wpNew::clickedCopyToBtn1() { createCopyImage(1); }
void wpNew::clickedCopyToBtn2() { createCopyImage(2); }
void wpNew::clickedCopyToBtn3() { createCopyImage(3); }





void wpNew::confirmOverwrite() {
    owDialog->close();

    // Save the .desktop file
    saveDesktopFile();
    // After saving, check if the file is saved
    checkDesktopFileSaved();
}





void wpNew::copyToOthers() {
    // Current source image number is defined in curCopyToBtn
    srcPath = new QString;

    // Store source
    for(i=0; i<4; i++) {
        if(curCopyToBtn == i)
            *srcPath = *imgPath[i];
    }

    // Copy from source path to imgPath[#], based on checked buttons
    // then updates imgBtnPrev[#] and copyToBtn[#]
    for(j=0; j<4; j++) {
        if(copyImageBtn[j]->isChecked() == true) {
            // Only update if the selected image exists
            if(QFile(*srcPath).exists() == true) {
                *imgPath[j] = *srcPath;
                updateImgBtnPrev(j);
                imgPrevName[j]->setText(QFileInfo(*imgPath[j]).fileName());
                setupCopyToBtn(j);
            }
            else
                QMaemo5InformationBox::information(this,
                                                   tr("That image doesn't exist (anymore)."),
                                                   QMaemo5InformationBox::DefaultTimeout);
        }
    }
    // updateSaveBtn afterwards
    updateSaveBtn();
}





void wpNew::resetScreen() {
    // nameEdit
    ui->nameEdit->setText("");
    ui->visibChkbox->setChecked(false);
    // imgPath, imgBtnPrev, copyToBtn
    for(i=0; i<4; i++) {
        *imgPath[i] = "";
        imgBtnPrev[i]->setText(tr("Image ") + QString::number(i+1));
        imgBtnPrev[i]->setStyleSheet("color: white; border-image: url(:/images/select.png);");
        imgPrevName[i]->setText("");
        copyToBtn[i]->setVisible(false);
    }
    // saveBtn
    updateSaveBtn();
}
//********* END wpNew public slots *******************
