#include "wpmanager.h"
#include "ui_wpmanager.h"

#include <QMaemo5InformationBox>
#include <QTextStream>

wpManager::wpManager(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::wpManager)
{
    ui->setupUi(this);

    // Set the Images folder as the fixed path, filter on .desktop files
    imageSetDir.setPath("/home/user/MyDocs/.images/");
    nameFilters << "*.desktop";
    delDialogIsMade = false;

    connect(ui->manDelBtn, SIGNAL(clicked()),
            this, SLOT(confirmDelSet()));
    connect(ui->manPrevBtn, SIGNAL(clicked()),
            this, SLOT(updatePrevs()));
    connect(ui->fileList, SIGNAL(itemSelectionChanged()),
            this, SLOT(itemChanged()));
    connect(ui->manEditBtn, SIGNAL(clicked()),
            this, SLOT(editCurrent()));

    for(i=0; i<4; i++)
        manPrev.append(new QLabel);
    manPrev[0] = ui->manPrev1;
    manPrev[1] = ui->manPrev2;
    manPrev[2] = ui->manPrev3;
    manPrev[3] = ui->manPrev4;

    updateList();
}

wpManager::~wpManager()
{
    delete ui;
}

void wpManager::updateList() {
    // Empty the lists before updating
    imageSetList.clear();
    ui->fileList->clear();

    // All the entries in Images folder (+filter) are stored in imageSetList (QStringList)
    // Filter on files, including hidden files
    imageSetList = imageSetDir.entryList(nameFilters, QDir::Files | QDir::Hidden);

    // Fill ui->fileList (update)
    for(i=0; i<imageSetList.count(); i++)
        itemList.append(new QListWidgetItem(imageSetList[i], ui->fileList));
    ui->fileList->sortItems(Qt::AscendingOrder);

    // Clear the previews
    for(i=0; i<4; i++)
        manPrev[i]->setPixmap(QPixmap(""));
}

void wpManager::updatePrevs() {
    // Open (read-only) the currently selected .desktop file
    imageSetPrev.setFileName(imageSetDir.absolutePath() + "/" + ui->fileList->currentItem()->text());
    imageSetPrev.open(QFile::ReadOnly);

    QTextStream readPrevFile(&imageSetPrev);

    // Read until the end of the file contents
    while (!readPrevFile.atEnd()) {
        lineInFile = readPrevFile.readLine();

        // If a line starts with "X-File#", update previews with that
        for(i=0; i<4; i++) {
            if(lineInFile.startsWith("X-File" + QString::number(i+1))) {
                lineInFile.remove(0, 8); // Remove "X-File#="
                manPrev[i]->setPixmap(QPixmap(lineInFile));
            }
        }
    }

    imageSetPrev.close();
}

void wpManager::confirmDelSet() {
    if(ui->fileList->currentRow() == -1)
        QMaemo5InformationBox::information(this,
                                           "Please select an image set from the list first",
                                           QMaemo5InformationBox::DefaultTimeout);
    else if(ui->fileList->currentRow() >= 0) {
        // Store the filename in QString delFileName
        delFileName = ui->fileList->currentItem()->text();
        // Store filename with path in QFile delFileWithPath
        delFileWithPath.setFileName(imageSetDir.absolutePath() + "/" + delFileName);

        // Don't create another QDialog if it has been made already, just update it
        if(delDialogIsMade == true)
            delText->setText(tr("Delete this file?") + "<br>" + tr("NOTE: images are NOT deleted.") + "<br><br>" + delFileName);
        else if(delDialogIsMade == false) {
            delDialog = new QDialog;
            delDialog->setWindowTitle(tr("Delete selected file?"));

            delText = new QLabel;
            delText->setText(tr("Delete this file?") + "<br>" + tr("NOTE: images are NOT deleted.") + "<br><br>" + delFileName);
            delText->adjustSize();

            delYes = new QPushButton;
            delYes->setText(tr("Yes"));
            delYes->setMinimumSize(190, 70);
            delYes->setMaximumSize(190, 70);
            connect(delYes, SIGNAL(clicked()),
                    this, SLOT(delSet()));

            delNo = new QPushButton;
            delNo->setText(tr("No"));
            delNo->setMinimumSize(190, 70);
            delNo->setMaximumSize(190, 70);
            connect(delNo, SIGNAL(clicked()),
                    wpManager::delDialog, SLOT(close()));

            delBtnsV = new QVBoxLayout;
            delBtnsV->addWidget(delYes);
            delBtnsV->addWidget(delNo);

            delLayoutH = new QHBoxLayout;
            delLayoutH->addWidget(delText);
            delLayoutH->addLayout(delBtnsV);
            delLayoutH->setAlignment(Qt::AlignVCenter);

            delDialog->setLayout(delLayoutH);
            delDialogIsMade = true;
        }

        delDialog->show();
    }
    else
        QMaemo5InformationBox::information(this,
                                           "Unknown list item, please try again.",
                                           QMaemo5InformationBox::DefaultTimeout);
}

void wpManager::delSet() {
    // If the file is removed, show confirmation
    if(delFileWithPath.remove()) {
        delDialog->close();
        QMaemo5InformationBox::information(this,
                                           "<br>" + delFileWithPath.fileName() + "<br>"
                                           + "<b>" + tr("deleted!") + "</b><br>",
                                           QMaemo5InformationBox::NoTimeout);
        updateList();
    }
    // Otherwise, the file doesn't exist, or needs permission
    else
        QMaemo5InformationBox::information(this,
                                           "<br>" + tr("File is ") + "<b>" + tr("NOT") + "</b>" + tr("deleted.") + "<br>"
                                           + tr("Perhaps it is already deleted?") + "<br>"
                                           + tr("Maybe you don't have permission to open or edit the file?"),
                                           QMaemo5InformationBox::NoTimeout);
}

void wpManager::itemChanged() {
    // Clear previews to avoid confusion
    for(i=0; i<4; i++)
        manPrev[i]->setPixmap(QPixmap(""));
    if(ui->fileList->currentItem()->isSelected() == false)
        ui->fileList->currentItem()->setSelected(true);
}

void wpManager::editCurrent() {
    if(ui->fileList->currentRow() == -1)
        QMaemo5InformationBox::information(this,
                                           "Please select an image set from the list first",
                                           QMaemo5InformationBox::DefaultTimeout);
    else if(ui->fileList->currentRow() >= 0) {
        // Create a new fourImages window for editing
        editWin = new fourImages(this);
        editWin->setAttribute(Qt::WA_Maemo5StackedWindow);
        editWin->setAttribute(Qt::WA_DeleteOnClose);
        connect(editWin, SIGNAL(destroyed()),
                this, SLOT(updateListAfterEdit()));

        // Open (read-only) the currently selected .desktop file
        editFile.setFileName(imageSetDir.absolutePath() + "/" + ui->fileList->currentItem()->text());
        editFile.open(QFile::ReadOnly);

        QTextStream readEditFile(&editFile);

        // Read until the end of the file contents
        while (!readEditFile.atEnd()) {
            lineInFile = readEditFile.readLine();

            // If a line starts with "Name=", update nameEdit (in editWin)
            if(lineInFile.startsWith("Name=")) {
                lineInFile.remove(0, 5); // Remove "Name="
                editWin->ui->nameEdit->setText(lineInFile);
            }

            // If a line starts with "X-File#", update imgPath[#] (in editWin)
            for(i=0; i<4; i++) {
                if(lineInFile.startsWith("X-File" + QString::number(i+1))) {
                    lineInFile.remove(0, 8); // Remove "X-File#="
                    *editWin->imgPath[i] = lineInFile;
                }
            }
        }

        editFile.close();

        // Put the "Invisible file" checkbox in the right state
        if(ui->fileList->currentItem()->text().startsWith("."))
            editWin->ui->visibChkbox->setChecked(true);
        else
            editWin->ui->visibChkbox->setChecked(false);

        // Update previews, buttons, copytobuttons, savebutton
        for(i=0; i<4; i++) {
            editWin->updateImg(i);
            editWin->setupCopyToBtn(i);
        }
        editWin->updateSaveBtn();

        // Before opening, save current item (position)
        curItemBeforeEdit = ui->fileList->currentRow();
        // And show it, finally
        editWin->show();
    }
    else
        QMaemo5InformationBox::information(this,
                                           "Unknown list item, please try again.",
                                           QMaemo5InformationBox::DefaultTimeout);
}

void wpManager::updateListAfterEdit() {
    updateList();
    // Reselect the item that was selected before editing
    ui->fileList->setCurrentRow(curItemBeforeEdit);
}
