/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/*
 * Changelog: See debian/changelog
 *
 */

/* Includes */
#include "hildon.h"

#include <gtk/gtk.h>
#define GTK_ENABLE_BROKEN
#include <gtk/gtktext.h> // BROKEN

#include <libosso.h>


#include <signal.h>
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "save_buffer.h"
#include "gpl.h"

#include "profile.h"
#include "profiles_gui.h"
//#include "profiles_backend.h"
#include "autoconnect_gui.h"
#include "profiles_launcher.h"
#include "dbus_listener.h"

/* widget for displaying infos */
GtkWidget *text=NULL; // TODO: static

/* program and main window */
static HildonProgram *program;
static HildonWindow *window;


static GtkWidget *connect, *disconnect;


#define DISCONNECTED 1
#define DISCONNECTING 2
#define CONNECTING 3
#define CONNECTED 4

/** status of the client, see above */
static int status=DISCONNECTED;

static void text_append(gchar *str) {

	gtk_text_insert(GTK_TEXT(text),
			NULL,
			NULL,
			NULL,
			str,
			strlen(str));
}

void show_about_cb(GtkBin *action, gpointer data) {

  GtkWindow *win=(GtkWindow *)data;

  const gchar *authors[3] = { "Michael Arndt <scriptkiller@gmx.de>", 0 };

  //const gchar *documenters[] = { 0 };
  gtk_show_about_dialog(win,
			"name", "vpnc GUI",
			"version", VERSION,
			"copyright", "Copyright(C) 2007-2008 Michael Arndt",
			//"comments", "A vpn-client frontend.",
			"authors", authors,
			//"documenters", documenters,
			//"translator-credits", 0,
			"website", "http://scriptkiller.de/",
			"website-label", "Home page",
			"license", GPL,
			"logo-icon-name", "vpnc-gui",
			NULL);
}


/* callback for connect button */
void button_connect_cb() {
	struct profile *p=profiles_gui_get_selected_profile();
	if(p!=NULL) {
		profiles_launcher_connect(p);
	}
	else {
		GtkWidget *window=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No Profile selected!");
		gtk_dialog_run (GTK_DIALOG (window));
		gtk_widget_destroy(window);
	}
}


/* callback for disconnect button */
void button_disconnect_cb() {
	struct profile *p=profiles_gui_get_selected_profile();
	if(p!=NULL) {
		profiles_launcher_disconnect(p);
	}
	else {
		GtkWidget *window=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No Profile selected!");
		gtk_dialog_run (GTK_DIALOG (window));
		gtk_widget_destroy(window);
	}
}

/**
 * Called when a connection has been established and
 * a profile should be launched.
 * @param profile_name name of the profile
 */
void autoconnect_cb(const gchar *profile_name) {
	printf("request for profile: %s\n", profile_name);
	
	/* do this only if we are really disconnected */
	if(status == DISCONNECTED) {
		/* select profile in gui and
		 * just simulate connect-button-press */
		profiles_gui_select_profile(profile_name);
		button_connect_cb();
	}
}

/**
 * Show an "enter password" dialog
 * @return NULL if user canceled the dialog or g_alloced string
 *         containing the password otherwise
 */
gchar *read_password_dialog() {

	GtkWidget *window;
	window=gtk_dialog_new_with_buttons("Enter Password",
			NULL,
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
			"Cancel",
			GTK_RESPONSE_CANCEL,
			"OK",
			GTK_RESPONSE_OK,
			NULL);

	GtkWidget *entry=gtk_entry_new();
	gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);

	GtkWidget *table=gtk_table_new(1, 2, FALSE);
	gtk_table_attach_defaults(GTK_TABLE(table),
			gtk_label_new("Password"),
			0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table),
			entry,
			1, 2, 0, 1);
	
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(window)->vbox), table, TRUE, TRUE, 0);
	gtk_widget_show_all(window);

	int retval=gtk_dialog_run(GTK_DIALOG(window));
	gchar *pw=g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
	
	gtk_widget_destroy(window);
	
	if(retval == GTK_RESPONSE_OK) {
		return pw; 
	}

	g_free(pw);
	return NULL;

}

void set_status(int s) {
	status=s;

	/* allow connect only when disconnected */
	if(status==DISCONNECTED) {
		profiles_gui_set_profile_chooser_active(TRUE);
		gtk_widget_set_sensitive(connect, TRUE);
		gtk_widget_set_sensitive(disconnect, FALSE);
	}
	/* allow disconnect always (when not already disconnected) */
	else if(status!=DISCONNECTED) {
		profiles_gui_set_profile_chooser_active(FALSE);
		gtk_widget_set_sensitive(connect, FALSE);
		gtk_widget_set_sensitive(disconnect, TRUE);
	}

}

/**
 * Called when launcher event occured.
 * @param p the involved profile
 * @param event the event
 * @param data pointer to more data
 */
void launcher_event_cb(struct profile *p,
		gint event,
		gpointer data) {
		
	switch(event) {
		case PROFILES_LAUNCHER_EVENT_STDOUT:
		case PROFILES_LAUNCHER_EVENT_STDERR:
			{ /* c99 hack */
				gchar *str=(gchar *)data;
				text_append(str);
				text_append("\n");
			}
		break;

		case PROFILES_LAUNCHER_EVENT_CONNECTING:
		set_status(CONNECTING);
		text_append("Connecting ...\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_CONNECTING_ERROR:
		text_append("Connecting error!\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_CONNECTED:
		set_status(CONNECTED);
		text_append("Connected!\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_DISCONNECTING:
		set_status(DISCONNECTING);
		text_append("Disconnecting ...\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_DISCONNECTING_ERROR:
		text_append("Disconnecting error!\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_DISCONNECTED:
		set_status(DISCONNECTED);
		text_append("Disconnected!\n");
		break;
		
		case PROFILES_LAUNCHER_EVENT_PASSWORD_CHALLENGE:
		text_append("Got password challenge!\n");
		

		gchar *password=read_password_dialog();
		if(password==NULL) {
			puts("user cancelled pw dialog");
			// TODO: fix!
			button_disconnect_cb();
		}
		else {
			//printf("pw: %s\n", password);
			profiles_launcher_feed_password(p, password);
			g_free(password);
		}
		
		break;

		case PROFILES_LAUNCHER_EVENT_AUTHENTICATION_FAILED:
		text_append("Authentication failed, wrong password?\n");
		break;

		case PROFILES_LAUNCHER_EVENT_TUNDEV_FAILED:
		text_append("Tunnel device could not be initilized.\n");
		break;

		case PROFILES_LAUNCHER_EVENT_TARGET_NOT_RESPONDING:
		text_append("Target server not responding, network problem?\n");
		break;

		case PROFILES_LAUNCHER_EVENT_UNKNOWN_HOST:
		text_append("Unknown host, DNS problem?\n");
		break;

	}
	
}

void termination_handler(int sig) {


	fprintf(stderr, "Caught signal %d, disconnecting ...\n", sig);
	
	/* disconnect current connection (to kill
	 * childen)
	 */
	// TODO: fix it
	if(status != DISCONNECTED) {

		struct profile *p=profiles_gui_get_selected_profile();
		if(p!=NULL) {
			profiles_launcher_disconnect(p);
		}
	}

	gtk_main_quit();

}

void window_closed() {
	termination_handler(SIGTERM);
}

/* Callback for "Close" menu entry */
void item_close_cb()
{
	window_closed();
}


/* Callback for "Save-Buffer" menu entry */
void item_save_buffer_cb()
{

#ifdef DEBUG
    g_print("saving buffer ...\n");
#endif

    save_buffer_show_dialog(GTK_WINDOW(window), text);
}

/* Create the menu items needed for the main view */
static void create_menu(HildonWindow * main_window) 
{


  /* Taken from GTK Tutorial:
   * http://www.gtk.org/tutorial1.2/gtk_tut-13.html */

  static GtkItemFactoryEntry menu_items[] = {
    { "/_Profiles",         NULL,         NULL, 0, "<Branch>" },
    { "/Profiles/_New",     NULL, profiles_gui_new_profile, 0, NULL },
    { "/Profiles/_Delete",  NULL, profiles_gui_delete_profile, 0, NULL },
    { "/Profiles/_Rename",  NULL, profiles_gui_rename_profile, 0, NULL },
    { "/Profiles/_Edit",    NULL, profiles_gui_edit_profile, 0, NULL },
    { "/_Configure Autoconnect", NULL, autoconnect_gui_edit_preferences, 0, NULL },
    { "/_Save Buffer", NULL, item_save_buffer_cb, 0, NULL},
    { "/_About", NULL, show_about_cb, 0, NULL},
    { "/_Close", NULL, item_close_cb, 0, NULL},
  };

  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

  accel_group = gtk_accel_group_new ();

  /* This function initializes the item factory.
     Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU,
     or GTK_TYPE_OPTION_MENU.
     Param 2: The path of the menu.
     Param 3: A pointer to a gtk_accel_group.  The item factory sets up
     the accelerator table while generating menus.
     */

  item_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<main>", 
      accel_group);

  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

  /* Attach the new accelerator group to the window. */
  gtk_window_add_accel_group (GTK_WINDOW (main_window), accel_group);

  GtkWidget *main_menu = gtk_item_factory_get_widget (item_factory, "<main>");

  /* set hildon window menu */
  hildon_window_set_menu(HILDON_WINDOW(main_window), GTK_MENU(main_menu));

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));

}


int main(int argc, char *argv[]) {

  /* Add signal handlers */
  signal(SIGINT, termination_handler);
  signal(SIGTERM, termination_handler);


  //  g_set_application_name (_("vpnc GUI"));
  
  /* NEEDED FOR libconic to work!! */
  dbus_listener_init();
  

  /* Initialize the GTK. */
  gtk_init(&argc, &argv);


  /* Initialize autoconnect GUI */
  autoconnect_gui_init();

  /* register autoconnect callback */
  autoconnect_gui_set_autoconnect_cb(autoconnect_cb);


  /* init launcher */
  profiles_launcher_init();
  
  /* register launcher callback */
  profiles_launcher_set_event_callback(launcher_event_cb);
  

  /*
  osso_context_t  *osso_context = osso_initialize("vpnc-gui", "0.0", TRUE, NULL);
  if (osso_context == NULL) {
      g_printerr("osso_initialize() failed!\n");
      exit(1);
  }
  */



  /* Create the hildon program and setup the title */
  program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("vpnc GUI");

  /* Create HildonWindow and set it to HildonProgram */
  window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(program, window);


  GtkWidget *vbox=gtk_vbox_new(FALSE, 10);

  ///////////// PROFILES

  GtkWidget *profiles_hbox=gtk_hbox_new(FALSE, 10);

  // TODO
  profiles_gui_init();
  GtkWidget *profiles_combo=GTK_WIDGET(profiles_gui_get_combo_box());

  gtk_box_pack_start(GTK_BOX(profiles_hbox), gtk_label_new("Profile"),
      // expand, fill, padding
      FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(profiles_hbox), profiles_combo,
      TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(vbox), profiles_hbox, TRUE, TRUE, 0);
  /////////////

  text=gtk_text_new(NULL, NULL);
  gtk_text_set_editable(GTK_TEXT(text), FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), text, TRUE, TRUE, 0);

  ////////////
  GtkWidget *hbox=gtk_hbox_new(FALSE, 10);

  connect=gtk_button_new_with_label("Connect");
  disconnect=gtk_button_new_with_label("Disconnect");

  gtk_box_pack_start(GTK_BOX(hbox), connect,
		     TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(hbox), disconnect,
		   TRUE, TRUE, 0);

  g_signal_connect(G_OBJECT(connect), "clicked",
		   GTK_SIGNAL_FUNC(button_connect_cb), NULL);

  g_signal_connect(G_OBJECT(disconnect), "clicked",
		   GTK_SIGNAL_FUNC(button_disconnect_cb), NULL);

  /////////////////

  gtk_box_pack_end(GTK_BOX(vbox), hbox,
      // expand, fill, padding
      TRUE, TRUE, 0);



  /* Add box to window */
  gtk_container_add(GTK_CONTAINER(window),
		    vbox);

  /* add menu */
  create_menu(window);

  /* Begin the main application */
  gtk_widget_show_all(GTK_WIDGET(window));

  /* Connect signal to X in the upper corner */
  g_signal_connect(G_OBJECT(window), "delete_event",
		   G_CALLBACK(window_closed), NULL);


  /* set initial status to be disconnected */
  set_status(DISCONNECTED);
  
  gtk_main();


  fprintf(stderr, "Exiting ...\n");

  /* Exit */
  return 0;
}
