/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* IMPORTS */
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <glib.h>

#include "profiles_backend.h"
#include "profile.h"
#include "fileaccess_vpnc.h"


/* CONSTANTS */
#define CONFIG_DIR "/.vpnc-gui/"


/* (local) VARIABLES */

/** path to config-dir, including trailing slash */
static char *config_path;

/** list of all profiles */
static GList *profiles;


/* LOCAL FUNCTIONS */


/* EXPORTS */

/**
 * Generate config-file-name for profile. Aborts if memory allocation
 * fails.
 * Result must be freed.
 * 
 * @param p the profile
 * @return a g_malloced pointer to a string containing the filename
 */
gchar *profiles_backend_profile_to_filename(struct profile *p) {

	char *suffix="FIXME";
	if(p->type == PROFILE_TYPE_VPNC) {
		suffix=".vpnc";
	}
	else if(p->type == PROFILE_TYPE_OPENVPN) {
		suffix=".openvpn";
	}
	else {
		printf("Implement profile type %d in %s:%d!\n", p->type, __FILE__, __LINE__);
	}

	int size=strlen(config_path)+strlen(p->name)+strlen(suffix)+1;
	gchar *file=g_malloc(size);
	g_strlcpy(file, config_path, size);
	g_strlcat(file, p->name, size);
	g_strlcat(file, suffix, size);

	printf("File: %s\n", file);

	return file;
}

/**
 * Initialize the profiles backend, must be called before
 * any other functions!
 * Aborts if memory allocation fails or any other bad thing happens.
 */
void profiles_backend_init() {

	struct passwd *pw=getpwuid(getuid());

	if(pw==NULL) {
		perror("getpwuid failed");
		exit(1);
	}

	config_path=g_malloc(strlen(pw->pw_dir)+
			strlen(CONFIG_DIR)+
			1);

	// TODO: g_strcpy, g_strcat
	strcpy(config_path, pw->pw_dir);
	strcpy(config_path+strlen(pw->pw_dir), CONFIG_DIR);

	/* try to create dir */
	if(mkdir(config_path, 0700) == -1 && errno != EEXIST) {
		perror("mkdir failed");
		exit(1);
	}


#ifdef DEBUG
	printf("Config-path: %s\n", config_path);
#endif



	DIR *d=opendir(config_path);

	struct dirent *de;

	if(d==NULL) {
		perror("opendir failed");
		exit(1);
	}
	
	while((de=readdir(d)) != NULL) {

		if(strlen(de->d_name) > strlen(".openvpn") &&
				strcmp(".openvpn",
					de->d_name+strlen(de->d_name)-strlen(".openvpn"))
				== 0) {

			struct profile *p=profile_alloc();
			p->name=g_malloc(strlen(de->d_name)-strlen(".openvpn")+1);
			strncpy(p->name, de->d_name, strlen(de->d_name)-strlen(".openvpn"));
			p->name[strlen(de->d_name)-strlen(".openvpn")]='\0';
			p->type=PROFILE_TYPE_OPENVPN;

			profiles=g_list_append(profiles, p);

		}
		if(strlen(de->d_name) > strlen(".vpnc") &&
				strcmp(".vpnc",
					de->d_name+strlen(de->d_name)-strlen(".vpnc"))
				== 0) {

			struct profile *p=profile_alloc();
			p->name=g_malloc(strlen(de->d_name)-strlen(".vpnc")+1);
			strncpy(p->name, de->d_name, strlen(de->d_name)-strlen(".vpnc"));
			p->name[strlen(de->d_name)-strlen(".vpnc")]='\0';
			p->type=PROFILE_TYPE_VPNC;
			gchar *fpath=profiles_backend_profile_to_filename(p);
			p->config_data=fileaccess_vpnc_load_file(fpath);
			g_free(fpath);

			profiles=g_list_append(profiles, p);

		}
	}


	closedir(d);

}

/**
 * Get a list of all profiles. Result must not be freed!
 * @return a pointer to a GList containing the first profile
 */
GList *profiles_backend_list_profiles() {
	return profiles;
}

/**
 * Test if a profile with that name already exists.
 * @param name the name
 * @return TRUE if it exists
 */
gboolean profiles_backend_profile_name_exists(const gchar *name) {
        for(GList *li = profiles; li!= NULL; li = g_list_next(li)) {
                struct profile *p = li->data;
		if(strcasecmp(p->name, name)==0)
			return TRUE;
        }
	return FALSE;
}

/**
 * Test if given name is a valid profile name.
 * If name is not valid returns a reason-string.
 * Returned strings must not be freed.
 * 
 * @param name the name
 * @return error-reason or NULL if name is OK
 */
gchar *profiles_backend_profile_name_invalid(const gchar *name) {
	if(strchr(name, '/'))
		return "No '/' allowed";
	if(strchr(name, '\n'))
		return "No newline allowed";
	// TODO: is that list complete?
	
	return NULL;
}

/**
 * Delete the profile with specified name.
 * This function calls profile_free on the profile.
 * 
 * @param name the name
 * @return TRUE if deletion was successful
 */
gboolean profiles_backend_delete_profile_by_name(char *name) {
	
	for(GList *li = profiles; li!= NULL; li = g_list_next(li)) {
                struct profile *p = li->data;
                if(strcmp(p->name, name)==0) {

			/* remove from filesystem */
			gchar *file=profiles_backend_profile_to_filename(p);
			if(unlink(file)!=0) {
				perror("unlink");
				g_free(file);
				return FALSE;
			}
			g_free(file);
			
			/* remove item from list */
			profiles=g_list_remove(profiles, p);
			
			// TODO: free config DATA!!!
			
			/* free profile */
			profile_free(p);
                        return TRUE;
		}
        }

	return FALSE;
}

/**
 * Create a new profile or change an existing one.
 * Memory of specified profile must not be freed after successfully
 * calling this function.
 * 
 * @param p pointer to the profile to be created
 * @return TRUE if creation was successful
 */
gboolean profiles_backend_save_profile(struct profile *p) {

	gboolean success=FALSE;
	gboolean new_creation=TRUE;

	if(profiles_backend_profile_name_exists(p->name)) {
		new_creation=FALSE;
	}
	
	if(p->type == PROFILE_TYPE_VPNC) {
		gchar *file=profiles_backend_profile_to_filename(p);
		success=fileaccess_vpnc_save_file(file, p->config_data);
		g_free(file);
	}
	else if(p->type == PROFILE_TYPE_OPENVPN) {
		// TODO: implement
		return FALSE;
	}
	else {
		printf("Implement profile type %d in %s:%d!\n", p->type, __FILE__,__LINE__);
		return FALSE;
	}
	
	if(success && new_creation)
		profiles=g_list_append(profiles, p);

	return success;
}

/**
 * Rename profile.
 * @param p the profile to be renamed
 * @param new_name the new name of the profile, no references are kept
 * @return true if successful
 */
gboolean profiles_backend_rename_profile(struct profile *p, const gchar *new_name) {
	gchar *old_file=profiles_backend_profile_to_filename(p);
	gchar *old_name=p->name;
	p->name=g_strdup(new_name);
	gchar *new_file=profiles_backend_profile_to_filename(p);

	if(rename(old_file, new_file)!=0) {
		perror("rename");
		g_free(p->name); // free new name
		p->name=old_name; // restore old name
		g_free(old_file);
		g_free(new_file);
		return FALSE;
	}
	
	g_free(old_name);
	g_free(old_file);
	g_free(new_file);
	
	return TRUE;
}
