#include "volumecontroldaemon.h"
#include <QDebug>
#include <QProcess>
#include <QTimer>
#include <QDBusInterface>
#include <QFile>

VolumeControlDaemon::VolumeControlDaemon(QObject *parent) :
    QObject(parent)
{
}

VolumeControlDaemon::~VolumeControlDaemon()
{
    qDebug() << QString("Stopping daemon...");

    QDBusConnection conn = QDBusConnection::sessionBus();
    conn.disconnect("", "/com/nokia/mafw/renderer/gstrenderer",
                    "com.nokia.mafw.extension",
                    "property_changed", this, SLOT(VolumeChangedSlot(const QDBusMessage&)));
    conn.disconnect("", "/com/nokia/mafw/renderer/gstrenderer",
                    "com.nokia.mafw.renderer",
                    "media_changed", this, SLOT(MediaChangedSlot(int, QString)));
    conn.disconnect("", "/com/nokia/mafw/renderer/gstrenderer",
                    "com.nokia.mafw.renderer",
                    "state_changed", this, SLOT(StateChangedSlot(int)));

    conn = QDBusConnection::systemBus();
    conn.disconnect("com.nokia.csd.Call", "/com/nokia/csd/call", "com.nokia.csd.Call",
                    "Coming", this, SLOT(IncomingCallSlot(const QDBusMessage&)));
    conn.disconnect("com.nokia.csd.Call", "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                    "AudioConnect", this, SLOT(AudioCallConnectedSlot(const QDBusMessage&)));

}

void VolumeControlDaemon::Start()
{
    qDebug() << QString("Starting daemon...");
    QTimer::singleShot(500, this, SLOT(Init()));
}

void VolumeControlDaemon::Init()
{
    QDBusConnection conn = QDBusConnection::sessionBus();
    if (!conn.connect("", "/com/nokia/mafw/renderer/gstrenderer",
                      "com.nokia.mafw.extension",
                      "property_changed", this, SLOT(VolumeChangedSlot(const QDBusMessage&)))){
        qCritical() << QString("Failed to connect to com.nokia.mafw.extension::property_changed");
        emit quit();
        return;
    }

    if (!conn.connect("", "/com/nokia/mafw/renderer/gstrenderer",
                      "com.nokia.mafw.renderer",
                      "media_changed", this, SLOT(MediaChangedSlot(int, QString)))){
        qCritical() << QString("Failed to connect to com.nokia.mafw.renderer::media_changed");
        emit quit();
        return;
    }

    if (!conn.connect("", "/com/nokia/mafw/renderer/gstrenderer",
                      "com.nokia.mafw.renderer",
                      "state_changed", this, SLOT(StateChangedSlot(int)))){
        qCritical() << QString("Failed to connect to com.nokia.mafw.renderer::state_changed");
        emit quit();
        return;
    }

    conn = QDBusConnection::systemBus();
    if (!conn.connect("com.nokia.csd.Call", "/com/nokia/csd/call", "com.nokia.csd.Call",
                      "Coming", this, SLOT(IncomingCallSlot(const QDBusMessage&)))){
        qCritical() << QString("Failed to connect to com.nokia.csd.Call::Coming");
        emit quit();
        return;
    }

    if (!conn.connect("com.nokia.csd.Call", "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                      "AudioConnect", this, SLOT(AudioCallConnectedSlot(const QDBusMessage&)))){
        qCritical() << QString("Failed to connect to com.nokia.csd.Call.Instance::AudioConnect");
        emit quit();
        return;
    }

    AdjustVolume();
    qDebug() << QString("Waiting for dbus events...");
}

void VolumeControlDaemon::SetHeadphoneVolume(int vol)
{
    qDebug() << QString("Setting headphone volume to %1%").arg(vol);
    QProcess p(this);
    p.start(QString("amixer -c 0 set Headphone %1%").arg(vol));
    p.waitForFinished();
    if (p.exitCode() != 0)
        qWarning() << QString("Amixer returned %1").arg(p.exitCode());
}

void VolumeControlDaemon::VolumeChangedSlot(const QDBusMessage &msg)
{   
    QList<QVariant> lst = msg.arguments();
    QString property;
    if (lst.count() > 1){
        property = lst.at(0).toString();

        if (property == "volume"){
            QVariant val = qvariant_cast<QDBusVariant>(msg.arguments()[1]).variant();
            int currVolume = val.toInt();
            AdjustVolume(currVolume);
        }
    }
}

void VolumeControlDaemon::MediaChangedSlot(int, QString)
{
    qDebug() << QString("Media changed");
    AdjustVolume();
}

void VolumeControlDaemon::StateChangedSlot(int state)
{
    if (state == 1){
        qDebug() << QString("State changed");
        AdjustVolume();
    }
}

void VolumeControlDaemon::IncomingCallSlot(const QDBusMessage&)
{
    qDebug() << QString("Call: incoming");
    AdjustVolume();
}

void VolumeControlDaemon::AudioCallConnectedSlot(const QDBusMessage&)
{
    qDebug() << QString("Call: audio connected");
    AdjustVolume();
}

void VolumeControlDaemon::AdjustVolume(int currentVolume)
{
    if (IsHeadphoneConnected()){
        if (currentVolume < 0){
            //Get current volume:
            QDBusConnection conn = QDBusConnection::sessionBus();
            QDBusInterface device("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                  "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.extension", conn);
            QDBusMessage msg = device.call("get_extension_property", "volume");

            QVariant val = qvariant_cast<QDBusVariant>(msg.arguments()[1]).variant();
            currentVolume = val.toInt();
        }
        qDebug() << QString("Current volume: %1%").arg(currentVolume);

        //Strange, some N900 reach 99% instead of 100%
        if (currentVolume >= 99)
            SetHeadphoneVolume(currentVolume);
    }
}

bool VolumeControlDaemon::IsHeadphoneConnected()
{
    QFile h(STATE_FILE);
    if (!h.open(QIODevice::ReadOnly)){
        qWarning() << QString("Failed to open %1").arg(STATE_FILE);
        return false;
    }

    QByteArray state = h.readAll();
    QString str(state);
    h.close();
    return str.trimmed() == "connected";
}
