//    Vodafone 190
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "desktopwidget.h"
#include <QApplication>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QX11Info>
#include <QDBusConnection>
#include <QNetworkInterface>
#include <QMaemo5InformationBox>
#include <QPainter>
#include <QDebug>
#include <QUuid>
#include "commonstatic.h"
#include "widgetmanager.h"
#include "widget/counterwidget.h"
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

DesktopWidget::DesktopWidget(QWidget *parent, QString appletId, bool debug) :
    QFrame(parent)
{
    QCoreApplication::instance()->setApplicationName("Vodafone 190 desktop widget");

    // Get required atoms
    Atom winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
    Atom homeAppletAtom = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
    Atom appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
    Atom utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false);

    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32,
            PropModeReplace, (unsigned char *) &homeAppletAtom, 1);

    if (!appletId.isEmpty())
        m_AppletId = appletId;
    else
        m_AppletId = QString("%1_%2").arg(QCoreApplication::instance()->applicationName().remove(' ')).arg(QUuid::createUuid().toString().remove("{").remove("}"));

    qDebug() << QString("AppletId = %1").arg(m_AppletId);
    QByteArray id (m_AppletId.toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8,
                    PropModeReplace, (unsigned char *)id.constData(), id.length());

    // Add setting button. This button is shown when hildon-desktop is in edit mode.
    Atom appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
    int settings = 0;
    XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32,
            PropModeReplace, (unsigned char*)&settings, 1);

    //Connect to newtork state change:
    QDBusConnection::systemBus().connect("com.nokia.icd", "/com/nokia/icd", "com.nokia.icd",
                                         "status_changed", this, SLOT(netStateChanged(QString,QString,QString,QString)));

    //Connect to display state change:
    //QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
    //                                     MCE_DISPLAY_SIG, this, SLOT(displayStateChanged(QString)));

    m_UpdateTimer = NULL;
    m_Updating = false;
    m_Vodafone = new VodafoneWidget("", "");
    m_Vodafone->DebugXml = debug;
    m_Vodafone->RequestTimeoutMsec = 60000;

    m_SettingsWindow = NULL;
    m_Settings = new Settings(this, m_AppletId);
    bool loaded = m_Settings->Load();
    //Apply stylesheet:
    if (!loaded || m_Settings->StyleSheetName == "Default")
        CommonStatic::SetDefaultStylesheet(this);
    else if (m_Settings->StyleSheetName == "System")
        CommonStatic::SetSystemStylesheet(this);
    else{
        QString filename = QString("%1/%2.css").arg(CommonStatic::GetStylesheetFolder())
                                               .arg(m_Settings->StyleSheetName);
        CommonStatic::SetStylesheet(this, filename);
    }

    m_WaitIcon = new QMovie(":/images/wait.gif");
    m_WaitIcon->setScaledSize(QSize(28, 28));

    setFrameStyle(QFrame::NoFrame | QFrame::Plain);
    setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
    setContentsMargins(6,6,6,6);

    m_Layout = new QGridLayout(this);
    m_Layout->setSpacing(3);
    m_Layout->setContentsMargins(0,0,0,0);
    m_Layout->setSizeConstraint(QLayout::SetMinAndMaxSize);

    setLayout(m_Layout);
    CreateWidgets();
    if (loaded)
        m_Vodafone->SetCredentials(m_Settings->Username, m_Settings->Password);
    SetWidgetSize();

} // DesktopWidget

DesktopWidget::~DesktopWidget()
{    
    if (m_UpdateTimer != NULL){
        m_UpdateTimer->stop();
        delete m_UpdateTimer;
        m_UpdateTimer = NULL;
    }

    QDBusConnection::systemBus().disconnect("com.nokia.icd", "/com/nokia/icd", "com.nokia.icd",
                                            "status_changed", this, SLOT(netStateChanged(QString,QString,QString,QString)));

    /*QDBusConnection::systemBus().disconnect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                            MCE_DISPLAY_SIG, this, SLOT(displayStateChanged(QString)));*/

    if (m_WaitIcon != NULL)
        delete m_WaitIcon;
    delete m_Vodafone;
    delete m_Layout;
}

void DesktopWidget::show()
{    
    ShowSavedData();
    QFrame::show();
    QTimer::singleShot(1000, this, SLOT(Update()));
}

void DesktopWidget::CreateTimer()
{
    if (m_Settings->UpdateInterval > 0){
        qDebug() << QString("Auto update interval set to %1").arg(CommonStatic::FormatHourMinutes(m_Settings->UpdateInterval));
        if (m_UpdateTimer == NULL){
            m_UpdateTimer = new QTimer(this);
            connect(m_UpdateTimer, SIGNAL(timeout()), this, SLOT(Update()));
        }
        m_UpdateTimer->stop();
        m_UpdateTimer->setInterval(m_Settings->UpdateInterval * 1000);
        m_UpdateTimer->setSingleShot(true);
        m_UpdateTimer->start();
    }else{
        if (m_UpdateTimer != NULL){
            m_UpdateTimer->stop();
            delete m_UpdateTimer;
            m_UpdateTimer = NULL;
        }
    }
}

//Filter show setting requests
bool DesktopWidget::x11Event(XEvent* event)
{
  static Atom appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);

  if (event->xclient.message_type == appletShowSettingAtom ) {
    showSettingsDialog();
    return true;
  }
  return false;
}

void DesktopWidget::showSettingsDialog()
{
    m_SettingsWindow = new SettingsWindow(this, m_Settings, m_Vodafone);
    m_SettingsWindow->show();
    connect(m_SettingsWindow, SIGNAL(SettingsChanged()), this, SLOT(settingsModified()));
} // showSettingsDialog

void DesktopWidget::settingsModified()
{
    QLayoutItem *child;
    while ((child = m_Layout->takeAt(0)) != 0) {
        delete child->widget();
        delete child;
    }
    CreateWidgets();
    m_Vodafone->SetCredentials(m_Settings->Username, m_Settings->Password);
    delete m_SettingsWindow;
    m_SettingsWindow = NULL;
    Update();
}

void DesktopWidget::CreateWidgets()
{    
    qDebug() << "Creating widgets";
    int row = 0;

    if (m_Settings->AllSIM.count() > 1){
        m_lblSIM = new QLabel(this);
        m_lblSIM->setWordWrap(false);
        m_lblSIM->setAlignment(Qt::AlignCenter);
        m_lblSIM->setText(m_Settings->SIM);
        m_Layout->addWidget(m_lblSIM, row, 0, 1, 1, Qt::AlignTop);

        row++;
    }

    Qt::Alignment align = Qt::AlignRight | Qt::AlignVCenter;
    int col = 1;
    int colSpan = 2;
    if (!m_Settings->ShowLabels){
        align = Qt::AlignCenter;
        col = 0;
        colSpan = 1;
    }

    if (m_Settings->ShowCredito){
        m_lblCreditoD = NULL;
        if (m_Settings->ShowLabels){
            m_lblCreditoD = new QLabel(this);
            m_lblCreditoD->setWordWrap(false);
            m_lblCreditoD->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
            m_lblCreditoD->setText(tr("Credito:"));
            m_Layout->addWidget(m_lblCreditoD, row, 0, 1, 1, Qt::AlignTop);
        }
        m_lblCredito = new QLabel(this);
        m_lblCredito->setWordWrap(false);
        m_lblCredito->setAlignment(align);
        m_lblCredito->setText("-");
        m_Layout->addWidget(m_lblCredito, row, col, 1, 1, Qt::AlignTop);

        row++;
    }

    if (m_Settings->ShowPunti){
        if (m_Settings->ShowLabels){
            QLabel* lbl = new QLabel(this);
            lbl->setWordWrap(false);
            lbl->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
            lbl->setText(tr("Punti:"));
            m_Layout->addWidget(lbl, row, 0, 1, 1, Qt::AlignTop);
        }

        m_lblPunti = new QLabel(this);
        m_lblPunti->setWordWrap(false);
        m_lblPunti->setAlignment(align);
        m_lblPunti->setText("-");
        m_Layout->addWidget(m_lblPunti, row, col, 1, 1, Qt::AlignTop);

        row++;
    }

    if (m_Settings->ShowCounters.count() > 0){
        m_CountersFrame = new QFrame(this);
        m_CountersFrame->setContentsMargins(0,0,0,0);
        m_CountersFrame->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
        m_CountersFrame->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
        m_Layout->addWidget(m_CountersFrame, row, 0, 1, colSpan, Qt::AlignTop);

        m_CountersLayout = new QVBoxLayout();
        m_CountersLayout->setContentsMargins(0,0,0,0);
        m_CountersLayout->setSpacing(3);
        m_CountersLayout->setSizeConstraint(QLayout::SetMinAndMaxSize);
        m_CountersFrame->setLayout(m_CountersLayout);

        row++;
    }
}

bool DesktopWidget::Update()
{
    bool result = UpdateData();
    if (!result)
        ShowSavedData();
    CreateTimer();
    return result;
}

bool DesktopWidget::UpdateData()
{
    if (m_Updating)
        return true;

    if (!IsNetConnected()){
        qDebug() << "No network connection, update aborted";
        return false;
    }

    qDebug() << "Updating data...";
    m_Updating = true;
    if (m_Settings->ShowCredito)
        StartWaitAnimation(m_lblCredito);

    qDebug() << "Logging in...";
    QString valuta;
    if (m_Vodafone->IsLoggedIn() || m_Vodafone->Login()){
        qDebug() << "Logged in";
        if (m_Vodafone->GetSims().count() > 1 && !m_Settings->SIM.isEmpty())
            m_Vodafone->SetCurrentSim(m_Settings->SIM);

        if (m_Settings->ShowCredito){
            float credito = 0;
            if (m_Vodafone->GetCurrentSimType() == VodafoneWidget::SimRicaricabile){
                if (m_lblCreditoD)
                    m_lblCreditoD->setText(tr("Credito:"));
                if (IsNetConnected())
                    credito = m_Vodafone->GetCredito(&valuta);
                if (valuta == "E")
                    valuta = QChar(8364);
            }else{
                if (m_lblCreditoD)
                    m_lblCreditoD->setText(tr("Traffico:"));
                InfoConto* conto = new InfoConto(this);
                if (IsNetConnected())
                    m_Vodafone->GetInfoConto(conto);
                credito = conto->TrafficoNonFatturato;
                valuta = QChar(8364);
            }

            StopWaitAnimation(m_lblCredito);

            m_Settings->LastCredito = credito;
            m_Settings->LastValuta = valuta;
            m_lblCredito->setText(QString("%1 %2").arg(credito, 4, 'f', 2).arg(valuta).trimmed());
        }


        if (m_Settings->ShowPunti && IsNetConnected()){
            StartWaitAnimation(m_lblPunti);
            qDebug() << "Getting saldo punti";

            SaldoPunti* punti = new SaldoPunti();
            if (IsNetConnected())
                m_Vodafone->GetSaldoPunti(punti);

            StopWaitAnimation(m_lblPunti);

            m_Settings->LastSaldoPunti = punti;
            m_lblPunti->setText(QString("%1").arg(punti->Punti));
        }

        if (m_Settings->ShowCounters.count() > 0 && IsNetConnected()){
            QLayoutItem *child;
            while ((child = m_CountersLayout->takeAt(0)) != 0) {
                delete child->widget();
                delete child;
            }

            qDebug() << "Getting counters details";
            QLabel* lbl = NULL;
            foreach(Counter* c, m_Settings->ShowCounters){
                lbl = new QLabel(this);
                lbl->setAlignment(Qt::AlignCenter);
                StartWaitAnimation(lbl);
                m_CountersLayout->addWidget(lbl, 0, Qt::AlignTop);

                if (IsNetConnected() && m_Vodafone->GetDettaglioContatore(c)){
                    StopWaitAnimation(lbl);
                    QLayoutItem *child = m_CountersLayout->takeAt(m_CountersLayout->count() - 1);
                    delete child->widget();
                    delete child;

                    c->WidgetSize = m_Settings->CounterStyle;
                    CounterWidget* w = new CounterWidget(this, c, m_Settings);
                    w->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
                    m_CountersLayout->addWidget(w, 0, Qt::AlignTop);
                    w->show();
                }else{
                    StopWaitAnimation(lbl);
                    QLayoutItem *child = m_CountersLayout->takeAt(m_CountersLayout->count() - 1);
                    delete child->widget();
                    delete child;
                }
                lbl = NULL;
            }
        }

        SetWidgetSize();
    }else{
        StopWaitAnimation(m_lblCredito);
        m_Updating = false;
        return false;
    }

    m_Settings->Save();
    qDebug() << "Data updated";
    m_Updating = false;
    return true;
}

/*
void DesktopWidget::displayStateChanged(QString displayState)
{
    if (displayState == QLatin1String(MCE_DISPLAY_OFF_STRING)) {
        if (m_UpdateTimer)
            m_UpdateTimer->stop();
    }else if (displayState == QLatin1String(MCE_DISPLAY_ON_STRING)) {
        if (m_UpdateTimer)
            CreateTimer();
    }
}
*/

void DesktopWidget::netStateChanged(QString, QString type, QString state, QString)
{
    qDebug() << QString("Network status changed: %1 %2").arg(type).arg(state);
    if (type == "WLAN_INFRA" && (m_Settings->UpdateOn == Settings::connWLAN || m_Settings->UpdateOn == Settings::connBoth) ||
        type == "GPRS" && (m_Settings->UpdateOn == Settings::connGPRS || m_Settings->UpdateOn == Settings::connBoth)){
        if (state == "CONNECTED"){
            Update();
        }else if (state == "DISCONNECTING"){
            m_Vodafone->AbortRequest();
            if (m_UpdateTimer)
                m_UpdateTimer->stop();
        }
    }
}

void DesktopWidget::StartWaitAnimation(QLabel* label)
{
    if (m_Settings->Animation){
        label->setMovie(m_WaitIcon);
        m_WaitIcon->start();
    }
}

void DesktopWidget::StopWaitAnimation(QLabel* label)
{
    m_WaitIcon->stop();
    label->setMovie(NULL);
    label->setPixmap(NULL);
}

bool DesktopWidget::IsNetConnected()
{
    QNetworkInterface wlan = QNetworkInterface::interfaceFromName("wlan0");
    QNetworkInterface gprs = QNetworkInterface::interfaceFromName("gprs0");
    if((wlan.isValid() && wlan.flags().testFlag(QNetworkInterface::IsUp)) ||
       (gprs.isValid() && gprs.flags().testFlag(QNetworkInterface::IsUp)))
        return true;
    return false;
}

void DesktopWidget::SetWidgetSize()
{
    setFixedSize(m_Settings->WidgetWidth, sizeHint().height());
}

void DesktopWidget::ShowSavedData()
{
    if (m_Settings->ShowCredito)
        m_lblCredito->setText(QString("%1 %2").arg(m_Settings->LastCredito, 4, 'f', 2).arg(m_Settings->LastValuta).trimmed());
    if (m_Settings->ShowPunti && m_Settings->LastSaldoPunti != NULL)
        m_lblPunti->setText(QString("%1").arg(m_Settings->LastSaldoPunti->Punti));

    QLayoutItem *child;
    while ((child = m_CountersLayout->takeAt(0)) != 0) {
        delete child->widget();
        delete child;
    }

    foreach(Counter* c, m_Settings->ShowCounters){
        if (c->Soglie.count() > 0){
            c->WidgetSize = m_Settings->CounterStyle;
            CounterWidget* w = new CounterWidget(this, c, m_Settings);
            w->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
            m_CountersLayout->addWidget(w, 0, Qt::AlignTop);
            w->show();
        }
    }

    SetWidgetSize();
}

QString DesktopWidget::GetAppletId()
{
    return m_AppletId;
}

void DesktopWidget::closeEvent(QCloseEvent *event)
{
    qDebug() << "Close event: removing widget";
    WidgetManager::RemoveWidget(m_AppletId);
    QFrame::closeEvent(event);
    deleteLater();
}

void DesktopWidget::mousePressEvent(QMouseEvent* event)
{
    if (event->button() == Qt::LeftButton) {
        m_Pressed = true;
        event->accept();
        return;
    }
    QFrame::mousePressEvent(event);
}


void DesktopWidget::mouseReleaseEvent(QMouseEvent* event)
{
    if (m_Pressed){
        m_Pressed = false;
        Update();
        event->accept();
        return;
    }
    QFrame::mouseReleaseEvent(event);
}
