//    Vodafone 190
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "smswindow.h"
#include "ui_smswindow.h"
#include <QMovie>
#include <QRegExp>

#include <glib-2.0/glib.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger/event.h>
#include <QUuid>
#include "commonstatic.h"
#include "captchawindow.h"
#include "contactswindow.h"

SmsWindow::SmsWindow(QWidget *parent, VodafoneWidget* vodafone, Settings* settings) :
    QDialog(parent),
    ui(new Ui::SmsWindow)
{
    ui->setupUi(this);

#ifdef Q_OS_SYMBIAN
    setWindowTitle(QString(tr("SMS (%1/10)")).arg(settings->SmsSent));
#else
    setWindowTitle(QString(tr("Vodafone 190 - SMS (%1/10)")).arg(settings->SmsSent));
#endif

    m_Settings = settings;
    m_Vodafone = vodafone;
    connect(ui->btnContacts, SIGNAL(clicked()), this, SLOT(ContactsSlot()));
    connect(ui->btnSend, SIGNAL(clicked()), this, SLOT(SendSlot()));    
    connect(ui->btnCancel, SIGNAL(clicked()), this, SLOT(CancelSlot()));
    connect(ui->txtMessage, SIGNAL(textChanged()), this, SLOT(MessageChangedSlot()));

#ifdef Q_WS_MAEMO_5
    ui->lblWait->setVisible(false);
    ui->txtReceiver->setInputMethodHints(Qt::ImhDigitsOnly);
#elif defined(Q_OS_SYMBIAN)
    ui->txtReceiver->setInputMethodHints(Qt::ImhDigitsOnly);
    ui->lblDest->setText(tr("Dest.:"));

    ui->btnSend->setVisible(false);
    ui->btnCancel->setVisible(false);

    QAction* softKey = new QAction( tr("Invia"), this );
    softKey->setSoftKeyRole( QAction::PositiveSoftKey );
    connect(softKey, SIGNAL(triggered()), this, SLOT(SendSlot()));
    addAction(softKey);

    softKey = new QAction( tr("Indietro"), this );
    softKey->setSoftKeyRole( QAction::NegativeSoftKey );
    connect(softKey, SIGNAL(triggered()), this, SLOT(CancelSlot()));
    addAction(softKey);
#else
    QRegExp re("[0-9]*");
    QRegExpValidator *validator = new QRegExpValidator(re, this);
    ui->txtReceiver->setValidator(validator);
#endif

    MessageChangedSlot();
}

SmsWindow::~SmsWindow()
{
    delete ui;
}

void SmsWindow::ContactsSlot()
{
    ContactsWindow* frm = new ContactsWindow(this);
#if defined(Q_OS_SYMBIAN)
    frm->showMaximized();
#else
    frm->show();
#endif

    frm->Init();
    if (frm->exec() == QDialog::Accepted)
        ui->txtReceiver->setText(frm->GetSelectedNumber());
    delete frm;
}

void SmsWindow::SendSlot()
{
    if (!ui->btnSend->isEnabled())
        return;

    if (ui->txtReceiver->text().isEmpty()){
        CommonStatic::ErrorMessage(this, tr("Inserire un destinatario"));
        return;
    }

    if (ui->txtMessage->toPlainText().isEmpty()){
        CommonStatic::ErrorMessage(this, tr("Inserire un messaggio"));
        return;
    }

    bool send = true;
    QString error;
    QByteArray captchaImage;
    QString captcha;

    ui->btnContacts->setEnabled(false);
    ui->btnSend->setEnabled(false);
    ui->txtReceiver->setEnabled(false);
    ui->txtMessage->setEnabled(false);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#else
    QMovie* waitIcon = new QMovie(":/images/wait.gif");
    waitIcon->setScaledSize(QSize(40, 40));
    ui->lblWait->setMovie(waitIcon);
    waitIcon->start();
#endif

    if (!m_Vodafone->IsLoggedIn()){
        if (!m_Vodafone->Login()){
            CommonStatic::ErrorMessage(this, tr("Login fallito"));
            send = false;
        }
    }

    if (send){
        if (m_Vodafone->PrepareMessage(ui->txtReceiver->text(), ui->txtMessage->toPlainText(), &captchaImage, &error)){
            if (captchaImage.count() > 0){
                QPixmap* pixmap = new QPixmap();
                pixmap->loadFromData(captchaImage);
#ifdef Q_OS_SYMBIAN
    *pixmap = pixmap->scaled(pixmap->width() * 2, pixmap->height() * 2);
#else
    *pixmap = pixmap->scaled(pixmap->width() * 3, pixmap->height() * 3);
#endif

                CaptchaWindow* frm = new CaptchaWindow(this, pixmap);
#ifdef Q_OS_SYMBIAN
    frm->showMaximized();
#endif
                if (frm->exec() == QDialog::Accepted)
                    captcha = frm->GetCaptcha();
                delete frm;
                delete pixmap;                
            }
            if (captchaImage.count() == 0 || !captcha.isEmpty()){
                if (m_Vodafone->SendMessage(ui->txtReceiver->text(), ui->txtMessage->toPlainText(), captcha, &error)){
                    StopWait();
                    CommonStatic::InfoMessage(this, tr("SMS inviato"));
                    SaveMsgToSentFolder(ui->txtReceiver->text(), ui->txtMessage->toPlainText());

                    if (!m_Settings->SmsDate.isValid() || m_Settings->SmsDate != QDate::currentDate()){
                        m_Settings->SmsDate = QDate::currentDate();
                        m_Settings->SmsSent = 0;
                    }
                    m_Settings->SmsSent++;

                    accept();
                }else{
                    StopWait();
                    CommonStatic::ErrorMessage(this, error.isEmpty() ? tr("Captcha errato") : error);
                }
            }
            captchaImage.clear();
        }else{
            StopWait();
            CommonStatic::ErrorMessage(this, error.isEmpty() ? tr("Invio fallito") : error);
        }
    }

    StopWait();
#ifndef Q_WS_MAEMO_5
    delete waitIcon;
#endif
    ui->btnContacts->setEnabled(true);
    ui->btnSend->setEnabled(true);
    ui->txtReceiver->setEnabled(true);
    ui->txtMessage->setEnabled(true);
}

void SmsWindow::StopWait()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#else
    ui->lblWait->setMovie(NULL);
    ui->lblWait->setPixmap(NULL);
#endif
}

void SmsWindow::CancelSlot()
{
    reject();
}

void SmsWindow::MessageChangedSlot()
{
    int chars = ui->txtMessage->toPlainText().length();
    ui->lblCharacters->setText(QString("%1/360").arg(chars));
    if (chars > 360)
        ui->lblCharacters->setStyleSheet("color: red;");
    else
        ui->lblCharacters->setStyleSheet("color: black;");
}

bool SmsWindow::SaveMsgToSentFolder(QString number, QString message)
{
#ifdef Q_WS_MAEMO_5
    RTComEl* el = rtcom_el_new();
    bool result = false;
    QString partNumber = number.right(7);

    RTComElEvent* ev = rtcom_el_event_new();

    RTCOM_EL_EVENT_SET_FIELD(ev, service, g_strdup(QString("RTCOM_EL_SERVICE_SMS").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, event_type, g_strdup(QString("RTCOM_EL_EVENTTYPE_SMS_OUTBOUND").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, start_time, time(NULL));
    RTCOM_EL_EVENT_SET_FIELD(ev, end_time, time(NULL));
    RTCOM_EL_EVENT_SET_FIELD(ev, is_read, TRUE);
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_ebook_uid, 0);
    RTCOM_EL_EVENT_SET_FIELD(ev, local_uid, g_strdup(QString("ring/tel/ring").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, local_name, g_strdup(QString("<SelfHandle>").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_uid, number.toLatin1().data());
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_name, g_strdup(QString("").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, group_uid, g_strdup(partNumber.toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, free_text, g_strdup(message.toUtf8().data()));

    int id = rtcom_el_add_event(el, ev, NULL);
    if (id >= 0){
        QUuid uid = QUuid::createUuid();
        rtcom_el_add_header(el, id, QString("message-token").toLatin1().data(), uid.toString().toLatin1().data(), NULL);
        result = true;
    }

    rtcom_el_event_free(ev);
    g_object_unref(el);

    return result;
#else
    return true;
#endif
}
