/*   VLC-REMOTE for MAEMO 5
 *   Copyright (C) 2010 Schutz Sacha <istdasklar@gmail.com>, Dru Moore <usr@dru-id.co.uk>, Yann Nave <yannux@onbebop.net>
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   or (at your option) any later version, as published by the Free
 *   Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "playlistmainwindow.h"
#include "ui_playlistmainwindow.h"
#include <QPushButton>
#include <QSettings>
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
#include <QMaemo5InformationBox>
#endif
#include "configdialog.h"
#include "aboutdialog.h"
#include "appsettings.h"
//#include "vlcstatus.h"

PlayListMainWindow::PlayListMainWindow(QWidget *parent) :
        QMainWindow(parent),
        ui(new Ui::PlayListMainWindow)
{

    ui->setupUi(this);
    mTimer = new QTimer(this);
    setWindowTitle("Vlc remote");

    mCurrentDepth = 0;
    mCurrentVlcIndex = 0;


    mNetManager = new QNetworkAccessManager(this);

    mContents = new QList<VlcPlayListElementSimple>();

    ui->listWidget->setTextElideMode(Qt::ElideLeft);
    ui->listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    ui->playButton->setIcon(QIcon::fromTheme("camera_playback"));
    ui->clearButton->setIcon(QIcon::fromTheme("general_delete"));
    ui->shuffleButton->setIcon(QIcon::fromTheme("mediaplayer_default_shuffle"));
    ui->loopButton->setIcon(QIcon::fromTheme("general_refresh"));
    ui->repeatButton->setIcon(QIcon::fromTheme("general_redo"));
    ui->removeButton->setIcon(QIcon::fromTheme("general_close"));

    ui->clearButton->setDisabled(false);
    ui->shuffleButton->setDisabled(false);
    ui->loopButton->setDisabled(false);
    ui->repeatButton->setDisabled(false);
    ui->removeButton->setDisabled(true);
    ui->playButton->setDisabled(true);

    connect(ui->listWidget,SIGNAL(doubleClicked(QModelIndex)),this,SLOT(onPlay()));
    connect(ui->playButton,SIGNAL(clicked()),this,SLOT(onPlay()));
    connect(ui->removeButton,SIGNAL(clicked()),this,SLOT(onRemove()));
    connect(ui->repeatButton,SIGNAL(clicked()),this,SLOT(onRepeat()));
    connect(ui->loopButton,SIGNAL(clicked()),this,SLOT(onLoop()));
    connect(ui->shuffleButton,SIGNAL(clicked()),this,SLOT(onShuffle()));
    connect(ui->clearButton,SIGNAL(clicked()),this,SLOT(onClear()));
    connect(ui->listWidget, SIGNAL(itemSelectionChanged()), this, SLOT(onListSelectionChanged()));

    init();

}
void PlayListMainWindow::init()  // CALL WHEN CONFIG CHANGES
{
    mIp = AppSettings::getCurrentIp(); // AccountDialog::currentIp();
}
void PlayListMainWindow::showPlayList()  // CALL WHEN SHOWN
{
    requestPlayList();
}

PlayListMainWindow::~PlayListMainWindow()
{
    delete ui;
}

void PlayListMainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PlayListMainWindow::onListSelectionChanged() {
    QList<QListWidgetItem *> items = ui->listWidget->selectedItems();
    if (0 < items.count()) {
        // TODO VLC doesn't seem to support removing nodes!
        // Playing nodes works fine, however.
        // should check to see if it is a leaf or not and disable the remove button
        mCurrentVlcIndex = items.at(0)->type() - LIST_ITEM_TYPE_OFFSET; // Qt reserves types up to 1000, we use an offset beyond that for index tracking. May prove to be too hacky!
        mCurrentElement = getElementFromId(mCurrentVlcIndex);
        if (0 == QString::compare("node", mCurrentElement.type)) {
            ui->removeButton->setDisabled(true);
            ui->playButton->setDisabled(false);
        }
        else {
            ui->removeButton->setDisabled(false);
            ui->playButton->setDisabled(false);
        }
    }
    else {
        mCurrentVlcIndex = 0;
        ui->removeButton->setDisabled(true);
        ui->playButton->setDisabled(true);
    }
}

void PlayListMainWindow::onRemove() {
    if (0 < this->mCurrentVlcIndex) {
        mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_delete&id=" + QString::number(this->mCurrentVlcIndex))));
        //connect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(requestPlayList()));
        // don't refresh just clear the item!
        delete ui->listWidget->takeItem(ui->listWidget->currentRow());
    }
}
void PlayListMainWindow::onPlay() {
    if (0 < this->mCurrentVlcIndex) {
        mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_play&id=" + QString::number(this->mCurrentVlcIndex))));
    }
}
void PlayListMainWindow::onRepeat() {
    mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_repeat")));
}
void PlayListMainWindow::onLoop() {
    mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_loop")));
}
void PlayListMainWindow::onShuffle() {
    mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_random")));
}
void PlayListMainWindow::onClear() {
    mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_empty")));
    mContents->clear();
    ui->listWidget->clear();
    //connect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(requestPlayList()));
}
void PlayListMainWindow::requestPlayList() {
  mResponse.clear();
  ui->removeButton->setDisabled(true);
  ui->playButton->setDisabled(true);
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
  QNetworkReply * reply =  mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/playlist.xml")));
  disconnect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(requestPlayList()));
  connect(reply,SIGNAL(readyRead()),this,SLOT(readReady()));
  connect(reply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(error(QNetworkReply::NetworkError)));
  connect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));
}
void PlayListMainWindow::error(QNetworkReply::NetworkError code) {
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
  qDebug() << code;
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  QMaemo5InformationBox::information(this, tr("Playlist could not be retrieved."), QMaemo5InformationBox::DefaultTimeout);
#endif
}
void PlayListMainWindow::readReady() {
  QNetworkReply * reply = qobject_cast<QNetworkReply*>(sender());
  // append to buffer
  mResponse += reply->readAll();
}
void PlayListMainWindow::finished(QNetworkReply * reply) {
  // now we can call parseXmlList to process the full buffers
  this->parseXmlPlayList();
  // only interested in finished signals
  disconnect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
  delete reply;
}

void PlayListMainWindow::parseXmlPlayList() {
  mContents->clear();
  QDomDocument doc;
  doc.setContent(this->mResponse);
  QDomElement docElem = doc.documentElement();
  QDomNodeList nodes = docElem.elementsByTagName("node");
  int depth = 0;

  int currentLeafId = 0;
  bool hasArt = false;
  QString extension = "";

  int ct = nodes.count();
  for (int idx = 0; idx < ct; ++idx) {
    QDomNode node = nodes.at(idx);
    QString current = "";
    //QString name = node.attributes().namedItem("name").nodeValue().replace("\\\\", "\\");
    int id = node.attributes().namedItem("id").nodeValue().toInt();
    if (4 > id && 0 == QString::compare(node.attributes().namedItem("ro").nodeValue(), "ro")) {
      // got the main playlist, let's build it up
      if (node.hasChildNodes()) {
        QDomNodeList leafs = node.childNodes();
        int leafct = leafs.count();
        if (0 < leafct) {
          depth = 1;
          for (int jdx = 0; jdx < leafct; ++jdx) {
            QDomNode leaf = leafs.at(jdx);
            parsePlayListItem(&leaf, &extension, &hasArt, &currentLeafId, 1);
          }
        }
      }

    }
  }

  mResponse.clear();

  emit this->idUpdated(currentLeafId, hasArt, extension);
  this->updateList();


}


int PlayListMainWindow::parsePlayListItem(QDomNode *node, QString *extension, bool *hasArt, int *currentLeafId, int depth) {
    int leafCt = 0;
    if (NULL != node) {
        QString current;
        VlcPlayListElementSimple* el = new VlcPlayListElementSimple();
        el->id = node->attributes().namedItem("id").nodeValue().toInt();
        el->path = node->attributes().namedItem("uri").nodeValue();
        el->name = node->attributes().namedItem("name").nodeValue().replace("\\\\", "\\");
        current = node->attributes().namedItem("current").nodeValue();
        el->playing = (0 < current.length());
        el->depth = depth;
        if (0 != QString::compare(node->nodeName(), "node")) {
            el->type = "leaf";
            el->extension = getExtension(el->path, NULL);
            if (el->playing) {
                *currentLeafId = el->id;
                QString art = node->toElement().namedItem("art_url").toElement().text();
                *hasArt = (!art.isNull() && !art.isEmpty());
                *extension = getExtension(el->path, NULL);
            }
            QString title = node->toElement().namedItem("title").toElement().text();
            QString artist = node->toElement().namedItem("artist").toElement().text();
            QString album = node->toElement().namedItem("album").toElement().text();
            if (0 < artist.length() || 0 < album.length()) {
                el->name = "" + title + "\n" + artist + " - " + album;
            }
            this->mContents->append(*el);
            ++leafCt;
            delete el;
        }
        else {
            el->type = "node";
            el->extension = getExtension(el->path, NULL);
            // empty nodes appear in the playlist when they can't be played!
            if (node->hasChildNodes()) {
                this->mContents->append(*el);
                ++leafCt;
            }
            delete el;
            // now parse the child nodes as leaf.
            if (node->hasChildNodes()) {
                QDomNodeList items = node->childNodes();
                int itemct = items.count();
                int added = 0;
                if (0 < itemct) {
                  ++depth;
                  for (int kdx = 0; kdx < itemct; ++kdx) {
                    QDomNode item = items.at(kdx);
                    added += parsePlayListItem(&item, extension, hasArt, currentLeafId, depth);
                  }
                  if (1 > added) {
                      // should remove the last PlayListElement
                      this->mContents->removeLast();
                      --leafCt;
                  }
                  else {
                      leafCt += added;
                  }
                }
            }
        }
    }
    return leafCt;
}

QString PlayListMainWindow::getExtension(QString path, QString extension) {
    // return extension if exists
    if (!extension.isNull() && !extension.isEmpty()) return extension;
    // return blank if no path
    if (path.isNull() || path.isEmpty()) return "";
    // otherwise extract the extension
    int dot_pos = path.lastIndexOf('.');
    if (0 < dot_pos) {
        return path.right(path.length() - (dot_pos + 1));
    }
    else { // no dot
        return "";
    }
}

VlcPlayListElementSimple PlayListMainWindow::getElementFromId(int id) {
  //if (0 != QString::compare("", text)) {
    for (int idx = 0; idx < mContents->count(); ++idx) {
      if (0 < id && (id == mContents->at(idx).id)) {
        return mContents->at(idx);
      }
    }
    //}
    return *(new VlcPlayListElementSimple());
}

void PlayListMainWindow::updateList() {
  ui->listWidget->clear();
  int ct = this->mContents->count();
  if (0 < ct) {
    QIcon icon_audio  = QIcon::fromTheme("general_audio_file");
    QIcon icon_video  = QIcon::fromTheme("general_video_file");
    QIcon icon_image  = QIcon::fromTheme("general_image");
    QIcon icon_flash  = QIcon::fromTheme("filemanager_flash_file");
    QIcon icon_media  = QIcon::fromTheme("filemanager_media_folder");
    QIcon icon_real   = QIcon::fromTheme("filemanager_real_music");
    QIcon icon_playl  = QIcon::fromTheme("filemanager_playlist");
    QIcon icon_unknown= QIcon::fromTheme("filemanager_unknown_file");
    for (int idx = 0; idx < ct; ++idx) {
      VlcPlayListElementSimple el = mContents->at(idx);
      QListWidgetItem* item;
      if (0 == QString::compare("node", el.type)) {
        item = new QListWidgetItem(icon_media, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id);
      }
      else {
          if ( 0 == QString::compare(el.extension, "jpg")  ||
               0 == QString::compare(el.extension, "jpeg") ||
               0 == QString::compare(el.extension, "gif")  ||
               0 == QString::compare(el.extension, "png")  ||
               0 == QString::compare(el.extension, "bmp")  ) {
              item = new QListWidgetItem(icon_image, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .jpg, .jpeg, .gif, .png, .bmp
          }
          else if ( 0 == QString::compare(el.extension, "mp3")  ||
                    0 == QString::compare(el.extension, "m4a")  ||
                    0 == QString::compare(el.extension, "ogg")  ||
                    0 == QString::compare(el.extension, "oga")  ||
                    0 == QString::compare(el.extension, "wav")  ||
                    0 == QString::compare(el.extension, "bmu")  ||
                    0 == QString::compare(el.extension, "wma")  ||
                    0 == QString::compare(el.extension, "flac")  ) {
              item = new QListWidgetItem(icon_audio, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .mp3, .m4a, .ogg, .oga, .wav, .bmu, .wma, .flac
          }
          else if ( 0 == QString::compare(el.extension, "m3u")  ||
                    0 == QString::compare(el.extension, "wpl")  ||
                    0 == QString::compare(el.extension, "pls")  ||
                    0 == QString::compare(el.extension, "asx")  ||
                    0 == QString::compare(el.extension, "xspf") ||
                    0 == QString::compare(el.extension, "cmml")  ) {
              item = new QListWidgetItem(icon_playl, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .m3u, .wpl, .pls, .asx, .xspf, .cmml
          }
          else if ( 0 == QString::compare(el.extension, "avi")  ||
                    0 == QString::compare(el.extension, "mpeg") ||
                    0 == QString::compare(el.extension, "mpg")  ||
                    0 == QString::compare(el.extension, "mov")  ||
                    0 == QString::compare(el.extension, "mp4")  ||
                    0 == QString::compare(el.extension, "m4v")  ||
                    0 == QString::compare(el.extension, "wmv")  ||
                    0 == QString::compare(el.extension, "mkv")  ||
                    0 == QString::compare(el.extension, "ogv")  ) {
              item = new QListWidgetItem(icon_video, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .avi, .mpg, .mpeg, .mov, .m4v, .mp4, .wmv, .mkv, .ogv
          }
          else if ( 0 == QString::compare(el.extension, "rm")  ||
                    0 == QString::compare(el.extension, "ra")  ||
                    0 == QString::compare(el.extension, "ram")  ) {
              item = new QListWidgetItem(icon_real, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .ram, 'rm, 'ra
          }
          else if ( 0 == QString::compare(el.extension, "flv")  ) {
              item = new QListWidgetItem(icon_flash, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .flv
          }
          //else if ( 0 == QString::compare(el.extension, "")  ) {
          //    item = new QListWidgetItem(icon_unknown, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id); // .flv
          //}
          else {
              if (el.name.contains("Flash")) {
                  item = new QListWidgetItem(icon_flash, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id);
              }
              else {
                  item = new QListWidgetItem(icon_media, el.name, ui->listWidget, LIST_ITEM_TYPE_OFFSET + el.id);
              }
          }
      }
      item->setSelected(el.playing);
      ui->listWidget->addItem(item);
      if (el.playing) {
          ui->listWidget->scrollToItem(item, QAbstractItemView::PositionAtCenter);
      }
    }
  }
}
void PlayListMainWindow::updateUiWithCurrentStatus(VlcStatus * status) {
    ui->loopButton->setChecked(status->loop);
    ui->repeatButton->setChecked(status->repeat);
    ui->shuffleButton->setChecked(status->random);
}
