  /*   VLC-REMOTE for MAEMO 5
  *   Copyright (C) 2010 Schutz Sacha <istdasklar@gmail.com>, Dru Moore <usr@dru-id.co.uk>, Yann Nave <yannux@onbebop.net>
  *   This program is free software; you can redistribute it and/or modify
  *   it under the terms of the GNU General Public License version 2,
  *   or (at your option) any later version, as published by the Free
  *   Software Foundation
  *
  *   This program is distributed in the hope that it will be useful,
  *   but WITHOUT ANY WARRANTY; without even the implied warranty of
  *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *   GNU General Public License for more details
  *
  *   You should have received a copy of the GNU General Public
  *   License along with this program; if not, write to the
  *   Free Software Foundation, Inc.,
  *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
  */
  //#include <QDebug>
  #include <QTime>
  #include <QtGui>
  #include "playermainwindow.h"
  #include "ui_playermainwindow.h"
  #include "configdialog.h"
  #include "aboutdialog.h"
  #include "accountdialog.h"
  #include "settingsdialog.h"
  #include "appsettings.h"
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  #include <QMaemo5InformationBox>
  #include <mce/dbus-names.h>
  #include <mce/mode-names.h>
  #include <QDBusConnection>
  #include <QDBusMessage>
  #include <QDBusInterface>
#endif
#include <QMessageBox>

#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
QDBusConnection PlayerMainWindow::dBusConnection = QDBusConnection::systemBus();
QDBusInterface* PlayerMainWindow::dBusInterface = new QDBusInterface(MCE_SERVICE, MCE_REQUEST_PATH,
                                   MCE_REQUEST_IF, dBusConnection);
#endif

  PlayerMainWindow::PlayerMainWindow(QWidget *parent) :
	  QMainWindow(parent),
	  ui(new Ui::PlayerMainWindow)
  {
      ui->setupUi(this);
      setWindowTitle("Vlc remote");

      mIsFirstStatusCall = true;

      mTimer = new QTimer(this);
      mNetManager = new QNetworkAccessManager(this);
      mPlayListMainWindow = new PlayListMainWindow;
      mBrowserMainWindow = new BrowseMainWindow;
      mFavouritesMainWindow = new FavouritesMainWindow;

      mVolume = 100;
      mMuted = false;

      mIsLandscape = true;

      ui->playlistButton->setIcon(QIcon::fromTheme("notes_bullets"));
      ui->browseButton->setIcon(QIcon::fromTheme("filemanager_media_folder"));

      ui->previousButton->setIcon(QIcon::fromTheme("pdf_viewer_first_page"));
      ui->nextButton->setIcon(QIcon::fromTheme("pdf_viewer_last_page"));
      ui->playpauseButton->setIcon(QIcon::fromTheme("camera_playback"));
      ui->stopButton->setIcon(QIcon::fromTheme("camera_video_stop"));
      ui->fullscreenButton->setIcon(QIcon::fromTheme("general_fullsize"));
      ui->volDown->setIcon(QIcon::fromTheme("statusarea_volumelevel1"));
      ui->volUp->setIcon(QIcon::fromTheme("statusarea_volumelevel4"));
      ui->volMute->setIcon(QIcon::fromTheme("statusarea_volume_mute"));

      ui->labelArtPortrait->setVisible(false);
      ui->labelArtLandscape->setVisible(false);

      ui->labelTitle->setTextFormat(Qt::RichText);
      ui->labelArtist->setTextFormat(Qt::RichText);
      ui->labelAlbum->setTextFormat(Qt::RichText);


  #if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
      // Handle screen state on / off
      dBusConnection.connect(MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                 MCE_DISPLAY_SIG, this, SLOT(displayStateChanged(const QDBusMessage &)));

      mPlayListMainWindow->setParent(this);
      mPlayListMainWindow->setAttribute(Qt::WA_Maemo5StackedWindow, true);
      setAttribute(Qt::WA_Maemo5StackedWindow, true);
      mPlayListMainWindow->setWindowFlags(mPlayListMainWindow->windowFlags() | Qt::Window);

      mBrowserMainWindow->setParent(this);
      mBrowserMainWindow->setAttribute(Qt::WA_Maemo5StackedWindow, true);
      setAttribute(Qt::WA_Maemo5StackedWindow, true);
      mBrowserMainWindow->setWindowFlags(mBrowserMainWindow->windowFlags() | Qt::Window);

      mFavouritesMainWindow->setParent(this);
      mFavouritesMainWindow->setAttribute(Qt::WA_Maemo5StackedWindow, true);
      setAttribute(Qt::WA_Maemo5StackedWindow, true);
      mFavouritesMainWindow->setWindowFlags(mFavouritesMainWindow->windowFlags() | Qt::Window);

      connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

  #endif

      connect(mTimer,SIGNAL(timeout()),this,SLOT(askStatus()));
      connect(ui->actionSettings,SIGNAL(triggered()),this,SLOT(showSettings()));
      connect(ui->actionConfiguration,SIGNAL(triggered()),this,SLOT(showConfig()));
      connect(ui->actionAbout,SIGNAL(triggered()),this,SLOT(showAbout()));
      connect(ui->actionFavourites,SIGNAL(triggered()),this,SLOT(showFavourites()));
      //connect(ui->actionPortrait,SIGNAL(triggered()),this,SLOT(setPortrait()));
      //connect(ui->actionLandscape,SIGNAL(triggered()),this,SLOT(setLandscape()));
      //connect(ui->actionAutoRotate,SIGNAL(triggered()),this,SLOT(setAutoRotate()));
      connect(ui->playlistButton,SIGNAL(clicked()),mPlayListMainWindow,SLOT(show()));
      connect(ui->playlistButton,SIGNAL(clicked()),mPlayListMainWindow,SLOT(showPlayList()));
      connect(ui->browseButton,SIGNAL(clicked()),mBrowserMainWindow,SLOT(show()));
      connect(ui->browseButton,SIGNAL(clicked()),mBrowserMainWindow,SLOT(showCurrentDirectory()));

      connect(ui->playpauseButton,SIGNAL(clicked()),this,SLOT(playpause()));
      connect(ui->stopButton,SIGNAL(clicked()),this,SLOT(stop()));
      connect(ui->previousButton,SIGNAL(clicked()),this,SLOT(previous()));
      connect(ui->nextButton,SIGNAL(clicked()),this,SLOT(next()));
      connect(ui->fullscreenButton,SIGNAL(clicked()),this,SLOT(fullscreen()));
      connect(ui->volUp,SIGNAL(clicked()),this,SLOT(volUp()));
      connect(ui->volDown,SIGNAL(clicked()),this,SLOT(volDown()));
      connect(ui->volMute,SIGNAL(clicked()),this,SLOT(volMute()));
      connect(ui->slider,SIGNAL(sliderMoved(int)),this,SLOT(slide(int)));

      connect(mPlayListMainWindow, SIGNAL(idUpdated(int,bool,QString)), this, SLOT(playlistIdUpdated(int, bool, QString)));
      connect(mBrowserMainWindow, SIGNAL(showFavouritesWindow()), this, SLOT(showFavourites()));

      connect(mFavouritesMainWindow, SIGNAL(browseDirectory(QString)), mFavouritesMainWindow, SLOT(close()));
      connect(mFavouritesMainWindow, SIGNAL(browseDirectory(QString)), mBrowserMainWindow, SLOT(browseDirectory(QString)));
      connect(mFavouritesMainWindow, SIGNAL(closeSignal()), mBrowserMainWindow, SLOT(show()));


      // check if last used connection is still valid or showConfig

      // check for network
      if (AppSettings::isConnected()) {
          QSettings settings;
          QString last_ip = AppSettings::getCurrentIp(); // AccountDialog::currentIp();
          if (!last_ip.isNull() && !last_ip.isEmpty()) {
              QTcpSocket * socket = new QTcpSocket;
              if(last_ip.contains(":"))
              {
                  QStringList hostSplit = last_ip.split(":");
                  QString ip   = hostSplit.at(0);
                  QString port = hostSplit.at(1);
                  socket->connectToHost(ip,port.toInt());
              }
              else {
                  socket->connectToHost(last_ip,8080);
              }
              if (!socket->waitForConnected(AppSettings::getConnectionTimeout())) {
                     showConfig();
                 }
              else {
                  mIp= last_ip;

                 mPlayListMainWindow->init();
                 mBrowserMainWindow->init();
                 mTimer->start(AppSettings::getStatusPollTimeout());
                 askStatus();
              }
              delete socket;
          }
          else {
            showConfig();
          }
      }
      else {
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
        QMaemo5InformationBox::information(this, tr("Network unavailable!"), QMaemo5InformationBox::DefaultTimeout);
#endif
          showConfig();
      }
  }
  

  PlayerMainWindow::~PlayerMainWindow()
  {
      delete ui;
  }

  void PlayerMainWindow::changeEvent(QEvent *e)
  {
      QMainWindow::changeEvent(e);
      switch (e->type()) {
      case QEvent::LanguageChange:
	  ui->retranslateUi(this);
	  break;
      default:
	  break;
      }
  }

#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
  void PlayerMainWindow::displayStateChanged(const QDBusMessage &message)
  {
      QString state = message.arguments().at(0).toString();
      if (!state.isEmpty()) {
          if (state == MCE_DISPLAY_ON_STRING) {
              mTimer->start(AppSettings::getStatusPollTimeout());
          }
          else if (state == MCE_DISPLAY_OFF_STRING) {
              mTimer->stop();
          }
      }
  }
#endif

  void PlayerMainWindow::showFavourites() {
      mFavouritesMainWindow->show();
      mFavouritesMainWindow->init();
  }

  void PlayerMainWindow::updateFromSettings()
  {
    #if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
    switch (AppSettings::getOrientation()) {
      case LANDSCAPE:
          this->setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
          break;
      case PORTRAIT:
          this->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
          break;
      case AUTO_ROTATE:
          this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
          break;
    }
    #endif
    if (AppSettings::getShowAlbumArt()) {
        this->mIsFirstStatusCall = true;
    }
    mTimer->start(AppSettings::getStatusPollTimeout());
  }

  void PlayerMainWindow::setPortrait()
  {
    #if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
      AppSettings::setOrientation(PORTRAIT);
    this->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    #endif
  }

  void PlayerMainWindow::setLandscape()
  {
    #if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
      AppSettings::setOrientation(LANDSCAPE);
    this->setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    #endif
  }

  void PlayerMainWindow::setAutoRotate()
  {
    #if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
      AppSettings::setOrientation(AUTO_ROTATE);
    this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    #endif
  }

  void PlayerMainWindow::orientationChanged() {
      QRect screenGeometry = QApplication::desktop()->screenGeometry();
      mIsLandscape = (screenGeometry.width() > screenGeometry.height());
      if (mHasImage) {
          if (mIsLandscape) {
              ui->labelArtPortrait->setVisible(false);
              ui->labelArtLandscape->setVisible(true);
          }
          else {
              ui->labelArtLandscape->setVisible(false);
              ui->labelArtPortrait->setVisible(true);
          }
      }
      else {
          ui->labelArtLandscape->setVisible(false);
          ui->labelArtPortrait->setVisible(false);
      }
  }

  void PlayerMainWindow::playpause()
  {
      // NB. There is no guarentee that our current state is the real current state.
      // This is due to the polling frequency and possibility of user interaction directly on the server.
      // Still this is probably better than nothing and our next real poll will set us straight again.
      if (PAUSED == mCurrentStatus.state) {
        mCurrentStatus.state = PLAYING;
        pause();
        updateUiWithCurrentStatus();
      }
      else if (PLAYING == mCurrentStatus.state) {
        mCurrentStatus.state = PAUSED;
        pause();
        updateUiWithCurrentStatus();
      }
      else {
        // could be STOP or UNKNOWN, either way there is no guarentee we will enter a playing state next.
        // So don't update the current state or UI
        // Ideally we would try to find a way to check the current state again but this could lead to an infinite loop!
        play();
      }
  }
  void PlayerMainWindow::play()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_play")));
  }
  void PlayerMainWindow::stop()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_stop")));
  }
  void PlayerMainWindow::pause()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_pause")));
  }
  void PlayerMainWindow::previous()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_previous")));
  }
  void PlayerMainWindow::next()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=pl_next")));
  }
  void PlayerMainWindow::fullscreen()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=fullscreen")));
  }
  void PlayerMainWindow::volUp()
  {
      QUrl url = QUrl("http://"+mIp+"/requests/status.xml?command=volume");
      url.addEncodedQueryItem(QByteArray("val"), QByteArray("%2B10"));
      mNetManager->get(QNetworkRequest(url));
  }
  void PlayerMainWindow::volDown()
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=volume&val=-10")));
  }
  void PlayerMainWindow::volMute()
  {
      this->mMuted = !this->mMuted;
      if (this->mMuted) {
	  mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=volume&val=0")));
      }
      else {
	  mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=volume&val="+QString::number(this->mVolume))));
      }
  }
  void PlayerMainWindow::slide(int value)
  {
      mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml?command=seek&val="+QString::number(value)+"%25")));
  }

  void PlayerMainWindow::showSettings()
  {
    SettingsDialog * dialog = new SettingsDialog(this);
    connect(dialog, SIGNAL(closeSignal()), this, SLOT(updateFromSettings()));
    dialog->exec();
  }
  void PlayerMainWindow::showConfig()
  {
      mTimer->stop();
      // check for network
      if (AppSettings::isConnected()) {
          AccountDialog * dialog = new AccountDialog(this);
          dialog->exec();

          mIp= AppSettings::getCurrentIp(); //AccountDialog::currentIp();

          mPlayListMainWindow->init();
          mBrowserMainWindow->init();
          mTimer->start(AppSettings::getStatusPollTimeout());
          askStatus();
      }
      else {
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
        QMaemo5InformationBox::information(this, tr("Network unavailable!"), QMaemo5InformationBox::DefaultTimeout);
#endif
        QTimer::singleShot(AppSettings::getRetryNetworkTimeout(), this, SLOT(showConfig()));
      }
  }
  void PlayerMainWindow::showAbout()
  {

      AboutDialog * dialog = new AboutDialog;
      dialog->exec();

  }

  void PlayerMainWindow::askStatus()
  {
      if (AppSettings::isConnected()) {
          QNetworkReply * reply =  mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/requests/status.xml")));
          connect(reply,SIGNAL(readyRead()),this,SLOT(parseXmlStatus()));
          connect(reply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(error(QNetworkReply::NetworkError)));
      }
      else {
        showConfig(); // this will handle stopping and restarting the timer.
      }
  }

  void PlayerMainWindow::parseXmlStatus()
  {
      QNetworkReply * reply = qobject_cast<QNetworkReply*>(sender());
      QDomDocument doc;
      doc.setContent(reply->readAll());
      delete reply;
      QDomElement docElem = doc.documentElement();
      VlcStatusState oldState = mCurrentStatus.state;
      // Get the raw values
      int volume = docElem.namedItem("volume").toElement().text().toInt();
      int length = docElem.namedItem("length").toElement().text().toInt();
      int time = docElem.namedItem("time").toElement().text().toInt();
      int position = docElem.namedItem("position").toElement().text().toInt();
      int random = docElem.namedItem("random").toElement().text().toInt();
      int loop = docElem.namedItem("loop").toElement().text().toInt();
      int repeat = docElem.namedItem("repeat").toElement().text().toInt();
      QString state = docElem.namedItem("state").toElement().text();
      QDomNode infoNode =  docElem.namedItem("information");
      QDomNode metaInfoNode =  infoNode.namedItem("meta-information");
      QString title = metaInfoNode.namedItem("title").toElement().text().replace("\\\\", "\\");
      // if it's a file style title fix it up
      if (40 < title.length()) {
          if (0 < title.lastIndexOf("\\")) {
              title = title.right(title.length() - (title.lastIndexOf("\\") + 1));
          }
          else if (0 < title.lastIndexOf("/")) {
              title = title.right(title.length() - (title.lastIndexOf("/") + 1));
          }
      }
      QString artist = metaInfoNode.namedItem("artist").toElement().text();
      QString album = metaInfoNode.namedItem("album").toElement().text();
      QString now_playing = metaInfoNode.namedItem("now_playing").toElement().text();
      QString art_url = metaInfoNode.namedItem("art_url").toElement().text();
      // Populate the current status structure
      // now would be a good time to work out if we are a new track / file or not.
      // key if we are going to look for album art later
      // for now we check length and title this will require further examination later
      if (!state.isNull() && !state.isEmpty()) {
          if (0 == QString::compare("playing", state, Qt::CaseInsensitive)) {
            mCurrentStatus.state = PLAYING;
          }
          else if (0 == QString::compare("paused", state, Qt::CaseInsensitive)) {
            mCurrentStatus.state = PAUSED;
          }
          else if (0 == QString::compare("stop", state, Qt::CaseInsensitive)) {
            mCurrentStatus.state = STOP;
          }
          else {
            mCurrentStatus.state = UNKNOWN;
            mIsFirstStatusCall = true;
          }
      }
      else {
          mCurrentStatus.state = UNKNOWN;
          mIsFirstStatusCall = true;
      }
      mCurrentStatus.newtrack = true;
      if (mIsFirstStatusCall) {
          mIsFirstStatusCall = false;
          mCurrentStatus.newtrack = true; // unneeded but self-documenting
      }
      else if (STOP == oldState && STOP != mCurrentStatus.state) {
          mCurrentStatus.newtrack = true; // unneeded but self-documenting
      }
      else {
          if ( (0 == mCurrentStatus.length || STOP == mCurrentStatus.state) // stopped or null
                  || // same track as current playing
               (mCurrentStatus.length == length && !mCurrentStatus.title.isNull() && !title.isNull() && 0 == QString::compare(mCurrentStatus.title, title)) ){
            mCurrentStatus.newtrack = false;
          }
      }
      mCurrentStatus.volume = volume;
      mCurrentStatus.length = length;
      mCurrentStatus.time = time;
      mCurrentStatus.position = position;
      mCurrentStatus.random = (1 == random);
      mCurrentStatus.loop = (1 == loop);
      mCurrentStatus.repeat = (1 == repeat);
      mCurrentStatus.title = title;
      mCurrentStatus.artist = artist;
      mCurrentStatus.album = album;
      mCurrentStatus.nowplaying = now_playing;
      mCurrentStatus.hasart = (!art_url.isNull() && !art_url.isEmpty());
      // What's our mute status?
      if (0 < mCurrentStatus.volume) {
          this->mVolume = mCurrentStatus.volume;
	  this->mMuted = false;
      }
      else {
	  this->mMuted = true;
      }
      // Update the UI
      updateUiWithCurrentStatus();

  }

  void PlayerMainWindow::updateUiWithCurrentStatus() {
      // position
      QTime timePosition(0,0,0) ;
      timePosition =  timePosition.addSecs(mCurrentStatus.time);

      ui->timeLabel->setText(timePosition.toString("h:mm:ss"));

      // duration
      if (0 < mCurrentStatus.length) {
          QTime timeDuration(0,0,0) ;
          timeDuration =  timeDuration.addSecs(mCurrentStatus.length);

          ui->durationLabel->setText(timeDuration.toString("h:mm:ss"));
      }
      else {
          ui->durationLabel->setText("0:00:00");
      }


      if (mCurrentStatus.position >= 0 && mCurrentStatus.position <= 100) {
          ui->slider->blockSignals(true);
          ui->slider->setValue(mCurrentStatus.position);
          ui->slider->blockSignals(false);
      }

      ui->labelTitle->setText(mCurrentStatus.title);
      ui->labelArtist->setText(mCurrentStatus.artist);
      ui->labelAlbum->setText(mCurrentStatus.album);

      if (PLAYING == mCurrentStatus.state) {
          ui->playpauseButton->setIcon(QIcon::fromTheme("camera_video_pause"));
      }
      else {
          ui->playpauseButton->setIcon(QIcon::fromTheme("camera_playback"));
      }

      if (STOP == mCurrentStatus.state || !AppSettings::getShowAlbumArt()) {
          ui->labelArtPortrait->setVisible(false);
          ui->labelArtLandscape->setVisible(false);
      }

      if (mCurrentStatus.newtrack && STOP != mCurrentStatus.state) {
          // potential actions:
          //   rebuild display layout
          //   retrieve album art
          if (AppSettings::getShowAlbumArt()) {
              mHasImage = false;
              QTimer::singleShot(AppSettings::getRetrieveArtTimeout(), mPlayListMainWindow, SLOT(requestPlayList()));
          }
      }
      // Update the buttons on the playlist window
      if (NULL != this->mPlayListMainWindow) {
        this->mPlayListMainWindow->updateUiWithCurrentStatus(& mCurrentStatus);
      }

  }
  void PlayerMainWindow::playlistIdUpdated(int id, bool hasart, QString extension) {
      Q_UNUSED(extension);
      if (hasart) {
          getCoverArt(id);
      }
      else {
          ui->labelArtLandscape->setVisible(false);
          ui->labelArtPortrait->setVisible(false);
          // could use a default graphic from extension here!
          // setCoverArtFromPixmap();
      }
  }
  void PlayerMainWindow::error(QNetworkReply::NetworkError code) {
      Q_UNUSED(code);
      //qDebug() << "Error Code: " << code;
  }
  void PlayerMainWindow::readReady() {
    QNetworkReply * reply = qobject_cast<QNetworkReply*>(sender());
    // append to buffer
    mResponse += reply->readAll();
  }
  void PlayerMainWindow::finished(QNetworkReply * reply) {
    // now we can call setCoverArt to process the full buffers
    this->setCoverArt(mResponse);
    // only interested in finished signals
    disconnect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));
    delete reply;
  }
  void PlayerMainWindow::getCoverArt(int id) {
    mResponse.clear();
    QNetworkReply * reply =  mNetManager->get(QNetworkRequest(QUrl("http://"+mIp+"/art?id=" + QString::number(id))));
    connect(reply,SIGNAL(readyRead()),this,SLOT(readReady()));
    connect(reply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(error(QNetworkReply::NetworkError)));
    connect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));

  }
  void PlayerMainWindow::setCoverArt(const QByteArray data) {
    QPixmap* image = new QPixmap();
    if (image->loadFromData(data)) {
        mHasImage = true;
        ui->labelArtLandscape->setPixmap(image->scaledToHeight(160, Qt::SmoothTransformation));
        ui->labelArtPortrait->setPixmap(image->scaledToHeight(310, Qt::SmoothTransformation));
        if (mIsLandscape) {
            ui->labelArtPortrait->setVisible(false);
            ui->labelArtLandscape->setVisible(true);
        }
        else {
            ui->labelArtLandscape->setVisible(false);
            ui->labelArtPortrait->setVisible(true);
        }
    }
    else {
        ui->labelArtPortrait->setVisible(false);
        ui->labelArtLandscape->setVisible(false);
    }
    delete image;
  }
  void PlayerMainWindow::setCoverArtFromPixmap(QPixmap image) {
    mHasImage = true;
    ui->labelArtLandscape->setPixmap(image.scaledToHeight(160, Qt::SmoothTransformation));
    ui->labelArtPortrait->setPixmap(image.scaledToHeight(310, Qt::SmoothTransformation));
    if (mIsLandscape) {
        ui->labelArtPortrait->setVisible(false);
        ui->labelArtLandscape->setVisible(true);
    }
    else {
        ui->labelArtLandscape->setVisible(false);
        ui->labelArtPortrait->setVisible(true);
    }
  }
  void PlayerMainWindow::closeEvent(QCloseEvent * event) {
      if (!AppSettings::getAlertOnClose() || PLAYING != mCurrentStatus.state) {
          event->accept();
      }
      else { // handle alert
          if (QMessageBox::Yes == QMessageBox::question(this
                                                       , tr("Really quit?")
                                                       , tr("You currently have media playing on your remote machine. Are you sure you wish to quit vlc-remote?")
                                                       , QMessageBox::Yes | QMessageBox::No
                                                       , QMessageBox::No)) {
            event->accept();
          }
          else {
            event->ignore();
          }
      }
  }

