/*
@version: 0.4
@author: Sudheer K. <scifi1947 at gmail.com>
@license: GNU General Public License
*/

#include "configwindow.h"
#include "ui_configwindow.h"
#include "gconfutility.h"
#include "dbusutility.h"
#include <QDebug>
#include <QMessageBox>

class ConfigWindowPrivate
{
public:
    ConfigWindowPrivate(ConfigWindow * p) :
            gconfUtility(new GConfUtility(p)),
            dbusUtility(new DbusUtility(p)),
            parent(p)
    {

    }

    ~ConfigWindowPrivate()
    {
        qDebug() << "VICaR: ConfigWindow Destructing";
    }

    GConfUtility * gconfUtility;
    DbusUtility * dbusUtility;
    ConfigWindow * const parent;
};

// ---------------------------------------------------------------------------

ConfigWindow::ConfigWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::ConfigWindow),
    d(new ConfigWindowPrivate(this))
{
    ui->setupUi(this);
    loadValues();
}

ConfigWindow::~ConfigWindow()
{
    delete ui;
}

void ConfigWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void ConfigWindow::closeEvent(QCloseEvent *e)
{
    if (verifyConfigData()){
        qDebug() << "Verification successful. Saving data to gconf";
        saveConfigData();
        e->accept();
    }
    else{
        e->ignore();
    }
}

void ConfigWindow::on_actionSave_triggered()
{
    if (verifyConfigData()){
        qDebug() << "Verification successful. Saving data to gconf";
        saveConfigData();
    }

}

void ConfigWindow::on_actionReset_triggered()
{
    ui->checkBoxIntlCallRedirEnabled->setChecked(false);
    ui->lineEditCallingCardNumber->clear();
    ui->lineEditCountryCodesToExclude->clear();
    ui->comboBoxDTMFFormat->clear();
    ui->lineEditDTMFPrefix->clear();
    ui->lineEditDTMFSuffix->clear();
    //ui->spinBoxDTMFDelay->setValue(1);
    qDebug() << "Settings reset";
    d->dbusUtility->displayNotification("VICaR: Settings reset.");
}

void ConfigWindow::loadValues(){

    bool isRoutingEnabled = d->gconfUtility->getGconfValueBoolean("routing_enabled");
    QString strCallingCardNumber = d->gconfUtility->getGconfValueString("calling_card_number");
    QString strNumbersToExclude = d->gconfUtility->getGconfValueString("numbers_to_exclude");
    QString strDTMFFormat = d->gconfUtility->getGconfValueString("dtmf_format");
    QString strDTMFPrefix = d->gconfUtility->getGconfValueString("dtmf_prefix");
    QString strDTMFSuffix = d->gconfUtility->getGconfValueString("dtmf_suffix");
    //int intDTMFDelay = d->gconfUtility->getGconfValueInteger("dtmf_delay");

    ui->checkBoxIntlCallRedirEnabled->setChecked(isRoutingEnabled);
    ui->lineEditCallingCardNumber->setText(strCallingCardNumber);
    ui->lineEditCountryCodesToExclude->setText(strNumbersToExclude);
    int intIndex = ui->comboBoxDTMFFormat->findText(strDTMFFormat);
    ui->comboBoxDTMFFormat->setCurrentIndex(intIndex);
    ui->lineEditDTMFPrefix->setText(strDTMFPrefix);    
    ui->lineEditDTMFSuffix->setText(strDTMFSuffix);
    //ui->spinBoxDTMFDelay->setValue(intDTMFDelay);

   //Accept numbers only for Calling Card Number field
    ui->lineEditCallingCardNumber->setValidator(new QRegExpValidator( QRegExp( "^-?\\d\\d*$"), this));    
    ui->lineEditDTMFPrefix->setValidator(new QRegExpValidator( QRegExp( "^[0-9p#\\*]*$"), this));
    ui->lineEditDTMFSuffix->setValidator(new QRegExpValidator( QRegExp( "^[0-9p#\\*]*$"), this));

    qDebug() << "Values loaded from GConf";

}

bool ConfigWindow::verifyConfigData(){
    //Verify whether user-input matches application requirements
    bool isRoutingEnabled = ui->checkBoxIntlCallRedirEnabled->isChecked();
    QString strCallingCardNumber = ui->lineEditCallingCardNumber->text();
    QString strNumbersToExclude = ui->lineEditCountryCodesToExclude->text();


    QString strMessage = QString("");

    if (isRoutingEnabled){
        //Call Routing is checked. Now validate other values
        if (strCallingCardNumber.isEmpty()){
            strMessage.append("Enter a calling card number\n");
            ui->lineEditCallingCardNumber->setFocus();
        }

        if (strCallingCardNumber.startsWith("+")||strCallingCardNumber.startsWith("00")){
            strMessage.append("Calling card number must be a local number. \nPlease remove the international dialing code.\n");
            ui->lineEditCallingCardNumber->setFocus();
        }

    }

    if (!strMessage.isEmpty()){
        QMessageBox::warning(this,"Invalid Data",strMessage);
        return false;
    }
    else{
        return true;
    }
}

void ConfigWindow::saveConfigData(){

        bool isRoutingEnabled = ui->checkBoxIntlCallRedirEnabled->isChecked();
        QString strCallingCardNumber = ui->lineEditCallingCardNumber->text();
        QString strNumbersToExclude = ui->lineEditCountryCodesToExclude->text();

        QString strDTMFFormat = ui->comboBoxDTMFFormat->currentText();
        QString strDTMFPrefix = ui->lineEditDTMFPrefix->text();
        QString strDTMFSuffix = ui->lineEditDTMFSuffix->text();
       // int intDTMFDelay = ui->spinBoxDTMFDelay->value();


        d->gconfUtility->setGconfValueBoolean("routing_enabled",isRoutingEnabled);

        d->gconfUtility->setGconfValueString("calling_card_number",strCallingCardNumber);

        if (!strNumbersToExclude.isEmpty()){
            strNumbersToExclude = strNumbersToExclude.remove(" ");
            strNumbersToExclude = strNumbersToExclude.remove("\t");
            strNumbersToExclude = strNumbersToExclude.replace("\n",",");
        }

        d->gconfUtility->setGconfValueString("numbers_to_exclude",strNumbersToExclude);
        //d->gconfUtility->setGconfValueInteger("dtmf_delay",intDTMFDelay);
        d->gconfUtility->setGconfValueString("dtmf_format",strDTMFFormat);
        d->gconfUtility->setGconfValueString("dtmf_prefix",strDTMFPrefix.trimmed());
        d->gconfUtility->setGconfValueString("dtmf_suffix",strDTMFSuffix);


        qDebug() << "Settings updated";
        d->dbusUtility->displayNotification("VICaR: Settings updated.");
}

