/*
@version: 0.1
@author: Sudheer K. <scifi.guy@hotmail.com>
@license: GNU General Public License
*/

#include "configwidget.h"
#include "ui_configwidget.h"
#include "gconfutility.h"
#include "dbusutility.h"
#include <QDebug>
#include <QMessageBox>

ConfigWidget::ConfigWidget(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ConfigWidget)
{
    ui->setupUi(this);
    gconfUtility = new GConfUtility();
    loadValues();
}

ConfigWidget::~ConfigWidget()
{
    delete ui;
    delete gconfUtility;
    gconfUtility = 0;
}

void ConfigWidget::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void ConfigWidget::on_pushButtonSave_clicked()
{
    if (verifyConfigData()){
        qDebug() << "Verification successful. Saving data to gconf";
        saveConfigData();
    }
}

void ConfigWidget::on_pushButtonReset_clicked()
{
    ui->checkBoxIntlCallRedirEnabled->setChecked(false);    
    ui->lineEditCallingCardNumber->clear();
    ui->plainTextEditCodesToExclude->clear();
    ui->comboBoxDTMFFormat->clear();
    ui->spinBoxDTMFDelay->setValue(1);
}

void ConfigWidget::loadValues(){

    bool isRoutingEnabled = gconfUtility->getGconfValueBoolean((char *)"routing_enabled");
    char *strCallingCardNumber = gconfUtility->getGconfValueString((char *)"calling_card_number");
    char *strNumbersToExclude = gconfUtility->getGconfValueString((char *)"numbers_to_exclude");
    char *strDTMFFormat = gconfUtility->getGconfValueString((char *)"dtmf_format");
    int intDTMFDelay = gconfUtility->getGconfValueInteger((char *)"dtmf_delay");

    QString qstrCallingCardNumber = QString(strCallingCardNumber);
    QString qstrNumbersToExclude = QString(strNumbersToExclude);
    QString qstrDTMFFormat = QString(strDTMFFormat);

    if (!qstrNumbersToExclude.isEmpty()){
       qstrNumbersToExclude = qstrNumbersToExclude.replace(",","\n");
    }

    ui->checkBoxIntlCallRedirEnabled->setChecked(isRoutingEnabled);
    ui->lineEditCallingCardNumber->setText(qstrCallingCardNumber);
    ui->plainTextEditCodesToExclude->setPlainText(qstrNumbersToExclude);
    int intIndex = ui->comboBoxDTMFFormat->findText(qstrDTMFFormat);
    ui->comboBoxDTMFFormat->setCurrentIndex(intIndex);
    ui->spinBoxDTMFDelay->setValue(intDTMFDelay);

   //Accept numbers only for Calling Card Number field
    ui->lineEditCallingCardNumber->setValidator(new QRegExpValidator( QRegExp( "^-?\\d\\d*$"), this));

    qDebug() << "Values loaded from GConf";

}

bool ConfigWidget::verifyConfigData(){
    //Verify whether user-input matches application requirements
    bool isRoutingEnabled = ui->checkBoxIntlCallRedirEnabled->isChecked();
    QString qstrCallingCardNumber = ui->lineEditCallingCardNumber->text();
    QString qstrNumbersToExclude = ui->plainTextEditCodesToExclude->toPlainText();


    QString strMessage = QString("");

    if (isRoutingEnabled){
        //Call Routing is checked. Now validate other values
        if (qstrCallingCardNumber.isEmpty()){
            strMessage.append("Enter a calling card number\n");
            ui->lineEditCallingCardNumber->setFocus();
        }

        if (qstrCallingCardNumber.startsWith("+")||qstrCallingCardNumber.startsWith("00")){
            strMessage.append("Calling card number must be a local number. \nPlease remove the international dialing code.\n");
            ui->lineEditCallingCardNumber->setFocus();
        }

    }

    if (!strMessage.isEmpty()){
        QMessageBox::warning(this,"Invalid Data",strMessage);
        return false;
    }
    else{
        return true;
    }
}

void ConfigWidget::saveConfigData(){

        bool isRoutingEnabled = ui->checkBoxIntlCallRedirEnabled->isChecked();
        QString qstrCallingCardNumber = ui->lineEditCallingCardNumber->text();
        QString qstrNumbersToExclude = ui->plainTextEditCodesToExclude->toPlainText();

        QString qstrDTMFFormat = ui->comboBoxDTMFFormat->currentText();
        int intDTMFDelay = ui->spinBoxDTMFDelay->value();


        gconfUtility->setGconfValueBoolean((char *)"routing_enabled",isRoutingEnabled);

        if (!qstrCallingCardNumber.isEmpty()){
            gconfUtility->setGconfValueString((char *)"calling_card_number",qstrCallingCardNumber.toAscii().data());
        }

        if (!qstrNumbersToExclude.isEmpty()){
            qstrNumbersToExclude = qstrNumbersToExclude.remove(" ");
            qstrNumbersToExclude = qstrNumbersToExclude.remove("\t");
            qstrNumbersToExclude = qstrNumbersToExclude.replace("\n",",");
            gconfUtility->setGconfValueString((char *)"numbers_to_exclude",qstrNumbersToExclude.toAscii().data());
        }

        gconfUtility->setGconfValueInteger((char *)"dtmf_delay",intDTMFDelay);
        gconfUtility->setGconfValueString((char *)"dtmf_format",qstrDTMFFormat.toAscii().data());

        qDebug() << "Data saved to Gconf";
        DbusUtility dbusUtility = DbusUtility();
        QDBusConnection connection = dbusUtility.getConnection();

        if (isRoutingEnabled){
            qDebug() << "Enable call routing immediately";
            dbusUtility.sendSignal(APPLICATION_DBUS_PATH,APPLICATION_DBUS_INTERFACE,"startOutgoingCallMonitor");
        }
        else{
            qDebug() << "disable call routing immediately";
            dbusUtility.sendSignal(APPLICATION_DBUS_PATH,APPLICATION_DBUS_INTERFACE,"stopOutgoingCallMonitor");
        }
}
