/*
Copyright (C) 2011  by Cuong Le <metacuong@gmail.com>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>
*/

#include "mainwindow.h"

#include <QtGui>
#include <QtCore/QCoreApplication>
#include <QDebug>

#include <QInputDialog>

#include "listitemdetaildelegate.h"

MainWindow::MainWindow(QWidget *parent):
    QMainWindow(parent),
    m_elv1db(new elv1db(parent))
{

    detail_dlg = new QMainWindow(this);
    detail_dlg->installEventFilter(this);

    m_contact_detail = new QListWidget(detail_dlg);
    m_contact_detail->setSelectionMode(QAbstractItemView::NoSelection);
    m_contact_detail->setObjectName("m_contact_detail");
    detail_dlg->setCentralWidget(m_contact_detail);
 #ifdef Q_WS_MAEMO_5
    detail_dlg->setAttribute(Qt::WA_Maemo5StackedWindow);
 #endif


    this->setWindowTitle(QString::fromUtf8("VN Call History"));

        m_contacts_widget = new QTableWidget(0, 3, this);
        m_contacts_widget->setProperty( "FingerScrollable", true );
        m_contacts_widget->setSelectionBehavior(QAbstractItemView::SelectRows);
        m_contacts_widget->setSelectionMode(QAbstractItemView::SingleSelection);
        m_contacts_widget->setIconSize( QSize( 48, 48 ) );

        m_contacts_widget->horizontalHeader()->setDefaultSectionSize( 30 );
        m_contacts_widget->horizontalHeader()->setResizeMode( 0, QHeaderView::Fixed );
        m_contacts_widget->horizontalHeader()->setResizeMode( 1, QHeaderView::Stretch );
        m_contacts_widget->horizontalHeader()->setResizeMode( 2, QHeaderView::Fixed );

        m_contacts_widget->verticalHeader()->hide();
        m_contacts_widget->horizontalHeader()->hide();

        connect(m_contacts_widget, SIGNAL(cellClicked(int,int)), this, SLOT(cellClicked(int,int)));

        m_elv1db->m_type_query = 0;

        m_in_search = false;
        m_type_query_temp = 0;

        connect(m_elv1db, SIGNAL(group_by_finished()), this, SLOT(elv1db_finised()));
        connect(m_elv1db, SIGNAL(detail_finished()), this, SLOT(elv1db_detail_finised()));
        connect(m_elv1db, SIGNAL(start_indicator()), this, SLOT(start_indicator()));
        connect(m_elv1db, SIGNAL(detail_start_indicator()), this, SLOT(detail_start_indicator()));

        this->setup_menu();
        this->setup_menu_detail();

        this->setCentralWidget(m_contacts_widget);

        m_elv1db->m_call_type = ALL_CALL;
        m_elv1db->m_all_call = false;
        m_elv1db->start();

#ifdef Q_WS_MAEMO_5
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

}

MainWindow::~MainWindow()
{
    delete this->m_contacts_widget;
    delete this->m_contact_detail;
    delete this->m_elv1db;
    delete this->m_no_search;
}

void MainWindow::cellClicked(int row, int col){
    qDebug() << "cellClicked : " << row << ":" << col;

    m_contact_detail->clear();

    const elv1rec *ef= m_elv1db->m_records.at(row);

    m_elv1db->m_contact_name = ef->contact_name();

    detail_dlg->setWindowTitle(ef->contact_name());
    detail_dlg->show();

    m_elv1db->m_type_query = 1;
    m_elv1db->start();
}

void MainWindow::elv1db_detail_finised(){
    QDate t_cache;

    m_contact_detail->setItemDelegate(new listitemdetailDelegate(m_contact_detail));

    foreach(elv1Detailrec *ef, m_elv1db->m_detail_records){
        QString detail_format("%1 | %2 | Duration: %3");
        QTime final_duration;

        uint secs = ef->get_duration() % 60;
        uint mins = ef->get_duration() / 60;
        uint hours = ef->get_duration() / 3600;

        final_duration.setHMS(hours,mins,secs);

        if ((t_cache.day()!=ef->get_startdate().day()) ||
                (t_cache.month()!=ef->get_startdate().month()) ||
                  (t_cache.year()!=ef->get_startdate().year()) ){

            QListWidgetItem *header = new QListWidgetItem();
            header->setData(Qt::DisplayRole, ef->get_startdate().toString().toUtf8());
            m_contact_detail->addItem(header);

            t_cache=ef->get_startdate();
        }

        QListWidgetItem *item = new QListWidgetItem();
        item->setData(Qt::UserRole + 1, ef->get_phonenumber());
        item->setData(Qt::UserRole + 2, detail_format.arg(ef->get_starttime().toString())
                      .arg(ef->get_endtime().toString())
                      .arg(final_duration.toString()));
        item->setData(Qt::UserRole + 4, ef->get_icon());
        item->setData(Qt::UserRole + 5, ef->get_type_call_icon());
        m_contact_detail->addItem(item);
    }

    this->me_indicator(this->detail_dlg, false);
    m_elv1db->m_type_query = 0;
}

void MainWindow::elv1db_finised(){
    qDebug() << " Process the elv1 database is done";

    m_contacts_widget->clear();
    m_contacts_widget->setRowCount(0);

    int i=0;

    foreach(elv1rec *ef,m_elv1db->m_records){
        QTableWidgetItem *m_thumbnail = new QTableWidgetItem;
        m_thumbnail->setIcon(ef->avatar());

        m_contacts_widget->insertRow(i);
        m_contacts_widget->setItem(i,0, m_thumbnail);

        QString _m_total_call(" (%1) ");
        _m_total_call = _m_total_call.arg(ef->total_call());

        QString _m_contact_name(" %1");
        _m_contact_name = _m_contact_name.arg(ef->contact_name());

        QLabel *m_total_call = new QLabel(_m_total_call);
        m_total_call->setAlignment(Qt::AlignCenter);

        m_contacts_widget->setCellWidget(i,1, new QLabel(_m_contact_name));
        m_contacts_widget->setCellWidget(i,2, m_total_call);
        i++;
    }

    m_contacts_widget->resizeRowsToContents();
    m_contacts_widget->resizeColumnsToContents();

    this->stop_indicator();

    m_contacts_widget->setEnabled(true);
}

void MainWindow::start_indicator(){
    this->me_indicator(this, true);
}

void MainWindow::stop_indicator(){
    this->me_indicator(this, false);
}

void MainWindow::detail_start_indicator(){
    this->me_indicator(this->detail_dlg, true);
}

void MainWindow::me_indicator(QMainWindow *qmw, bool m_show){
    qmw->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, m_show);
}

void MainWindow::setup_menu(){
    QActionGroup* m_type_group = new QActionGroup(this);
    m_type_group->setExclusive(true);
    m_type_group->setObjectName("m_type_group");

    QAction* m_all_calls = new QAction(QString::fromUtf8("All call"), m_type_group );
    m_all_calls->setCheckable(true);
    m_all_calls->setObjectName( "m_all_calls" );
    connect(m_all_calls, SIGNAL(triggered()), this, SLOT(select_all_call()));

    QAction* m_gsm_calls = new QAction(QString::fromUtf8("GSM cellular"), m_type_group );
    m_gsm_calls->setCheckable(true);
    m_gsm_calls->setObjectName("m_gsm_calls");
    connect(m_gsm_calls, SIGNAL(triggered()), this, SLOT(select_gsm_calls()));

    m_gsm_calls->setChecked(true);


    QActionGroup* m_type_group_af = new QActionGroup(this);
    m_type_group_af->setExclusive(true);
    m_type_group_af->setObjectName("m_type_group_detail");

    QAction* m_all = new QAction(QString::fromUtf8("All"), m_type_group_af );
    m_all->setCheckable(true);
    m_all->setObjectName( "m_all" );
    connect(m_all, SIGNAL(triggered()), this, SLOT(select_all_typecall()));

    QAction* m_incoming = new QAction(QString::fromUtf8("Incoming"), m_type_group_af );
    m_incoming->setCheckable(true);
    m_incoming->setObjectName( "m_incoming" );
    connect(m_incoming, SIGNAL(triggered()), this, SLOT(select_incoming_calls()));

    QAction* m_outgoing = new QAction(QString::fromUtf8("Outgoing"), m_type_group_af );
    m_outgoing->setCheckable(true);
    m_outgoing->setObjectName( "m_outgoing" );
    connect(m_outgoing, SIGNAL(triggered()), this, SLOT(select_outgoing_calls()));

    QAction* m_missed = new QAction(QString::fromUtf8("Missed"), m_type_group_af );
    m_missed->setCheckable(true);
    m_missed->setObjectName( "m_missed" );
    connect(m_missed, SIGNAL(triggered()), this, SLOT(select_missed_calls()));

    m_all->setChecked(true);

    this->menuBar()->addActions(m_type_group->actions());
    this->menuBar()->addActions(m_type_group_af->actions());

/*    QAction *m_clear_log = this->menuBar()->addAction("Clear Log");
    QAction *m_options = this->menuBar()->addAction(QString::fromUtf8("Options"));

    connect(m_clear_log,SIGNAL(triggered()), this, SLOT(clearLog()));
    connect(m_options, SIGNAL(triggered()), this, SLOT(dlgOptions()));*/

    QAction *m_search = this->menuBar()->addAction("Search");
    connect(m_search,SIGNAL(triggered()), this, SLOT(dlgSearch()));

    m_no_search = this->menuBar()->addAction("Search is Off");
    connect(m_no_search,SIGNAL(triggered()), this, SLOT(noSearch()));

    QAction *m_about = this->menuBar()->addAction("About");
    QAction *m_about_qt = this->menuBar()->addAction("About Qt");

    connect(m_about,SIGNAL(triggered()), this, SLOT(dlgAbout()));
    connect(m_about_qt,SIGNAL(triggered()), qApp, SLOT(aboutQt()));

}

void MainWindow::noSearch(){
    if (m_in_search){
        m_contacts_widget->setEnabled(false);
        m_no_search->setText("Search is Off");
        m_elv1db->m_type_query = m_type_query_temp;
        m_in_search = false;
        m_elv1db->start();
    }
}

void MainWindow::dlgSearch(){
    bool ok;
    QString text = QInputDialog::getText(this, tr("Search"),
                                         tr("Give me a contact name to do a search :"), QLineEdit::Normal,
                                         *m_elv1db->m_search_val, &ok);
    if (ok && !text.isEmpty() && !text.trimmed().isEmpty()){
        m_no_search->setText("Search is On");
        m_in_search = true;
        this->doSearch(text);
    }
}

void MainWindow::doSearch(QString val){
    m_contacts_widget->setEnabled(false);
    *m_elv1db->m_search_val = val;
    m_elv1db->m_type_query = 2;
    m_elv1db->start();
}

void MainWindow::select_all_call(){ //ring/tel/ring, spirit/skype, gabble/jabber
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_all_call = true;
    m_type_query_temp = 0;
    m_elv1db->start();
}

void MainWindow::select_gsm_calls(){ //ring/tel/ring
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_all_call = false;
    m_type_query_temp = 0;
    m_elv1db->start();
}

void MainWindow::select_all_typecall(){ //incoming, outgoing and missed calls
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_call_type = ALL_CALL;
    m_elv1db->start();
}

void MainWindow::select_incoming_calls(){
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_call_type = INCOMING;
    m_elv1db->start();
}

void MainWindow::select_outgoing_calls(){
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_call_type = OUTGOING;
    m_elv1db->start();
}

void MainWindow::select_missed_calls(){
    m_contacts_widget->setEnabled(false);
    m_elv1db->m_call_type = MISSED;
    m_elv1db->start();
}

void MainWindow::clearLog(){
}

void MainWindow::setup_menu_detail(){
}

void MainWindow::dlgAbout(){
    QMessageBox m_about;
    m_about.setWindowTitle("About");
    m_about.setText(
                "<font size=4>VN Call History</font><br>"
                "<font size=3>Version : 0.0.2 (09/29/2011)</font><br>"
                "Author : Cuong Le <a href=\"mailto:metacuong@gmail.com\">metacuong@gmail.com</a><br>"
                "<font size=2>"
                "Homepage : <a href=\"http://vncallhistory.garage.maemo.org/\">http://vncallhistory.garage.maemo.org/</a><br>"
                "Bugs report : <a href=\"https://garage.maemo.org/tracker/?group_id=2259\">https://garage.maemo.org/tracker/?group_id=2259</a><br>"
                "License : GNU/GPL version 2"
                "</font>"
                );
    m_about.exec();
}

void MainWindow::dlgOptions(){
    dlg_options = new QMainWindow(this);
    dlg_options->setWindowTitle(QString::fromUtf8("Options"));
    dlg_options->installEventFilter(this);
#ifdef Q_WS_MAEMO_5
    dlg_options->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    dlg_options->show();
}

void MainWindow::setOrientation(ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}

void MainWindow::showExpanded()
{
#ifdef Q_OS_SYMBIAN
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}
